#-------------------------------------------------------------------------------
#
#  Define a 'scrolled' control for the Enable toolkit.
#
#  Written by: David C. Morrill
#
#  Date: 11/28/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: Scrolled
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from enthought.traits.api    import Instance, Trait
from enthought.traits.ui.api import Group, View, Include
from base                import xy_in_bounds 
from base                import transparent_color, add_rectangles
from events              import MouseEvent                             
from enable_traits       import white_color_trait, black_color_trait, \
                                border_size_trait
from base_container      import BaseContainer, default_container
from component           import Component
from scrollbar           import ScrollBar
from scroll_handler      import ScrollHandler

#-------------------------------------------------------------------------------
#  'Scrolled' class:
#-------------------------------------------------------------------------------

class Scrolled ( Component, BaseContainer, ScrollHandler ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    component      = Trait( None, Component )
    bg_color       = white_color_trait
    border_color   = black_color_trait
    border_size    = border_size_trait
    scroll_handler = Instance( ScrollHandler )

    #---------------------------------------------------------------------------
    #  Trait initializers:
    #---------------------------------------------------------------------------

    def _scroll_handler_default ( self ):
        return self
    
    #---------------------------------------------------------------------------
    #  Trait view definitions:
    #---------------------------------------------------------------------------
    
    traits_view = View( Group( '<component>',          id = 'component' ),
                        Group( '<links>', 'component', id = 'links' ),
                        Group( 'bg_color{Background color}', '_',
                               'border_color', '_', 
                               'border_size',
                               id    = 'scrolled',
                               style = 'custom' ) )
    
    colorchip_map = {
        'bg_color':  'bg_color',
        'alt_color': 'border_color'
    }

    #---------------------------------------------------------------------------
    #  Standard scroll bar height and width (filled in on demand):
    #---------------------------------------------------------------------------
                            
    _sb_height = None
    _sb_width  = None

    #---------------------------------------------------------------------------
    #  Initialize the object: 
    #---------------------------------------------------------------------------
    
    def __init__ ( self, component = None, **traits ):
        Component.__init__( self, **traits )
        if component is None:
            component = Component()
        self.component = component
        
    #---------------------------------------------------------------------------
    #  Add a component to the container:
    #---------------------------------------------------------------------------
    
    def add ( self, *components ):
        if len( components ) > 0:
            self.component = components[-1]
                
    #---------------------------------------------------------------------------
    #  Remove a component from the container:
    #---------------------------------------------------------------------------
    
    def remove ( self, *components ):
        for component in components:
            if component is self.component:
                component.container = default_container
                self.component      = Component()
                break
                
    #---------------------------------------------------------------------------
    #  Handle visual only trait changes:
    #---------------------------------------------------------------------------
    
    def _bg_color_changed ( self ):
        self.redraw()
        
    def _border_color_changed ( self ):
        self.redraw()
    
    #---------------------------------------------------------------------------
    #  Handle layout related trait changes:
    #---------------------------------------------------------------------------
    
    def _border_size_changed ( self ):
        self._layout()
    
    #---------------------------------------------------------------------------
    #  Handle the component being changed: 
    #---------------------------------------------------------------------------
    
    def _component_changed ( self, old, new ):
        if old is not None:
            old.on_trait_change( self.update_bounds, 'min_width',
                                 remove = True)
            old.on_trait_change( self.update_bounds, 'min_height',
                                 remove = True )
        if new is None:
            self.component = Component()
            return
        new.container.remove( new )
        new.container = self
        new.on_trait_change( self.update_bounds, 'min_width' )
        new.on_trait_change( self.update_bounds, 'min_height' )
        self._hsb = self._release_sb( self._hsb )
        self._vsb = self._release_sb( self._vsb )
        self._layout()
        
    #---------------------------------------------------------------------------
    #  Handle the bounds of the Scrolled component being changed: 
    #---------------------------------------------------------------------------
    
    def _bounds_changed ( self, old, new ):
        Component._bounds_changed( self, old, new )
        self.update_bounds()
        
    def update_bounds ( self ):
        # Compute the proposed new component size:
        component    = self.component
        min_dx       = component.min_width
        min_dy       = component.min_height
        bs           = self.border_size
        bs2          = 2.0 * bs
        x, y, dx, dy = self.bounds
        dx          -= bs2
        dy          -= bs2
        adjust_dx    = adjust_dy = 0
        
        # Check to see if scroll bars will be needed and adjust the new 
        # component size accordingly:
        if dx < min_dx:
            adjust_dy = self.sb_height()
            if (dy - adjust_dy) < min_dy:
                adjust_dx = self.sb_width()
        elif dy < min_dy:
            adjust_dx = self.sb_width()
            if (dx - adjust_dx) < min_dx:
                adjust_dy = self.sb_height()
                
        # Assign the new component bounds:
        component._invalidate_visible_bounds()
        bounds           = component.bounds
        component.bounds = self._clip_bounds = (
                              x + bs, y + bs + adjust_dy, 
                              dx - adjust_dx, dy - adjust_dy )
        self._layout()

    #---------------------------------------------------------------------------
    #  Handle the scroll handler being changed: 
    #---------------------------------------------------------------------------

    def _scroll_handler_changed ( self, old, new ):

        if self._vsb is not None:
            # Reset the old scroll handler.
            old.handle_vertical_scroll( 0.0 )

            # Set the new scroll handler to the current position.
            new.handle_vertical_scroll( self._vsb.position )

        if self._hsb is not None:
            # Reset the old scroll handler.
            old.handle_horizontal_scroll( 0.0 )

            # Set the new scroll handler to the current position.
            new.handle_horizontal_scroll( 0.0 )
            
    #---------------------------------------------------------------------------
    #  Returns the standard scroll bar height and width:
    #---------------------------------------------------------------------------
        
    def sb_height ( self ):
        if Scrolled._sb_height is None:
            Scrolled._sb_height = ScrollBar( style = 'horizontal' ).min_height
        return Scrolled._sb_height
        
    def sb_width ( self ):
        if Scrolled._sb_width is None:
            Scrolled._sb_width = ScrollBar( style = 'vertical' ).min_width
        return Scrolled._sb_width
        
    #---------------------------------------------------------------------------
    #  Handle the layout of the component being changed: 
    #---------------------------------------------------------------------------
    
    def _layout ( self ):
        component = self.component
        if component is None:
            return
        x,   y,  dx,  dy = self.bounds
        cx, cy, cdx, cdy = component.bounds
        bs  = self.border_size
        bs2 = bs * 2.0
        vdx = dx - bs2
        vdy = dy - bs2
        hsb = vsb = None
        while True:
            changed = False
            if hsb is None:
                if cdx > vdx:
                    hsb = self._hsb
                    if hsb is None:
                        self._hsb = hsb = ScrollBar( style = 'horizontal' )
                        hsb.on_trait_change( self._hscroll, 'position' )
                        hsb.container = self
                        hpos          = 0.0
                    else:
                        hpos = hsb.position / (hsb.high - hsb.page_size)
                    vdy -= hsb.min_height
                    changed = True
                else:
                    self._hsb = self._release_sb( self._hsb )
            if vsb is None:
                if cdy > vdy:
                    vsb = self._vsb
                    if vsb is None:
                        self._vsb = vsb = ScrollBar( style = 'vertical' )
                        vsb.on_trait_change( self._vscroll, 'position' )
                        vsb.container = self
                        vpos          = 0.0
                    else:
                        vpos = vsb.position / (vsb.high - vsb.page_size)
                    vdx -= vsb.min_width
                    changed = True
                else:
                    self._vsb = self._release_sb( self._vsb )
            if not changed:
                break
                
        if hsb is not None:
            self.min_width  = hsb.min_width
            self.min_height = hsb.min_height
            hdy             = hsb.min_height
            hsb.bounds      = ( x, y, vdx + bs2, hdy )
            hsb.range       = ( 0.0, cdx, vdx, 16.0 )
            hsb.position    = hpos * (cdx - vdx)
            self._hscroll()
            y              += hdy
        else:
            self.min_width  = 0.0
            self.min_height = 0.0
            
        if vsb is not None:
            self.min_width  += vsb.min_width
            self.min_height += vsb.min_height
            vsb.bounds       = ( x + vdx + bs2, y, vsb.min_width, vdy + bs2 )
            vsb.range        = ( 0.0, cdy, vdy, 16.0 )
            vsb.position     = vpos * (cdy - vdy)
            self._vscroll()
        
    #---------------------------------------------------------------------------
    #  Release a scrollbar:
    #---------------------------------------------------------------------------
    
    def _release_sb ( self, sb ):
        if sb is not None:
            sb.on_trait_change( self._scroll, 'position', remove = True )
        return None
       
    #---------------------------------------------------------------------------
    #  Handle a horizontal scroll bar position change:
    #---------------------------------------------------------------------------
    
    def _hscroll ( self ):
        self.scroll_handler.handle_horizontal_scroll( self._hsb.position )

    #---------------------------------------------------------------------------
    #  ScrollHandler implementation:
    #---------------------------------------------------------------------------
    
    def handle_horizontal_scroll( self, position ):
        self.component.x = self._clip_bounds[0] - round( position )
        
    #---------------------------------------------------------------------------
    #  Handle a vertical scroll bar position change:
    #---------------------------------------------------------------------------
    
    def _vscroll ( self ):
        self.scroll_handler.handle_vertical_scroll( self._vsb.position )

    #---------------------------------------------------------------------------
    #  ScrollHandler implementation:
    #---------------------------------------------------------------------------
    
    def handle_vertical_scroll(self, position):
        self.component.y = self._clip_bounds[1] - round( position ) 
                       
    #---------------------------------------------------------------------------
    #  Generate any additional components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------
       
    def _components_at ( self, x, y ):
        if self._vsb is not None:
            result = self._vsb.components_at( x, y )
            if len( result ) > 0:
                return result
                
        if self._hsb is not None:
            result = self._hsb.components_at( x, y )
            if len( result ) > 0:
                return result
                
        return [ self ] + self.component.components_at( x, y )
                
    #---------------------------------------------------------------------------
    #  Draw the component in a specified graphics context:
    #---------------------------------------------------------------------------
    
    def _draw ( self, gc ):
        gc.save_state()
        
        # Set up all the control variables for quick access:
        bs  = self.border_size
        bsd = bs + bs
        bsh = bs / 2.0
        x, y, dx, dy = self.bounds
        
        # Fill the background region (if required);
        bg_color = self.bg_color_
        if bg_color is not transparent_color:
            gc.set_fill_color( bg_color )
            gc.begin_path()
            gc.rect( x + bs, y + bs, dx - bsd, dy - bsd ) 
            gc.fill_path()
            
        # Draw the border (if required):
        if bs > 0:
            border_color = self.border_color_
            if border_color is not transparent_color:
                gc.set_stroke_color( border_color )
                gc.set_line_width( bs )
                gc.begin_path()
                gc.rect( x + bsh, y + bsh, dx - bs, dy - bs )
                gc.stroke_path()

        # Draw the scroll bars (if necessary):
        vsb = self._vsb
        if vsb is not None:
            vsb.draw( gc )
        
        hsb = self._hsb
        if hsb is not None:
            hsb.draw( gc )
            
        # Draw the component:
        gc.clip_to_rect(*self._clip_bounds)
        self.component.draw( gc )
        
        gc.restore_state()
       
    #---------------------------------------------------------------------------
    #  Vertically scrolls the canvas when the mouse scroll wheel is spun:
    #---------------------------------------------------------------------------
        
    def _pre_mouse_wheel_changed ( self, event ):
        bounds = self._clip_bounds
        x, y   = event.x, event.y
        if xy_in_bounds( x, y, bounds ):
            sb = self._hsb
            if ((sb is not None) and
                xy_in_bounds( x, y, add_rectangles( bounds, 
                              ( 0, 0, -0.1 * bounds[2], -0.9 * bounds[3] ) ) )):
                event.handled = True
                sb.position  += (event.mouse_wheel * sb.page_size) / 20
            else:
                sb = self._vsb
                if sb is not None:
                    event.handled = True
                    sb.position  += (event.mouse_wheel * sb.page_size) / 20
                    
