#-------------------------------------------------------------------------------
#  
#  Defines the OMOuterComponent class of the Enable 'om' (Object Model) package. 
#
#  The OMOuterComponent class defines the base class for the 'outer component'
#  portion of an OMComponent object.
#  
#  Written by: David C. Morrill
#  
#  Date: 02/09/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

import re

from om_base               import om_handler
from om_traits             import StyleDelegate, OffsetXY, SizeXY
                           
from enthought.enable      import Component
from enthought.enable.base import gc_image_for

from enthought.traits.api      import HasPrivateTraits, Range, Instance, Property, \
                                  Str, true
from enthought.traits.ui.api   import View

#-------------------------------------------------------------------------------
#  Trait definitions:  
#-------------------------------------------------------------------------------

# Margin around the sides of the component:
ComponentMargin  = Range( 0, 31 )

# Padding around the sides of the component:
ComponentPadding = Range( 0, 31, 5 )

# Aspect ratio (width/height) for the component (0.0 = Ignore):
AspectRatio      = Range( 0.0, 5.0 )

#-------------------------------------------------------------------------------
#  Constants:  
#-------------------------------------------------------------------------------

# Regular expressions used to extract 'margin' info from image file names:
left_margin_pat    = re.compile( r'_M?L(\d+)_', re.IGNORECASE )
right_margin_pat   = re.compile( r'_M?R(\d+)_', re.IGNORECASE )
top_margin_pat     = re.compile( r'_M?T(\d+)_', re.IGNORECASE )
bottom_margin_pat  = re.compile( r'_M?B(\d+)_', re.IGNORECASE )

left_padding_pat   = re.compile( r'_PL(\d+)_', re.IGNORECASE )
right_padding_pat  = re.compile( r'_PR(\d+)_', re.IGNORECASE )
top_padding_pat    = re.compile( r'_PT(\d+)_', re.IGNORECASE )
bottom_padding_pat = re.compile( r'_PB(\d+)_', re.IGNORECASE )

#-------------------------------------------------------------------------------
#  'OMOuterComponentStyle' class:  
#-------------------------------------------------------------------------------

class OMOuterComponentStyle ( HasPrivateTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
    
    # Aspect ratio (width/height) for the component (0.0 = Ignore):
    aspect_ratio           = AspectRatio
    
    # Amount of padding added to left side of the component:
    left_padding           = ComponentMargin
    
    # Amount of padding added to right side of the component:
    right_padding          = ComponentMargin
    
    # Amount of padding added to top side of the component:
    top_padding            = ComponentMargin
    
    # Amount of padding added to bottom side of the component:
    bottom_padding         = ComponentMargin
    
    # Amount of margin added to left side of the component:
    left_margin            = ComponentMargin
    
    # Amount of margin added to right side of the component:
    right_margin           = ComponentMargin
    
    # Amount of margin added to top side of the component:
    top_margin             = ComponentMargin
    
    # Amount of margin added to bottom side of the component:
    bottom_margin          = ComponentMargin
    
    # The root name of the image set used to represent the component:
    image                  = Str( '=component_%d.gif' )
    
    # Is the component horizontally symmetric?
    horizontally_symmetric = true
    
    # Is the component vertically symmetric?
    vertically_symmetric   = true
    
    # Show tooltip information?
    show_tooltip           = true
    
    # Show status information?
    show_status            = true
    
    #---------------------------------------------------------------------------
    #  Initializes the object:  
    #---------------------------------------------------------------------------
        
    def __init__ ( self, **traits ):
        super( OMOuterComponentStyle, self ).__init__( **traits )
        if self._images is None:
            self._image_changed()
    
#-- Event Handlers -------------------------------------------------------------    
    
    #---------------------------------------------------------------------------
    #  Handle the image file root name being changed: 
    #---------------------------------------------------------------------------
    
    def _image_changed ( self ):
        image  = self.image
        images = []
        first  = 0
        if image.find( '%d' ) >= 0:
            image_name = image
            image      = image.replace( '%d', '' )
        else:
            image_name = image + '_%d'
        for i in range( 10 ):
            try:
                images.append( self.image_for( image_name % i ) )
                if len( images ) == 9:
                    break
            except:
                if i == (first + 4):
                    images.append( None )
                elif i == 0:
                    first = 1
                else:
                    return
        self._images      = images
        self._image_sizes = [ images[0].width(),  images[1].width(),
                              images[2].width(),  images[3].width(), 
                              1,                  images[5].width(),
                              images[6].width(),  images[7].width(),
                              images[8].width(),  images[0].height(),
                              images[3].height(), images[6].height() ]
        if images[4] is not None:
            self._image_sizes[4] = images[4].width()
            
        # Attempt to extract margin info from the image file name:
        self._get_margin( image, left_margin_pat,   'left_margin',    0 ) 
        self._get_margin( image, right_margin_pat,  'right_margin',   2 ) 
        self._get_margin( image, top_margin_pat,    'top_margin',    -3 ) 
        self._get_margin( image, bottom_margin_pat, 'bottom_margin', -1 ) 
        
        # Attempt to extract padding info from the image file name:
        self._get_margin( image, left_padding_pat,   'left_padding' ) 
        self._get_margin( image, right_padding_pat,  'right_padding' ) 
        self._get_margin( image, top_padding_pat,    'top_padding' )
        self._get_margin( image, bottom_padding_pat, 'bottom_padding' )
            
    #---------------------------------------------------------------------------
    #  Return an image specified by name: 
    #---------------------------------------------------------------------------
                
    def image_for ( self, image ):
        path   = ''
        prefix = image[:1]
        if prefix == '=':
            path  = self
            image = image[1:]
        elif prefix == '.':
            path  = None
            image = image[1:]
        return gc_image_for( image, path )
        
    #---------------------------------------------------------------------------
    #  Attempts to extract margin/padding information from the image file name:  
    #---------------------------------------------------------------------------
                
    def _get_margin ( self, image_name, re_pat, trait_name, 
                      size_index = None ):
        match = re_pat.search( image_name )
        if (match is None) and (size_index is None):
            return
            
        value = 0
        if match is not None:
            try:
                value = int( match.group( 1 ) )
            except:
                pass
                
        setattr( self, trait_name, value )
                
        # If this was a 'margin' value, automatically compute the
        # corresponding 'padding' value by subtracting the margin from
        # the corresponding 'size_index' image size:
        if size_index is not None:
            setattr( self, trait_name[:-6] + 'padding', 
                     self._image_sizes[ size_index ] - value )
                     
#-- Pickling Protocol ----------------------------------------------------------

    def __getstate__ ( self ):
        dict = self.__dict__.copy()
        try:
            del dict[ '_images' ]
            del dict[ '_image_sizes' ]
        except:
            pass
        return dict
        
    def __setstate__ ( self, state ):
        self.__dict__.update( state )
        self._image_changed()
        
# Create a default outer component style:    
default_outer_component_style = OMOuterComponentStyle()

#-------------------------------------------------------------------------------
#  'OMOuterComponent' class:
#-------------------------------------------------------------------------------

class OMOuterComponent ( Component ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
    
    # The location and size of the component (override of Component definition):
    bounds = Property
    
    # Origin of the component relative to its containing component:
    origin = OffsetXY
    
    # Size of the component:
    size   = SizeXY
    
    # The style to use for the component:
    style  = Instance( OMOuterComponentStyle, default_outer_component_style )
    
    # Aspect ratio (width/height) for the component (0.0 = Ignore):
    aspect_ratio           = StyleDelegate
    
    # Amount of padding added to left side of the component:
    left_padding           = StyleDelegate
    
    # Amount of padding added to right side of the component:
    right_padding          = StyleDelegate
    
    # Amount of padding added to top side of the component:
    top_padding            = StyleDelegate
    
    # Amount of padding added to bottom side of the component:
    bottom_padding         = StyleDelegate
    
    # Amount of margin added to left side of the component:
    left_margin            = StyleDelegate
    
    # Amount of margin added to right side of the component:
    right_margin           = StyleDelegate
    
    # Amount of margin added to top side of the component:
    top_margin             = StyleDelegate
    
    # Amount of margin added to bottom side of the component:
    bottom_margin          = StyleDelegate
    
    # The root name of the image set used to represent the component:
    image                  = StyleDelegate
    
    # The cached image information associated with the 'image' trait:
    _images                = StyleDelegate
    _image_sizes           = StyleDelegate
    
    # Is the component horizontally symmetric?
    horizontally_symmetric = StyleDelegate
    
    # Is the component vertically symmetric?
    vertically_symmetric   = StyleDelegate
    
    # Show tooltip information?
    show_tooltip           = StyleDelegate
    
    # Show status information?
    show_status            = StyleDelegate
    
    # Tooltip text:
    tooltip                = Str
    
    # Status text:
    status                 = Str
    
    #---------------------------------------------------------------------------
    #  Traits view definitions:    
    #---------------------------------------------------------------------------
    
    traits_view = View( [ 
        [ 'image', '|[Image]<>' ],
        [ 'aspect_ratio', '|[Aspect Ratio]<>' ],           
        [ 'horizontally_symmetric{Is component horizontally symmetric?}', 
          'vertically_symmetric{Is component vertically symmetric?}',
          '-[Options]>' ],
        [ 'left_padding{Left}', 'right_padding{Right}', 
          'top_padding{Top}',   'bottom_padding{Bottom}',
          '|[Padding]' ],
        [ 'left_margin{Left}',  'right_margin{Right}', 
          'top_margin{Top}',    'bottom_margin{Bottom}',
          '|[Margin]' ] ],
        handler = om_handler
    )
    
#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'bounds' property:  
    #---------------------------------------------------------------------------

    def _get_bounds ( self ):
        x, y, dx, dy = self.container.bounds
        ox, oy       = self.origin
        return ( x + ox, y + oy ) + self.size
        
    def _set_bounds ( self, bounds ):
        x, y, dx, dy     = bounds
        cx, cy, cdx, cdy = self.container.bounds
        self.origin      = ( x - cx, y - cy )
        self.size        = ( int( dx ), int( dy ) )

#-- Public Methods -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Force the component to be updated:    
    #---------------------------------------------------------------------------

    def update ( self ):
        self.container.update( True )
        
#-- Overridden Methods ---------------------------------------------------------
                
    #---------------------------------------------------------------------------
    #  Draws the component in a specified graphics context:
    #---------------------------------------------------------------------------
    
    def _draw ( self, gc ):
        gc.save_state()
        
        xl, yb, dx, dy = self.bounds
        ( ixlt, ixmt, ixrt, ixlm, ixmm, ixrm, ixlb, ixmb, ixrb, 
          iyt, iym, iyb ) = self._image_sizes
        yt     = yb + dy  - iyt
        ym     = yb + iyb
        dym    = dy - iyt - iyb
        images = self._images
        
        # Draw the four fixed-size image corners:
        gc.draw_image( images[0], ( xl, yt, ixlt, iyt ) )
        gc.draw_image( images[2], ( xl + dx - ixrt, yt, ixrt, iyt ) )
        gc.draw_image( images[6], ( xl, yb, ixlb, iyb ) )
        gc.draw_image( images[8], ( xl + dx - ixrb, yb, ixrb, iyb ) )
        
        # Draw all of the 'stretchable' images along the sides and middle:
        gc.stretch_draw( images[1], xl + ixlt, yt, dx - ixlt - ixrt, iyt )
        gc.stretch_draw( images[3], xl, ym, ixlm, dym )
        gc.stretch_draw( images[5], xl + dx - ixrm, ym, ixrm, dym )
        gc.stretch_draw( images[7], xl + ixlb, yb, dx - ixlb - ixrb, iyb )
                            
        # Only draw the center region if we have an image for it:
        if images[4] is not None:                    
            gc.stretch_draw( images[4], xl + ixlm, ym, dx - ixlm - ixrm, dym )

        gc.restore_state()
            
    #---------------------------------------------------------------------------
    #  Returns the component if it contains a specified (x,y) point:
    #---------------------------------------------------------------------------
       
    def _components_at ( self, x, y ):
        if self._image_at( x, y ) >= 0:
            return [ self ]
        return []

#-- Private Methods ------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Returns the sub-image at a specified point:
    #---------------------------------------------------------------------------
       
    def _image_at ( self, x, y = None ):
        # Allow ( x = mouse_event, y = None ) case:
        if y is None:
            y = x.y
            x = x.x
        
        if not self.xy_in_bounds( x, y ):
            return -1

        xl, yb, dx, dy = self.bounds
        tx = x - xl
        ty = y - yb
        ( ixlt, ixmt, ixrt, ixlm, ixmm, ixrm, ixlb, ixmb, ixrb, 
          iyt, iym, iyb ) = self._image_sizes
        yt = dy - iyt
        
        # Reduce (x,y) to valid image coordinates:
        if ty < iyb:
            idy = iyb
            if tx < ixlb:
                index = 6
                tx    = int( tx )
            elif tx < (dx - ixrb):
                index = 7
                tx    = int( tx - ixlb ) % ixmb
            else:
                index = 8
                tx    = int( tx - (dx - ixrb) )
        elif ty < yt:
            idy = iym
            ty  = int( ty - iyb ) % iym
            if tx < ixlm:
                index = 3
                tx    = int( tx )
            elif tx < (dx - ixrm):
                index = 4
                tx    = int( tx - ixlm ) % ixmm
            else:
                index = 5
                tx    = int( tx - (dx - ixrm) )
        else:
            idy = iyt
            ty -= yt
            if tx < ixlt:
                index = 0
                tx    = int( tx )
            elif tx < (dx - ixrt):
                index = 1
                tx    = int( tx - ixlt ) % ixmt
            else:
                index = 2
                tx    = int( tx - (dx - ixrt) )
         
        # If the image alpha value of the point under (x,y) has a high alpha,
        # return the specified image index, otherwise indicate it is not over
        # an image:
        image = self._images[ index ]
        if ((image is not None) and ((image.bmp_array.shape[2] < 4) or
            (image.bmp_array[ idy - int( ty ) - 1, int( tx ), 3 ] >= 128))):
            return index
        return -1
        
    #---------------------------------------------------------------------------
    #  Handle mouse events: 
    #---------------------------------------------------------------------------
    
#-- 'normal' State -------------------------------------------------------------

    def normal_mouse_move ( self, event ):
        event.handled = True
        self.pointer  = 'arrow'
        
    def normal_left_down ( self, event ):
        event.handled           = True
        self.window.mouse_owner = self
        self.event_state        = 'drag_pending'
        self.pointer            = 'sizing'
        self._x_y               = ( event.x, event.y )
        
    def normal_left_dclick ( self, event ):
        event.handled = True
        component     = self.container
        component.controller.edit_component( component )

    def normal_right_up ( self, event ):
        event.handled = True
        component     = self.container
        component.controller.popup_menu( component, event )
        
#-- 'drag_pending' State -------------------------------------------------------

    def drag_pending_left_up ( self, event ):
        event.handled           = True
        self.window.mouse_owner = None
        self.event_state        = 'normal'
        self.pointer            = 'arrow'
        component               = self.container
        controller              = component.controller
        if controller.can_select_component( component ):
            canvas = component.container
            canvas.controller.components_selected( canvas, [ component ], 
                                                   event.control_down )
        
    def drag_pending_mouse_move ( self, event ):
        event.handled = True
        x, y          = self._x_y
        if (( abs( x - event.x ) + abs( y - event.y ))) > 2:        
            self.event_state = 'normal'
            component        = self.container
            if component.controller.can_drag_component( component ): 
                self.window.mouse_owner = None
                component.drag( event )
        
