#-------------------------------------------------------------------------------
#
#  Defines the OMContact class of the Enable 'om' (Object Model) package.
#
#  The OMContact class defines the base class for visual connection points.
#  A connection point is an element contained within an OMComponent object
#  that represents a point of contact within an object of the underlying object
#  model (e.g. the object itself, or an attribute of the object).
#
#  Written by: David C. Morrill
#
#  Date: 01/27/2005
#
#  (c) Copyright 2005 by Enthought, Inc.
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from om_base          import om_handler
from om_traits        import ContactPosition, ContactLabelPosition, OffsetXY, \
                             SizeXY, ContactCategory, ContactGroup, \
                             ContactState, StyleDelegate, OUTSIDE, INSIDE, \
                             LEFT, RIGHT, TOP, BOTTOM, ERGBAColor
from om_link          import OMLink

from enthought.enable      import Component
from enthought.enable.base import xy_in_bounds
from enthought.kiva.traits.kiva_font_trait import KivaFont
from enthought.traits.api      import HasStrictTraits, Instance, Default, List, \
                                  Any, Dict, Str, Int, Property, true
from enthought.traits.ui.api   import View

#-------------------------------------------------------------------------------
#  Data:
#-------------------------------------------------------------------------------

# The default, singleton, OMContactController object:
_default_contact_controller = None

#-------------------------------------------------------------------------------
#  'OMContactStyle' class:
#-------------------------------------------------------------------------------

class OMContactStyle ( HasStrictTraits ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # The font used to display label:
    label_font     = KivaFont( 'modern 7' )

    # The color used to display the label text:
    label_color    = ERGBAColor( 'black' )

    # The position of the label relative to the contact (and component edge):
    label_position = ContactLabelPosition

    # The offset of the label from the contact:
    label_offset   = OffsetXY( ( 3, 3 ) )

    # Should the label be rotated if the contact is on the top or bottom edge?
    label_rotate   = true

    # The dictionary mapping the contact state to an image name:
    image_map      = Dict( Str, Str )

    # Show tooltip information?
    show_tooltip   = true

    # Show status information?
    show_status    = true

    # Tooltip text:
    tooltip        = Str

    # Status text:
    status         = Str

    # Context menu:
    menu           = Property

#-- Property implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'menu' property (normally overridden):
    #---------------------------------------------------------------------------

    def _get_menu ( self ):
        return self.get_menu()

    def get_menu ( self ):
        return None

# Create a default contact style:
default_contact_style = OMContactStyle()

#-------------------------------------------------------------------------------
#  'OMContact' class:
#-------------------------------------------------------------------------------

class OMContact ( Component ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # The controller for the contact:
    controller       = Instance( 'enthought.enable.om.OMContactController' )

    # The location and size of the component (override of Component definition):
    bounds           = Property

    # The style to use for drawing the contact:
    style            = Instance( OMContactStyle, default_contact_style )

    # The font used to display the label:
    label_font       = StyleDelegate

    # The color used to display the label text:
    label_color      = StyleDelegate

    # The position of the label relative to the contact (and component edge):
    label_position   = StyleDelegate

    # The offset of the label from the contact:
    label_offset     = StyleDelegate

    # Should the label be rotated if the contact is on the top or bottom edge?
    label_rotate     = StyleDelegate

    # The dictionary mapping the contact state to an image name:
    image_map        = StyleDelegate

    # Show tooltip information?
    show_tooltip     = StyleDelegate

    # Show status information?
    show_status      = StyleDelegate

    # Context menu:
    menu             = StyleDelegate

    # Location of the contact relative to the component it is contained in:
    position         = ContactPosition

    # The name of the contact:
    name             = Str

    # The label for the contact:
    label            = Property

    # The tooltip for the contact:
    tooltip          = Str

    # The name of the image used to represent the contact:
    image            = Str( '=contact' )

    # The set of links the contact is connected to:
    links            = List # ( 'enthought.enable.om.OMLink' )

    # The category that the contact is associated with (e.g. 'input', 'output'):
    category         = ContactCategory

    # The group that the contact is associated with (e.g. 'porosity'):
    group            = ContactGroup

    # Maximum number of links that can be connected:
    max_links        = Int( 1 )

    # Type of data that is accepted (input) or produced (output):
    type             = Any

    # Contact data (to help controller relate contact back to object model):
    data             = Any

    # Origin of the contact relative to its containing component:
    origin           = OffsetXY

    # Size of the contact (including the label):
    size             = Property

    # Offset to the center of the contact when the label is included:
    center           = Property

    # Size of the contact image (excluding the label):
    image_size       = SizeXY

    # Offset to the center of the contact image (excluding the label):
    image_center     = OffsetXY

    #---------------------------------------------------------------------------
    #  Traits view definition:
    #---------------------------------------------------------------------------

    traits_view = View(
        [ [ 'label', '_', 'label_color{Color}@', '_',
            'label_font{Font}@', 'label_position{Position}',
            'label_offset{Offset}@', 'label_rotate{Rotate}',
            '|[Label]' ],
         [ 'position', 'image',
           '|[Contact]' ] ],
        help = False,
        handler = om_handler
    )

    #---------------------------------------------------------------------------
    #  Initializes the object:
    #---------------------------------------------------------------------------

    def __init__ ( self, **traits ):
        """ Initializes the object.
        """
        self.event_state = 'unconnected'
        super( OMContact, self ).__init__( **traits )
        if self._image is None:
            self._image_changed()

#-- Default Trait Value Handlers -----------------------------------------------

    #---------------------------------------------------------------------------
    #  Returns the default value for the 'controller' trait:
    #---------------------------------------------------------------------------

    def _controller_default ( self ):
        global _default_contact_controller

        if _default_contact_controller is None:
            from om_contact_controller import OMContactController
            _default_contact_controller = OMContactController()
        return _default_contact_controller

#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'bounds' property:
    #---------------------------------------------------------------------------

    def _get_bounds ( self ):
        x, y, dx, dy = self.container.bounds
        dx, dy       = self.size
        ox, oy       = self.origin
        return ( x + ox, y + oy, dx, dy )

    #---------------------------------------------------------------------------
    #  Implementation of the 'label' property:
    #---------------------------------------------------------------------------

    def _get_label ( self ):
        if self._label is not None:
            return self._label
        return self.name

    def _set_label ( self, value ):
        self._label = value
        self._size = self._center = None

    #---------------------------------------------------------------------------
    #  Implementation of the 'size' property:
    #---------------------------------------------------------------------------

    def _get_size ( self ):
        if self._size is None:
            self._compute_contact_size()
        return self._size

    #---------------------------------------------------------------------------
    #  Implementation of the 'center' property:
    #---------------------------------------------------------------------------

    def _get_center ( self ):
        if self._center is None:
            self._compute_contact_size()
        return self._center

#-- Draw Related Methods -------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Draws the contact:
    #---------------------------------------------------------------------------

    def _draw ( self, gc ):
        gc.save_state()

        # Draw the contact image:
        x, y, dx, dy = self.bounds
        ix, iy       = self._image_origin
        gc.draw_image( self._image, ( x + ix, y + iy ) + self.image_size )

        # Draw the label (if any):
        label = self.label
        if label != '':
            tx, ty = self._text_origin
            gc.set_font( self.label_font )
            gc.set_fill_color( self.label_color_ )
            gc.set_text_position( x + tx, y + ty )
            gc.show_text( label )

        gc.restore_state()

#-- Overridden Methods ---------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Return the components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------

    def _components_at ( self, x, y, margin = 0 ):
        cx, cy, cdx, cdy = self.bounds
        idx, idy         = self.image_size
        ix, iy           = self._image_origin
        ix              += cx
        iy              += cy

        # Allow a 1-pixel extra margin around the contact to allow selecting
        # a contact with the mouse to not be quite so sensitive when using
        # relatively small contacts. Also allow an extra margin when releasing
        # the mouse button to allow for finger twitching:
        if xy_in_bounds( x, y, ( ix - 1 - margin, iy - 1 - margin,
                                 idx + 2 + 2 * margin, idy + 2 + 2 * margin ) ):
            return [ self ]
        return []

#-- Public Methods -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Edit the properties of the contact:
    #---------------------------------------------------------------------------

    def edit ( self ):
        self.edit_traits()

    #---------------------------------------------------------------------------
    #  Force the component to be updated:
    #---------------------------------------------------------------------------

    def update ( self ):
        self.container.update( True )

    #---------------------------------------------------------------------------
    #  Push the current state of a contact:
    #---------------------------------------------------------------------------

    def push_state ( self, state ):
        self._save_state, self.event_state = self.event_state, state

    #---------------------------------------------------------------------------
    #  Restores the previous state of a contact:
    #---------------------------------------------------------------------------

    def restore_state ( self ):
        if self._save_state is not None:
            self.event_state, self._save_state = self._save_state, None

#-- Event Handlers -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Handles the image being changed:
    #---------------------------------------------------------------------------

    def _image_changed ( self ):
        image       = self.image
        self._image = img = self.image_for( image )
        self.image_size   = size = ( img.width(), img.height() )
        center = ( size[0] / 2, size[1] / 2 )
        col    = image.find( '.' )
        if col >= 0:
            image = image[:col]
        items = image.split( '_' )
        if items >= 2:
            try:
                center = ( int( items[-2] ), int( items[-1] ) )
            except:
                pass
        self.image_center = center
        self._size = self._center = None
        self.redraw()

        # fixme: Why do we need to do this...?
        try:
            self.container.container.redraw()
        except:
            pass

    #---------------------------------------------------------------------------
    #  Handles the event state being changed:
    #---------------------------------------------------------------------------

    def _event_state_changed ( self ):
        self.image = self.controller.get_contact_image( self, self.event_state )

    #---------------------------------------------------------------------------
    #  Mouse event handlers:
    #---------------------------------------------------------------------------

#-- Stateless Event Handlers ---------------------------------------------------

    def _right_up_changed ( self, event ):
        event.handled           = True
        self.pointer            = 'arrow'
        self.window.mouse_owner = None
        if self.event_state[-6:] == '_hover':
            self.event_state = self.event_state[:-6]
        self.controller.popup_menu( self, event )

#-- 'unconnected' State -------------------------------------------------------------

    def unconnected_mouse_move ( self, event ):
        event.handled           = True
        self.event_state        = 'unconnected_hover'
        self.window.mouse_owner = self
        if self.controller.can_drag_link( self ):
            self.pointer = 'cross'
        else:
            self.pointer = 'hand'

#-- 'unconnected_hover' State --------------------------------------------------

    def unconnected_hover_mouse_move ( self, event ):
        self._hover_move( event, 'unconnected' )

    def unconnected_hover_left_down ( self, event ):
        if not self._begin_drag_special( event ):
            self._begin_drag_link( event )

    def unconnected_hover_left_up ( self, event ):
        self._handle_click( event, 'unconnected' )

    def unconnected_hover_left_dclick ( self, event ):
        self._edit_contact( event )

#-- 'partially_connected' State -------------------------------------------------------------

    def partially_connected_mouse_move ( self, event ):
        event.handled           = True
        self.pointer            = 'cross'
        self.event_state        = 'partially_connected_hover'
        self.window.mouse_owner = self

#-- 'partially_connected_hover' State --------------------------------------------------

    def partially_connected_hover_mouse_move ( self, event ):
        self._hover_move( event, 'partially_connected' )

    def partially_connected_hover_left_down ( self, event ):
        if not self._begin_drag_special( event ):
            self._begin_drag_link( event )

    def partially_connected_hover_left_up ( self, event ):
        self._handle_click( event, 'partially_connected' )

    def partially_connected_hover_left_dclick ( self, event ):
        self._edit_contact( event )

#-- 'connected' State -------------------------------------------------------------

    def connected_mouse_move ( self, event ):
        event.handled           = True
        self.pointer            = 'question arrow'
        self.event_state        = 'connected_hover'
        self.window.mouse_owner = self

#-- 'connected_hover' State --------------------------------------------------

    def connected_hover_mouse_move ( self, event ):
        event.handled = True
        if not self.xy_in_bounds( event ):
            self.event_state        = 'connected'
            self.window.mouse_owner = None
            self.pointer            = 'arrow'

    def connected_hover_left_down ( self, event ):
        self._begin_drag_special( event )

    def connected_hover_left_up ( self, event ):
        self._handle_click( event, 'connected' )

#-- Private Helper Methods -----------------------------------------------------

    #---------------------------------------------------------------------------
    #  Begins a 'drag link' operation:
    #---------------------------------------------------------------------------

    def _begin_drag_link ( self, event ):
        event.handled = True
        if self.controller.can_drag_link( self ):
            self.controller.begin_drag_link( self )
            self.window.mouse_owner = None
            canvas = self.container.container
            ex, ey = event.x, event.y
            dc     = OMDragContact( contact   = self,
                                    container = canvas,
                                    bounds    = ( ex, ey, 1, 1 ),
                                    anchor    = ( int( ex ), int( ey ) ) )
            dc.on_trait_change( self.controller.drag_contact_done, 'resized' )
            self.window.drag_resize( dc, canvas.bounds, event,
                                     drag_handler = dc.drag_update )

    #---------------------------------------------------------------------------
    #  Begins a special 'drag' operation (if requested):
    #---------------------------------------------------------------------------

    def _begin_drag_special ( self, event ):
        """ Begins a special 'drag' operation (if requested).
        """
        if not (event.control_down or event.shift_down or event.alt_down):
            return False

        event.handled = True
        if event.control_down:
            rc = self.controller.begin_control_drag( self, event )
        elif event.shift_down:
            rc = self.controller.begin_shift_drag( self, event )
        else:
            rc = self.controller.begin_alt_drag( self, event )

        if rc is None:
            self.window.mouse_owner = None

        return True

    #---------------------------------------------------------------------------
    #  Handle a mouse click event on the contact:
    #---------------------------------------------------------------------------

    def _handle_click ( self, event, state ):
        event.handled           = True
        self.event_state        = state
        self.window.mouse_owner = None
        self.controller.clicked_contact( self, event )

    #---------------------------------------------------------------------------
    #  Edits the contact:
    #---------------------------------------------------------------------------

    def _edit_contact ( self, event ):
        event.handled           = True
        self.window.mouse_owner = None
        self.pointer            = 'arrow'
        self.controller.edit_contact( self )

    #---------------------------------------------------------------------------
    #  Handles a mouse move event while over a unconnected/partially connected
    #  contact:
    #---------------------------------------------------------------------------

    def _hover_move ( self, event, state ):
        event.handled = True
        if self.xy_in_bounds( event ):
            self.window.mouse_owner = self
            if self.controller.can_drag_link( self ):
                self.pointer = 'cross'
            else:
                self.pointer = 'hand'
        else:
            self.window.mouse_owner = None
            self.pointer            = 'arrow'
            self.event_state        = state

    #---------------------------------------------------------------------------
    #  Computes the size/center of the Contact:
    #---------------------------------------------------------------------------

    def _compute_contact_size ( self ):
        # If there is no label, then the result is only based on the image:
        label = self.label
        if len( label ) == 0:
            self._size         = self.image_size
            self._center       = self.image_center
            self._image_origin = ( 0, 0 )
            return

        # Get the text metrics for the label:
        gc = self.gc_temp()
        gc.set_font( self.label_font )
        tdx, tdy, descent, leading = gc.get_full_text_extent( label )

        # Gather up all of the other pertinent info:
        cdx, cdy = self.image_size
        cx, cy   = self.image_center
        ox, oy   = self.label_offset
        lpos     = self.label_position_
        cpos     = self.position_

        # Convert any inside/outside settings to their correct
        # left/right/top/bottom settings:
        if lpos & (OUTSIDE | INSIDE):
            if cpos & (LEFT | RIGHT):
                if ((cpos & LEFT) != 0) ^ ((lpos & OUTSIDE) != 0):
                    lpos = ((lpos | RIGHT) & (~LEFT))
                else:
                    lpos = ((lpos | LEFT)  & (~RIGHT))

            if cpos & (TOP | BOTTOM):
                if ((cpos & TOP) != 0) ^ ((lpos & OUTSIDE) != 0):
                    lpos = ((lpos | BOTTOM) & (~TOP))
                else:
                    lpos = ((lpos | TOP)    & (~BOTTOM))

        # Compute the various horizontal values:
        if lpos & LEFT:
            tx   = 0
            ix   = tdx + ox
            cdx += ix
            cx  += ix
        elif lpos & RIGHT:
            tx   = cdx + ox
            ix   = 0
            cdx += tdx + ox
        else:
            tdx2 = tdx / 2
            tx   = max( 0, cx - tdx2 )
            ix   = max( 0, tdx2 - cx )
            left = max( tdx2, cx )
            cdx  = left + max( tdx - tdx2, cdx - cx )
            cx   = left

        # Compute the various vertical values:
        if lpos & TOP:
            ty   = cdy + oy
            iy   = 0
            cdy += tdy + oy
        elif lpos & BOTTOM:
            ty   = 0
            iy   = tdy + oy
            cdy += iy
            cy  += iy
        else:
            tdy2   = tdy / 2
            ty     = max( 0, cy - tdy2 ) - descent + leading
            iy     = max( 0, tdy2 - cy )
            bottom = max( tdy2, cy )
            cdy    = bottom + max( tdy - tdy2, cdy - cy )
            cy     = bottom

        # Save the final results:
        self._size         = ( int( cdx ), int( cdy ) )
        self._center       = ( int( cx ), int( cy ) )
        self._text_origin  = ( int( tx ), int( ty ) )
        self._image_origin = ( int( ix ), int( iy ) )

#-- Pickling Protocol ----------------------------------------------------------

    def __getstate__ ( self ):
        dict = self.__dict__.copy()
        try:
            del dict[ '_image' ]
            del dict[ 'image_size' ]
            del dict[ 'image_center' ]
        except:
            pass
        return dict

    def __setstate__ ( self, state ):
        self.__dict__.update( state )
        self._image_changed()

#-------------------------------------------------------------------------------
#  'OMDragContact' class:
#-------------------------------------------------------------------------------

class OMDragContact ( Component ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # Starting contact:
    contact = Any

    # Contact dragged to:
    drag_contact = Any

    # Drag anchor point:
    anchor  = OffsetXY

    # Opposite achor point:
    anchor2 = Property

    # Color of the drag line:
    color = ERGBAColor( 'black' )

    #---------------------------------------------------------------------------
    #  Handles a drag update event while performing a drag link operation:
    #---------------------------------------------------------------------------

    def drag_update ( self, event, drag_bounds ):
        contact      = self.contact
        drag_contact = contact.controller.drag_update( contact, event )
        if drag_contact is not None:
            self.drag_contact = drag_contact

        return drag_bounds

#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Returns the alternate anchor point:
    #---------------------------------------------------------------------------

    def _get_anchor2 ( self ):
        x1, y1 = self.anchor
        x2     = self.left
        if x1 == x2:
            x2 = self.right
        y2 = self.bottom
        if y1 == y2:
            y2 = self.top
        return ( int( x2 ), int( y2 ) )

    #---------------------------------------------------------------------------
    #  Draws the drag contact:
    #---------------------------------------------------------------------------

    def _draw ( self, gc ):
        gc.save_state()

        x1, y1 = self.anchor
        x2, y2 = self.anchor2

        if x2 > x1:
            x2 -= 0.5
            x1 += 0.5
        else:
            x2 += 0.5
            x1 -= 0.5

        if y2 > y1:
            y2 -= 0.5
            y1 += 0.5
        else:
            y2 += 0.5
            y1 -= 0.5

        gc.set_stroke_color( self.color_ )
        gc.set_line_width( 1 )
        gc.begin_path()
        gc.move_to( x1, y1 )
        gc.line_to( x2, y2 )
        gc.stroke_path()

        gc.restore_state()

