#-------------------------------------------------------------------------------
#
#  Define a base set of constants and functions used by the remainder of the
#  Enable package.
#
#  Written by: David C. Morrill
#
#  Date: 09/22/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Functions defined: bounding_box
#                     intersect_coordinates
#                     union_coordinates
#                     intersect_bounds
#                     union_bounds
#                     disjoint_intersect_coordinates
#                     does_disjoint_intersect_coordinates
#                     bounding_coordinates
#                     bounds_to_coordinates
#                     coordinates_to_bounds
#                     coordinates_to_size
#                     add_rectangles
#                     xy_in_bounds
#                     gc_image_for
#                     send_event_to
#                     subclasses_of
#
#  Constants:         half_pixel_bounds_inset
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from __future__ import generators

import sys

from os.path   import dirname, splitext, abspath, join
from enthought.util.numerix import array, UnsignedInt8
from types     import TypeType, TupleType
from zipfile   import ZipFile, is_zipfile
from cStringIO import StringIO

from enthought.traits.api import TraitError

from enthought.kiva import GraphicsContext, font_metrics_provider
from enthought.kiva.backend_image import GraphicsContext as GraphicsContextArray
from enthought.kiva.constants import DEFAULT, DECORATIVE, ROMAN, SCRIPT, SWISS,\
                                     MODERN, NORMAL, BOLD, ITALIC
from enthought.kiva.fonttools import Font
from enthought.kiva.fonttools import str_to_font as kiva_str_to_font
from enthought.kiva.backend_image import Image, FontType

#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

# Special 'empty rectangle' indicator:
empty_rectangle = -1

half_pixel_bounds_inset = ( 0.5, 0.5, -1.0, -1.0 )

# Positions:
TOP          = 32
VCENTER      = 16
BOTTOM       =  8
LEFT         =  4
HCENTER      =  2
RIGHT        =  1

TOP_LEFT     = TOP    + LEFT
TOP_RIGHT    = TOP    + RIGHT
BOTTOM_LEFT  = BOTTOM + LEFT
BOTTOM_RIGHT = BOTTOM + RIGHT

# Text engraving style:
ENGRAVED = 1
EMBOSSED = 2
SHADOWED = 3

engraving_style = {
    'none':     0,
    'engraved': ENGRAVED,
    'embossed': EMBOSSED,
    'shadowed': SHADOWED
}

#-------------------------------------------------------------------------------
#  Standard Colors and Line Styles:
#-------------------------------------------------------------------------------

transparent_color = ( 0.0, 0.0, 0.0, 0.0 )

# Dictionary of standard colors:
standard_colors = {
   'aquamarine':          ( 0.439216, 0.858824, 0.576471, 1.0 ),
   'black':               ( 0.000000, 0.000000, 0.000000, 1.0 ),
   'blue':                ( 0.000000, 0.000000, 1.000000, 1.0 ),
   'blue violet':         ( 0.623529, 0.372549, 0.623529, 1.0 ),
   'brown':               ( 0.647059, 0.164706, 0.164706, 1.0 ),
   'cadet blue':          ( 0.372549, 0.623529, 0.623529, 1.0 ),
   'coral':               ( 1.000000, 0.498039, 0.000000, 1.0 ),
   'cornflower blue':     ( 0.258824, 0.258824, 0.435294, 1.0 ),
   'cyan':                ( 0.000000, 1.000000, 1.000000, 1.0 ),
   'dark green':          ( 0.184314, 0.309804, 0.184314, 1.0 ),
   'dark grey':           ( 0.184314, 0.184314, 0.184314, 1.0 ),
   'dark olive green':    ( 0.309804, 0.309804, 0.184314, 1.0 ),
   'dark orchid':         ( 0.600000, 0.196078, 0.800000, 1.0 ),
   'dark slate blue':     ( 0.419608, 0.137255, 0.556863, 1.0 ),
   'dark slate grey':     ( 0.184314, 0.309804, 0.309804, 1.0 ),
   'dark turquoise':      ( 0.439216, 0.576471, 0.858824, 1.0 ),
   'dim grey':            ( 0.329412, 0.329412, 0.329412, 1.0 ),
   'firebrick':           ( 0.556863, 0.137255, 0.137255, 1.0 ),
   'forest green':        ( 0.137255, 0.556863, 0.137255, 1.0 ),
   'gold':                ( 0.800000, 0.498039, 0.196078, 1.0 ),
   'goldenrod':           ( 0.858824, 0.858824, 0.439216, 1.0 ),
   'green':               ( 0.000000, 1.000000, 0.000000, 1.0 ),
   'green yellow':        ( 0.576471, 0.858824, 0.439216, 1.0 ),
   'grey':                ( 0.501961, 0.501961, 0.501961, 1.0 ),
   'indian red':          ( 0.309804, 0.184314, 0.184314, 1.0 ),
   'khaki':               ( 0.623529, 0.623529, 0.372549, 1.0 ),
   'light blue':          ( 0.749020, 0.847059, 0.847059, 1.0 ),
   'light grey':          ( 0.752941, 0.752941, 0.752941, 1.0 ),
   'light steel':         ( 0.000000, 0.000000, 0.000000, 1.0 ),
   'lime green':          ( 0.196078, 0.800000, 0.196078, 1.0 ),
   'magenta':             ( 1.000000, 0.000000, 1.000000, 1.0 ),
   'maroon':              ( 0.556863, 0.137255, 0.419608, 1.0 ),
   'medium aquamarine':   ( 0.196078, 0.800000, 0.600000, 1.0 ),
   'medium blue':         ( 0.196078, 0.196078, 0.800000, 1.0 ),
   'medium forest green': ( 0.419608, 0.556863, 0.137255, 1.0 ),
   'medium goldenrod':    ( 0.917647, 0.917647, 0.678431, 1.0 ),
   'medium orchid':       ( 0.576471, 0.439216, 0.858824, 1.0 ),
   'medium sea green':    ( 0.258824, 0.435294, 0.258824, 1.0 ),
   'medium slate blue':   ( 0.498039, 0.000000, 1.000000, 1.0 ),
   'medium spring green': ( 0.498039, 1.000000, 0.000000, 1.0 ),
   'medium turquoise':    ( 0.439216, 0.858824, 0.858824, 1.0 ),
   'medium violet red':   ( 0.858824, 0.439216, 0.576471, 1.0 ),
   'midnight blue':       ( 0.184314, 0.184314, 0.309804, 1.0 ),
   'navy':                ( 0.137255, 0.137255, 0.556863, 1.0 ),
   'orange':              ( 0.800000, 0.196078, 0.196078, 1.0 ),
   'orange red':          ( 1.000000, 0.000000, 0.498039, 1.0 ),
   'orchid':              ( 0.858824, 0.439216, 0.858824, 1.0 ),
   'pale green':          ( 0.560784, 0.737255, 0.560784, 1.0 ),
   'pink':                ( 0.737255, 0.560784, 0.917647, 1.0 ),
   'plum':                ( 0.917647, 0.678431, 0.917647, 1.0 ),
   'purple':              ( 0.690196, 0.000000, 1.000000, 1.0 ),
   'red':                 ( 1.000000, 0.000000, 0.000000, 1.0 ),
   'salmon':              ( 0.435294, 0.258824, 0.258824, 1.0 ),
   'sea green':           ( 0.137255, 0.556863, 0.419608, 1.0 ),
   'sienna':              ( 0.556863, 0.419608, 0.137255, 1.0 ),
   'sky blue':            ( 0.196078, 0.600000, 0.800000, 1.0 ),
   'slate blue':          ( 0.000000, 0.498039, 1.000000, 1.0 ),
   'spring green':        ( 0.000000, 1.000000, 0.498039, 1.0 ),
   'steel blue':          ( 0.137255, 0.419608, 0.556863, 1.0 ),
   'tan':                 ( 0.858824, 0.576471, 0.439216, 1.0 ),
   'thistle':             ( 0.847059, 0.749020, 0.847059, 1.0 ),
   'turquoise':           ( 0.678431, 0.917647, 0.917647, 1.0 ),
   'violet':              ( 0.309804, 0.184314, 0.309804, 1.0 ),
   'violet red':          ( 0.800000, 0.196078, 0.600000, 1.0 ),
   'wheat':               ( 0.847059, 0.847059, 0.749020, 1.0 ),
   'white':               ( 1.000000, 1.000000, 1.000000, 1.0 ),
   'yellow':              ( 1.000000, 1.000000, 0.000000, 1.0 ),
   'yellow green':        ( 0.600000, 0.800000, 0.196078, 1.0 ),
   'clear':               transparent_color,
   'transparent':         transparent_color,
   'none':                transparent_color
}

# Colors:
black_color  = standard_colors[ 'black' ]
white_color  = standard_colors[ 'white' ]
red_color    = standard_colors[ 'red' ]
green_color  = standard_colors[ 'green' ]
yellow_color = standard_colors[ 'yellow' ]
blue_color   = standard_colors[ 'blue' ]

#-------------------------------------------------------------------------------
#  Convert a string into a valid 'font' object (if possible):
#-------------------------------------------------------------------------------

# Default font:
# !! We really should search for these.
default_font_name = 'modern 12'
#if FontType('default').name == '':
#    default_font_name = 'bitstream vera sans 10'

font_families = {
   'default':    DEFAULT,
   'decorative': DECORATIVE,
   'roman':      ROMAN,
   'script':     SCRIPT,
   'swiss':      SWISS,
   'modern':     MODERN
}

font_styles = {
   'italic': ITALIC
}

font_weights = {
   'bold': BOLD
}

font_noise = [ 'pt', 'point', 'family' ]

def str_to_font ( object, name, value ):
    try:
        return kiva_str_to_font(value)
    except:
        raise TraitError, ( object, name, 'a font descriptor string',
                        repr( value ) )

str_to_font.info = ( "a string describing a font (e.g. '12 pt bold italic "
                     "swiss family Arial' or 'default 12')" )

default_font = str_to_font( None, None, default_font_name )

#-------------------------------------------------------------------------------
#  Non-drawing temporary Kiva graphics context:
#  This is currently only used for calculating font metrics.
#-------------------------------------------------------------------------------

#gc_temp = GraphicsContext( ( 1, 1 ) )
gc_temp = font_metrics_provider()

#-------------------------------------------------------------------------------
#  'GraphicsContextEnable' class:
#-------------------------------------------------------------------------------

class GraphicsContextEnable ( GraphicsContext ):

    #---------------------------------------------------------------------------
    #  Set up the Kiva clipping rectangle using the (incorrect) Kiva clipping model:
    #---------------------------------------------------------------------------

    def clip_to_rect ( self, x, y, dx, dy ):
        super( GraphicsContext, self ).clip_to_rect(x, y, dx, dy)

    #---------------------------------------------------------------------------
    #  Clip and clear a Kiva graphics context to a specified area and color:
    #---------------------------------------------------------------------------

    def clear_clip ( self, color, coordinates ):
        bounds = coordinates_to_bounds( coordinates )
        self.clip_to_rect( *bounds )
        self.set_fill_color( color )
        self.begin_path()
        self.rect( *bounds )
        self.fill_path()

    #---------------------------------------------------------------------------
    #  Clip and clear a Kiva graphics context to a specified region and color:
    #---------------------------------------------------------------------------

    def clear_clip_region ( self, color, update_region ):
        bounds = coordinates_to_bounds( bounding_coordinates( update_region ) )
        self.clip_to_rect( *bounds )
        self.set_fill_color( color )
        for coordinates in update_region:
            bounds = coordinates_to_bounds( coordinates )
            self.begin_path()
            self.rect( *bounds )
        self.fill_path()

    #---------------------------------------------------------------------------
    #  Sets the 'alpha' channel of a Kiva graphics context to a specified value:
    #---------------------------------------------------------------------------

    #~ def alpha ( self, alpha ):
        #~ self.bmp_array[:,:,3] = (255.0 * (alpha * self.bmp_array[:,:,3] / 255.0)
                              #~ ).astype( UnsignedInt8 )
        #~ # Much faster, but works only on 32 bit images with alpha as the
        #~ # last channel
        #~ #graphics_context_multiply_alpha(alpha, self.bmp_array)

    def alpha(self, alpha):
        raise NotImplementedError, \
            "The alpha() method is not compatible with DisplayPDF; use clear() instead."

    #---------------------------------------------------------------------------
    #  Draws an image 'stretched' to fit a specified area:
    #---------------------------------------------------------------------------

    def stretch_draw ( self, image, x, y, dx, dy ):
        idx  = image.width()
        idy  = image.height()
        self.save_state()
        self.clip_to_rect(x, y, dx, dy)
        #if clip is not empty_rectangle:
        #cx, cy, cdx, cdy = clip
        cx, cy, cdx, cdy = x, y, dx, dy
        yt = cy + cdy
        xr = cx + cdx
        x += (int( cx - x ) / idx) * idx
        y += (int( cy - y ) / idy) * idy
        while y < yt:
            x0 = x
            while x0 < xr:
                self.draw_image( image, ( x0, y, idx, idy ) )
                x0 += idx
            y += idy
        #self.pop_clip()
        self.restore_state()

    #---------------------------------------------------------------------------
    #  Draw a text string within a specified area with various attributes:
    #---------------------------------------------------------------------------

    def text ( self, text, bounds,
                     font         = default_font,
                     color        = black_color,
                     shadow_color = white_color,
                     style        = 0,
                     alignment    = LEFT,
                     y_offset     = 0.0,
                     info         = None ):

            #self.push_clip( bounds )

            # Set up the font and get the text bounding box information:
            self.set_font( font )
            if info is None:
                info = self.get_full_text_extent( text )
            tdx, tdy, descent, leading = info

            # Calculate the text position, based on its alignment:
            xl, yb, dx, dy = bounds
            y = yb + y_offset + (dy - tdy) / 2.0
            if alignment & LEFT:
                x = xl
            elif alignment & RIGHT:
                x = xl + dx - tdx - 4.0  # 4.0 is a hack!!!
            else:
                x = xl + (dx - tdx) / 2.0

            self.save_state()
            self.clip_to_rect(*bounds)

            # Draw the normal text:
            self.set_fill_color( color )
            self.show_text(text, (x,y))

            #self.pop_clip()
            self.restore_state()
            self.set_fill_color( color )

#-------------------------------------------------------------------------------
#  Draws a filled rectangle with border:
#-------------------------------------------------------------------------------

def filled_rectangle ( gc, bounds,
                           bg_color     = ( 1.0, 1.0, 1.0, 1.0 ),
                           border_color = ( 0.0, 0.0, 0.0, 1.0 ),
                           border_size  = 1.0 ):
    """ Draws a filled rectangle with border.
    """
    gc.save_state()

    # Set up all the control variables for quick access:
    bsd = border_size + border_size
    bsh = border_size / 2.0
    x, y, dx, dy = bounds

    # Fill the background region (if required):
    if bg_color is not transparent_color:
        gc.set_fill_color( bg_color )
        gc.begin_path()
        gc.rect( x + border_size, y + border_size, dx - bsd, dy - bsd )
        gc.fill_path()

    # Draw the border (if required):
    if border_size > 0:
        if border_color is not transparent_color:
            gc.set_stroke_color( border_color )
            gc.set_line_width( border_size )
            gc.begin_path()
            gc.rect( x + bsh, y + bsh, dx - border_size, dy - border_size )
            gc.stroke_path()

    gc.restore_state()

#-------------------------------------------------------------------------------
#  Convert an image file name to a cached Kiva gc containing the image:
#-------------------------------------------------------------------------------

# Image cache dictionary (indexed by 'normalized' filename):
_image_cache = {}
_zip_cache   = {}
_app_path    = None
_enable_path = None

def gc_image_for ( name, path = None ):
    global _app_path, _enable_path
    filename = name
    if dirname( name ) == '':
        name = name.replace( ' ', '_' )
        if splitext( name )[1] == '':
            name += '.png'
        if path is None:
           if _enable_path is None:
              import enthought.enable.base
              _enable_path = join( dirname( enthought.enable.base.__file__ ),
                                   'images' )
           path = _enable_path
        elif path == '':
           if _app_path is None:
              _app_path = join( dirname( sys.argv[0] ), 'images' )
           path = _app_path
        else:
            if not isinstance(path, basestring):
                if not isinstance( path, TypeType ):
                    path = path.__class__
                path = join( dirname( sys.modules[ path.__module__ ].__file__ ),
                             'images' )
        filename = join( path, name.replace( ' ', '_' ).lower() )
    else:
        path = None
    filename = abspath( filename )
    image     = _image_cache.get( filename )
    if image is None:
        cachename = filename
        if path is not None:
            zip_path = abspath( path + '.zip' )
            zip_file = _zip_cache.get( zip_path )
            if zip_file is None:
               if is_zipfile( zip_path ):
                   zip_file = ZipFile( zip_path, 'r' )
               else:
                   zip_file = False
               _zip_cache[ zip_path ] = zip_file
            if isinstance( zip_file, ZipFile ):
                try:
                    filename = StringIO( zip_file.read( name ) )
                except:
                    pass
        try:
            _image_cache[ cachename ] = image = Image( filename )
        except:
            _image_cache[ filename ] = info = sys.exc_info()[:2]
            raise info[0], info[1]
    elif type( image ) is TupleType:
        raise image[0], image[1]
    return image

#-------------------------------------------------------------------------------
#  Compute the bounding box for a set of components:
#-------------------------------------------------------------------------------

def bounding_box ( components ):
    bxl, byb, bxr, byt = bounds_to_coordinates( components[0].bounds )
    for component in components[1:]:
        xl, yb, xr, yt = bounds_to_coordinates( component.bounds )
        bxl = min( bxl, xl )
        byb = min( byb, yb )
        bxr = max( bxr, xr )
        byt = max( byt, yt )
    return ( bxl, byb, bxr, byt )

#-------------------------------------------------------------------------------
#  Compute the intersection of two coordinate based rectangles:
#-------------------------------------------------------------------------------

def intersect_coordinates ( coordinates1, coordinates2 ):
    if (coordinates1 is empty_rectangle) or ( coordinates2 is empty_rectangle):
        return empty_rectangle
    xl1, yb1, xr1, yt1 = coordinates1
    xl2, yb2, xr2, yt2 = coordinates2
    xl = max( xl1, xl2 )
    yb = max( yb1, yb2 )
    xr = min( xr1, xr2 )
    yt = min( yt1, yt2 )
    if (xr > xl) and (yt > yb):
        return ( xl, yb, xr, yt )
    return empty_rectangle

#-------------------------------------------------------------------------------
#  Compute the intersection of two bounds rectangles:
#-------------------------------------------------------------------------------

def intersect_bounds ( bounds1, bounds2 ):
    if (bounds1 is empty_rectangle) or (bounds2 is empty_rectangle):
        return empty_rectangle

    intersection = intersect_coordinates(
                        bounds_to_coordinates( bounds1 ),
                        bounds_to_coordinates( bounds2 ) )
    if intersection is empty_rectangle:
        return empty_rectangle
    xl, yb, xr, yt = intersection
    return ( xl, yb, xr - xl, yt - yb )

#-------------------------------------------------------------------------------
#  Compute the union of two coordinate based rectangles:
#-------------------------------------------------------------------------------

def union_coordinates ( coordinates1, coordinates2 ):
    if coordinates1 is empty_rectangle:
        return coordinates2
    elif coordinates2 is empty_rectangle:
        return coordinates1
    xl1, yb1, xr1, yt1 = coordinates1
    xl2, yb2, xr2, yt2 = coordinates2
    return ( min( xl1, xl2 ), min( yb1, yb2 ),
             max( xr1, xr2 ), max( yt1, yt2 ) )

#-------------------------------------------------------------------------------
#  Compute the union of two bounds rectangles:
#-------------------------------------------------------------------------------

def union_bounds ( bounds1, bounds2 ):
    xl, yb, xr, yt = union_coordinates(
                        bounds_to_coordinates( bounds1 ),
                        bounds_to_coordinates( bounds2 ) )
    if xl is None:
        return empty_rectangle
    return ( xl, yb, xr - xl, yt - yb )

#-------------------------------------------------------------------------------
#  Return the disjoint union of an already disjoint list of rectangles and a
#  new rectangle:
#
#  Note: The 'infinite' area rectangle is indicated by 'None'. The coordinates
#        list may be empty.
#-------------------------------------------------------------------------------

def disjoint_union_coordinates ( coordinates_list, coordinates ):
    # If we already have an 'infinite' area, then we are done:
    if coordinates_list is None:
        return None

    result = []
    todo   = [ coordinates ]

    # Iterate over each item in the todo list:
    i = 0
    while i < len( todo ):
        xl1, yb1, xr1, yt1 = todo[i]
        j      = 0
        use_it = True

        # Iterate over each item in the original list of rectangles:
        while j < len( coordinates_list ):
            xl2, yb2, xr2, yt2 = coordinates_list[j]

            # Test for non-overlapping rectangles:
            if (xl1 >= xr2) or (xr1 <= xl2) or (yb1 >= yt2) or (yt1 <= yb2):
                j += 1
                continue

            # Test for rect 1 being wholly contained in rect 2:
            x1inx2 = ((xl1 >= xl2) and (xr1 <= xr2))
            y1iny2 = ((yb1 >= yb2) and (yt1 <= yt2))
            if x1inx2 and y1iny2:
                use_it = False
                break

            # Test for rect 2 being wholly contained in rect 1:
            x2inx1 = ((xl2 >= xl1) and (xr2 <= xr1))
            y2iny1 = ((yb2 >= yb1) and (yt2 <= yt1))
            if x2inx1 and y2iny1:
                del coordinates_list[j]
                continue

            # Test for rect 1 being within rect 2 along the x-axis:
            if x1inx2:
                if yb1 < yb2:
                    if yt1 > yt2:
                        todo.append( ( xl1, yt2, xr1, yt1 ) )
                    yt1 = yb2
                else:
                    yb1 = yt2
                j += 1
                continue

            # Test for rect 2 being within rect 1 along the x-axis:
            if x2inx1:
                if yb2 < yb1:
                    if yt2 > yt1:
                        coordinates_list.insert( j, ( xl2, yt1, xr2, yt2 ) )
                        j += 1
                    coordinates_list[j] = ( xl2, yb2, xr2, yb1 )
                else:
                    coordinates_list[j] = ( xl2, yt1, xr2, yt2 )
                j += 1
                continue

            # Test for rect 1 being within rect 2 along the y-axis:
            if y1iny2:
                if xl1 < xl2:
                    if xr1 > xr2:
                        todo.append( ( xr2, yb1, xr1, yt1 ) )
                    xr1 = xl2
                else:
                    xl1 = xr2
                j += 1
                continue

            # Test for rect 2 being within rect 1 along the y-axis:
            if y2iny1:
                if xl2 < xl1:
                    if xr2 > xr1:
                        coordinates_list.insert( j, ( xr1, yb2, xr2, yt2 ) )
                        j += 1
                    coordinates_list[j] = ( xl2, yb2, xl1, yt2 )
                else:
                    coordinates_list[j] = ( xr1, yb2, xr2, yt2 )
                j += 1
                continue

            # Handle a 'corner' overlap of rect 1 and rect 2:
            if xl1 < xl2:
                xl = xl1
                xr = xl2
            else:
                xl = xr2
                xr = xr1
            if yb1 < yb2:
                yb  = yb2
                yt  = yt1
                yt1 = yb2
            else:
                yb  = yb1
                yt  = yt2
                yb1 = yt2
            todo.append( ( xl, yb, xr, yt ) )
            j += 1

        # If there is anything left of rect 1 to use, add it to the result:
        if use_it:
            result.append( ( xl1, yb1, xr1, yt1 ) )

        # Advance to the next rectangle in the todo list:
        i += 1

    # Return whatever's left in the original list plus whatever made it to the
    # result:
    return coordinates_list + result

#-------------------------------------------------------------------------------
#  Return the disjoint intersection of an already disjoint list of rectangles
#  and a new rectangle:
#
#  Note: The 'infinite' area rectangle is indicated by 'None'. The coordinates
#        list may be empty.
#-------------------------------------------------------------------------------

def disjoint_intersect_coordinates ( coordinates_list, coordinates ):
    # If new rectangle is empty, the result is empty:
    if coordinates is empty_rectangle:
        return []

    # If we have an 'infinite' area, then return the new rectangle:
    if coordinates_list is None:
        return [ coordinates ]

    result             = []
    xl1, yb1, xr1, yt1 = coordinates

    # Intersect the new rectangle against each rectangle in the list:
    for xl2, yb2, xr2, yt2 in coordinates_list:
        xl = max( xl1, xl2 )
        yb = max( yb1, yb2 )
        xr = min( xr1, xr2 )
        yt = min( yt1, yt2 )
        if (xr > xl) and (yt > yb):
            rectangle = ( xl, yb, xr, yt )
            result.append( rectangle )
            if rectangle == coordinates:
                break
    return result

#-------------------------------------------------------------------------------
#  Return whether a rectangle intersects a disjoint set of rectangles anywhere:
#-------------------------------------------------------------------------------

def does_disjoint_intersect_coordinates ( coordinates_list, coordinates ):
    # If new rectangle is empty, the result is empty:
    if coordinates is empty_rectangle:
        return False

    # If we have an 'infinite' area, then return the new rectangle:
    if coordinates_list is None:
        return True

    # Intersect the new rectangle against each rectangle in the list until an
    # non_empty intersection is found:
    xl1, yb1, xr1, yt1 = coordinates
    for xl2, yb2, xr2, yt2 in coordinates_list:
        if ((min( xr1, xr2 ) > max( xl1, xl2 )) and
            (min( yt1, yt2 ) > max( yb1, yb2 ))):
            return True
    return False

#-------------------------------------------------------------------------------
#  Return the bounding rectangle for a list of rectangles:
#-------------------------------------------------------------------------------

def bounding_coordinates ( coordinates_list ):
    if coordinates_list is None:
        return None
    if len( coordinates_list ) == 0:
        return empty_rectangle
    xl, yb, xr, yt = 1.0E10, 1.0E10, -1.0E10, -1.0E10
    for xl1, yb1, xr1, yt1 in coordinates_list:
        xl = min( xl, xl1 )
        yb = min( yb, yb1 )
        xr = max( xr, xr1 )
        yt = max( yt, yt1 )
    return ( xl, yb, xr, yt )

#-------------------------------------------------------------------------------
#  Convert a bounds rectangle to a coordinate rectangle:
#-------------------------------------------------------------------------------

def bounds_to_coordinates ( bounds ):
    x, y, dx, dy = bounds
    return ( x, y, x + dx, y + dy )

#-------------------------------------------------------------------------------
#  Convert a coordinates rectangle to a bounds rectangle:
#-------------------------------------------------------------------------------

def coordinates_to_bounds ( coordinates ):
    xl, yb, xr, yt = coordinates
    return ( xl, yb, xr - xl, yt - yb )

#-------------------------------------------------------------------------------
#  Convert a coordinates rectangle to a size tuple:
#-------------------------------------------------------------------------------

def coordinates_to_size ( coordinates ):
    xl, yb, xr, yt = coordinates
    return ( xr - xl, yt - yb )

#-------------------------------------------------------------------------------
#  Add two bounds or coordinate rectangles:
#-------------------------------------------------------------------------------

def add_rectangles ( rectangle1, rectangle2 ):
    return ( rectangle1[0] + rectangle2[0],
             rectangle1[1] + rectangle2[1],
             rectangle1[2] + rectangle2[2],
             rectangle1[3] + rectangle2[3] )

#-------------------------------------------------------------------------------
#  Test whether a specified (x,y) point is in a specified bounds:
#-------------------------------------------------------------------------------

def xy_in_bounds ( x, y, bounds ):
    x0, y0, dx, dy = bounds
    return (x0 <= x < x0 + dx) and (y0 <= y < y0 + dy)

#-------------------------------------------------------------------------------
#  Send an event to a specified set of components until it is 'handled':
#-------------------------------------------------------------------------------

def send_event_to ( components, event_name, event ):
    pre_event_name = 'pre_' + event_name
    for component in components:
        setattr( component, pre_event_name, event )
        if event.handled:
            return len( components )
    for i in xrange( len( components ) - 1, -1, -1 ):
        setattr( components[i], event_name, event )
        if event.handled:
            return i
    return 0

#-------------------------------------------------------------------------------
#  Generate all of the classes (and subclasses) for a specified class:
#-------------------------------------------------------------------------------

def subclasses_of ( klass ):
    yield klass
    for subclass in klass.__bases__:
        for result in subclasses_of( subclass ):
            yield result

#-------------------------------------------------------------------------------
#  Interface for draggable objects that handle the 'dropped_on' event:
#-------------------------------------------------------------------------------

class IDroppedOnHandler:

    def was_dropped_on ( self, component, event ):
        raise NotImplementedError

