/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Author: Stéphane Démurget  <stephane.demurget@enst-bretagne.fr>
 */

#include <glib.h>
#include <glib/gi18n.h>
#include <stdio.h>
#include <string.h>

#include "ef-backend.h"

#define CPUFREQ_PATH "/sys/devices/system/cpu/cpu0/cpufreq/"
#define THERMALZONE_DIR "/proc/acpi/thermal_zone"

/* stuff allocated for the whole backend life cycle */
static char *temp_path = NULL;
static gboolean avail_freq_present = TRUE;

/* stuff to be allocated and freed for each reload */
static char *driver = NULL;
static GPtrArray *avail_governors = NULL;
static GPtrArray *avail_frequencies = NULL;
static char *governor = NULL;

static int freq = -1;
static int temp = -1;

GQuark emifreq_error_quark ()
{
	static GQuark q = 0;

	if (q == 0)
		q = g_quark_from_static_string ("emifreq-error-quark");

	return q;
}

static char *
read_text_from (const char *path, gboolean optional)
{
	GIOChannel *channel;
	GError *error = NULL;
	char *text = NULL;
	int len = -1;

	g_return_val_if_fail (path != NULL, NULL);

	channel = g_io_channel_new_file (path, "r", &error);

	if (channel == NULL) {
		/* this is not critical for scaling_setspeed, no need to display a warning */
		if (!optional)
			g_warning (_("Error while opening %s: %s\n"), path, error->message);

		g_error_free (error);

		return NULL;
	}

	if (g_io_channel_read_line (channel, &text, &len, NULL, &error) != G_IO_STATUS_NORMAL) {
		g_warning (_("Error while reading %s: %s\n"), path, error->message);
		g_error_free (error);
		g_io_channel_unref (channel);

		return NULL;
	}

	if (len > 1 && text[len - 1] == '\n')
		text[len - 1] = '\0';

	/* that will also close the channel because this is the only & last reference */
	g_io_channel_unref (channel);

	return g_strchomp (text);
}

static gpointer
parse_internal (char *text)
{
	g_return_val_if_fail (text != NULL, NULL);

	if (g_ascii_isdigit (text[0])) {
		int *value = g_new0 (int, 1);

		*value = (int) g_ascii_strtod (text, NULL);

		return value;
	}

	return g_strdup (text);
}

static int
parse_info (char *path, char *tag, gboolean optional)
{
	char *text;
	char *format;
	int val, n;

	g_return_val_if_fail (path != NULL, -1);
	g_return_val_if_fail (tag != NULL, -1);

	if ((text = read_text_from (path, optional)) == NULL)
		return -1;

	format = g_strdup_printf ("%s: %%d", tag);
	n = sscanf (text, format, &val);
	g_free (format);

	g_free (text);

	return (n == 1) ? val : -1;
}

static gpointer
parse (const char *path, gboolean as_array, gboolean optional)
{
	gpointer v = NULL;
	char *text;
	char **values = NULL;

	g_return_val_if_fail (path != NULL, NULL);

	if ((text = read_text_from (path, optional)) == NULL)
		return NULL;

	values = g_strsplit (text, " ", 0);
	g_free (text);

	if (values == NULL) {
		g_warning (_("Parse error in file %s\n"), path);

		return NULL;
	}		

	if (values[1] == NULL && !as_array) {
		v = parse_internal (values[0]);
	} else {
		GPtrArray *a = g_ptr_array_new ();
		int i;

		for (i = 0; values[i] != NULL; i++)
			g_ptr_array_add (a, parse_internal (values[i]));

		v = a;
	}

	g_strfreev (values);

	return v;
}

static EFBackendGovernors
get_governor_from_name (char *name)
{
	g_return_val_if_fail (name != NULL, 0);

	if (!strcmp (name, "userspace"))
		return GOV_USERSPACE;

	if (!strcmp (name, "powersave"))
		return GOV_POWERSAVE;

	if (!strcmp (name, "performance"))
		return GOV_PERFORMANCE;

	if (!strcmp (name, "ondemand"))
		return GOV_ONDEMAND;

	/* Note: to be rock solid, we don't assert here because it's possible
	         to reach this point if the sysfs support is removed */
	return 0;
}

const char *
ef_backend_get_profile_name ()
{
	EFBackendGovernors gov = get_governor_from_name (governor);

	switch (gov) {
		case GOV_PERFORMANCE: return _("Performance profile");
		case GOV_POWERSAVE:   return _("Powersaving profile");
		case GOV_ONDEMAND:    return _("Automatic management");
		default:	      return _("Fixed speed");
	}

	return NULL;
}

static const char *
get_governor_name (EFBackendGovernors gov)
{
	switch (gov) {
		case GOV_PERFORMANCE: return "performance";
		case GOV_USERSPACE:   return "userspace";
		case GOV_POWERSAVE:   return "powersave";
		case GOV_ONDEMAND:    return "ondemand";
	}

	/* This MAY happen if the sysfs support is removed, so this IS NOT
	   an assert point */

	return NULL;
}

static int min_frequency () {
	if (avail_frequencies == NULL)
		return -1;

	g_assert (avail_frequencies->len > 0);

	return *((int *) avail_frequencies->pdata[0]);
}

static int max_frequency () {
	if (avail_frequencies == NULL)
		return -1;

	g_assert (avail_frequencies->len > 0);

	return *((int *) avail_frequencies->pdata[avail_frequencies->len - 1]);
}

static void
free_internal ()
{
	if (driver) g_free (driver);

	if (avail_governors) g_ptr_array_free (avail_governors, TRUE);
	if (avail_frequencies) g_ptr_array_free (avail_frequencies, TRUE);

	if (governor) g_free (governor);
}

gboolean
ef_backend_is_available ()
{
	return g_file_test (CPUFREQ_PATH, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR);
}

gboolean
ef_backend_init (GError **error)
{
	GDir *dir;

	if (!ef_backend_is_available ()) {
		g_set_error (error, EMIFREQ_ERROR,
			     EMIFREQ_INIT_ERROR,
			     _("The applet didn't manage to find CpuFreq support in the sysfs. You need your kernel to have built-in support for CpuFreq or use the module corresponding to your processor type in order to run this applet."));

		return FALSE;
	}

	dir = g_dir_open (THERMALZONE_DIR, 0, error);

	if (dir != NULL) {
		const gchar *name;

		while ((name = g_dir_read_name (dir)) != NULL) {
			temp_path = g_build_filename (THERMALZONE_DIR, name, "temperature", NULL);

			if (g_file_test (temp_path, G_FILE_TEST_EXISTS))
				break;

			g_free (temp_path);
			temp_path = NULL;
		}

		g_dir_close (dir);
	}

	/* not having any thermal zone is not an assert point anymore, we simply
	   not show it */

	ef_backend_reload ();

	return TRUE;
}

gint
freq_compare (gconstpointer a, gconstpointer b)
{
	int f1 = **((int **) a);
	int f2 = **((int **) b);

	return f1 - f2;
}

GPtrArray *
load_avail_frequencies ()
{
	GPtrArray *af = (GPtrArray *) parse (CPUFREQ_PATH "scaling_available_frequencies", TRUE, TRUE);

	/* there's no file listing all the available frequencies, so it's
	   certainly a driver with just a low and high frequency */
	if (af == NULL) {
		int *min_freq_p = (int *) parse (CPUFREQ_PATH "scaling_min_freq", FALSE, FALSE);
		int *max_freq_p = (int *) parse (CPUFREQ_PATH "scaling_max_freq", FALSE, FALSE);

		af = g_ptr_array_new ();

		g_ptr_array_add (af, min_freq_p);
		g_ptr_array_add (af, max_freq_p);
	} else {
		int i, last = -1;

		/* the contents of this file is not sorted with some drivers */
		g_ptr_array_sort (af, freq_compare);

		/* and sometimes there's the same value multiple times */
		for (i = 0; i < af->len; ) {
			int current = *((int *) af->pdata[i]);

			if (i > 0 && current == last) {
				g_free (af->pdata[i]);
				g_ptr_array_remove_index (af, i);

				continue;
			}

			last = current;
			i++;
		}		
	}

	return af;
}

void
ef_backend_reload ()
{
	int *p;

	free_internal ();

	/* get the common stuff */
	driver = (char *) parse (CPUFREQ_PATH "scaling_driver", FALSE, FALSE);

	avail_governors = (GPtrArray *) parse (CPUFREQ_PATH "scaling_available_governors", TRUE, TRUE);
	governor = (char *) parse (CPUFREQ_PATH "scaling_governor", FALSE, FALSE);

	avail_frequencies = load_avail_frequencies ();

	p = (int *) parse (CPUFREQ_PATH "scaling_cur_freq", FALSE, TRUE);

	if (p == NULL)
		freq = -1;
	else {
		freq = *p;
		g_free (p);
	}

	if (temp_path != NULL)
		temp = parse_info (temp_path, "temperature", TRUE);
}

void
ef_backend_reload_usage ()
{
	if (avail_frequencies) g_ptr_array_free (avail_frequencies, TRUE);
	if (governor) g_free (governor);

	avail_frequencies = load_avail_frequencies ();
	governor = (char *) parse (CPUFREQ_PATH "scaling_governor", FALSE, FALSE);
}

char *ef_backend_get_driver ()        { return driver; }
char *ef_backend_get_governor_name () { return governor; }

int
ef_backend_get_temperature (gboolean use_metric)
{
	if (temp == -1)
		return -1;

	if (use_metric)
		return temp;

	return (temp * 9/5) + 32;
}

EFBackendGovernors ef_backend_get_governor () {
       return get_governor_from_name (governor);
}

int
ef_backend_get_frequency () {
	return freq;
}

float
ef_backend_get_usage ()
{
	int min = min_frequency ();
	int current = ef_backend_get_frequency ();
	int max = max_frequency ();

	g_return_val_if_fail (min != -1, -1.0);
	g_return_val_if_fail (min != max, -1.0);

	return 100.0 * (current - (float) min) / (max - (float) min);
}

/* returns the avail governors as a OR'ed bits mask */
EFBackendGovernors
ef_backend_get_avail_governors ()
{
	int val = 0;
	int i = 0;

	g_return_val_if_fail (avail_governors != NULL, -1);

	for (i = 0; i < avail_governors->len; i++)
		val += get_governor_from_name (avail_governors->pdata[i]);

	return val;
}


int
ef_backend_governors_count (EFBackendGovernors gov)
{
	int count = 0;
	int i;

	for (i = 0; ; i++) {
		int value = (1 << i);

		if (value == GOV_LAST)
		       break;

		if ((gov & value) == value)
			count++;
	}

	return count;
}

/* mustn't be freed ! */
GPtrArray *
ef_backend_get_avail_frequencies ()
{
	return avail_frequencies;
}

gboolean
ef_backend_is_avail_frequency (int freq)
{
	int i;

	g_return_val_if_fail (avail_frequencies != NULL, FALSE);

	for (i = 0; i < avail_frequencies->len; i++)
		if (*((int *) avail_frequencies->pdata[i]) == freq)
			return TRUE;

	return FALSE;
}

int
ef_backend_get_medium_frequency ()
{
	int avail_frequencies_nb = avail_frequencies->len;

	g_assert (avail_frequencies_nb > 0);

	return *((int *) avail_frequencies->pdata[avail_frequencies_nb / 2]);
}

void
ef_backend_set_governor (EFBackendGovernors gov)
{
	FILE *f = fopen (CPUFREQ_PATH "scaling_governor", "wt");
	fprintf (f, "%s\n", get_governor_name (gov));
	fclose (f);
}

void
ef_backend_set_frequency (int freq)
{
	FILE *f = fopen (CPUFREQ_PATH "scaling_setspeed", "wt");
	fprintf (f, "%d\n", freq);
	fclose (f);
}

void
ef_backend_shutdown ()
{
	free_internal ();

	g_free (temp_path);
}
