/* $Id: e2_filelist.c 550 2007-07-22 10:30:40Z tpgww $

Copyright (C) 2004-2007 tooar <tooar@gmx.net>.

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
@file src/e2_filelist.c
@brief directory content liststore functions

This file contains functions related to creation, filling and emptying a
liststore of directory content data, and fns for the associated model.
Also filtering of store content, and other related utilties.
*/

#include "e2_filelist.h"
#include <string.h>
#include <sys/time.h>
#include <langinfo.h>
#include <pwd.h>
#include <time.h>
#include <grp.h>
#include "e2_option.h"
#include "e2_task.h"
#include "e2_dialog.h"

#ifdef E2_SELTXT_RECOLOR
extern GdkColor selectedtext;
#endif

#ifdef E2_FAM_DNOTIFY
#include <signal.h>
extern sigset_t dnotify_signal_set;
#endif

//disable-refresh counter for both filelists
gint refresh_refcount;
//disable-refresh counter for single filelists
gint refresh_active_refcount;
gint refresh_inactive_refcount;

//static gboolean case_sensitive; //local copy of option value, for faster access

/**
@brief determine whether pane defined by @a pane is the active one
@param pane enumerator for pane to be checked
@return TRUE if @a pane is the active one
*/
static gboolean _e2_filelist_check_active_pane (E2_ListChoice pane)
{
	gboolean active;
	switch (pane)
	{
		case PANE1:
			active = (curr_view == &app.pane1_view);
			break;
		case PANE2:
			active = (curr_view == &app.pane2_view);
			break;
		case PANEINACTIVE:
			active = FALSE;
			break;
		default:
			active = TRUE;
			break;
	}
	return active;
}
/**
@brief disable refreshing of a single pane filelist
This is mainly intended to block changes of filesystem CWD when a process is
working in a displayed directory
@param pane enumerator for pane to be blocked
@return
*/
void e2_filelist_disable_one_refresh (E2_ListChoice pane)
{
	if (_e2_filelist_check_active_pane (pane))
	{
		refresh_active_refcount++;
		if (refresh_active_refcount == 1)
			curr_view->blocked = TRUE;
	}
	else
	{
		refresh_inactive_refcount++;
		if (refresh_inactive_refcount == 1)
			other_view->blocked = TRUE;
	}
}
/**
@brief enable refreshing of a single pane filelist

@param pane enumerator for pane to be unblocked

@return
*/
void e2_filelist_enable_one_refresh (E2_ListChoice pane)
{
	if (_e2_filelist_check_active_pane (pane))
	{
		refresh_active_refcount--;
		if (refresh_active_refcount == 0 && refresh_refcount == 0)
			curr_view->blocked = FALSE;
		if (refresh_active_refcount < 0)
			refresh_active_refcount = 0;
	}
	else
	{
		refresh_inactive_refcount--;
		if (refresh_inactive_refcount == 0 && refresh_refcount == 0)
			other_view->blocked = FALSE;
		if (refresh_inactive_refcount < 0)
			refresh_inactive_refcount = 0;
	}
}
/**
@brief disable -refresh action

@param from the button, menu item etc which was activated
@param art action runtime data

@return TRUE always
*/
gboolean e2_filelist_disable_refresh_action (gpointer from, E2_ActionRuntime *art)
{
	e2_filelist_disable_refresh ();
	return TRUE;
}
/* These functions must encapsulate any instructions that access the FileInfo
structs produced when a selection is interrogated, because a refresh in the
middle of such an operation would free FileInfo structs as a side-effect.
*/
/**
@brief  disable refreshing of config data and pane filelists

Increase ref count, if == 1, stop the timer process which calls the refresh fns
Why is config data check bundled here ?

@return
*/
void e2_filelist_disable_refresh (void)
{
	refresh_refcount++;
#ifdef E2_REFRESH_DEBUG
	printd (DEBUG, "disable refresh, refresh ref now = %d", refresh_refcount);
#endif
	if (refresh_refcount == 1)
	{
		if (app.timers[REFRESHBEGIN_T] > 0)
		{
			//we're waiting for a pending refresh
			g_source_remove (app.timers[REFRESHBEGIN_T]);
			app.timers[REFRESHBEGIN_T] = 0;
		}
		if (e2_option_bool_get ("auto-refresh"))
		{
			//don't cancel check-dirty timer, to reduce risk that FAM
			//dirty-reports queue may over-fill
			curr_view->blocked = TRUE;
			other_view->blocked = TRUE;
		}
		if (e2_option_bool_get ("auto-refresh-config"))
		{
			//CHECKME just block this too?
			//(can't reasonably be many changes to the config file)
			if (app.timers[CONFIG_T] > 0)
			{
				g_source_remove (app.timers[CONFIG_T]);
				app.timers[CONFIG_T] = 0;
			}
		}

		e2_window_disable_status_update ();
	}
}
/**
@brief  cause polling of config data and pane filelists at specified intervals

Polling (re)starts after the interval
reference count is back to 0

@param interval  milliseconds between polls

@return
*/
/*void e2_filelist_set_refresh (guint interval)
{
	e2_filelist_disable_refresh ();
	e2_filelist_enable_refresh_custom (interval);
}

void e2_filelist_set_refresh_default (void)
{
	e2_filelist_disable_refresh ();
	e2_filelist_enable_refresh_custom (E2_FILESCHECK_INTERVAL);
} */
/**
@brief enable -refresh action

@param from the button, menu item etc which was activated
@param art action runtime data

@return TRUE always
*/
gboolean e2_filelist_enable_refresh_action (gpointer from, E2_ActionRuntime *art)
{
	e2_filelist_enable_refresh_custom (E2_FILESCHECK_INTERVAL);
	return TRUE;
}
/**
@brief reset polling of config data and pane filelists

@return
*/
void e2_filelist_reset_refresh (void)
{
	if (refresh_refcount > 0)
	{
		refresh_refcount = 1;
		e2_filelist_enable_refresh_custom (E2_FILESCHECK_INTERVAL);
	}
}
/**
@brief re-enable polling of config data and pane filelists

Polling (re)starts after the default 1-second interval, if the
reference count is back to 0

@return
*/
void e2_filelist_enable_refresh (void)
{
	e2_filelist_enable_refresh_custom (E2_FILESCHECK_INTERVAL);
}
/**
@brief  re-enable polling of config data and pane filelists, with a specified interval

Decrease ref count, if == 0, restart the timers which call the refresh fns
 Config timer is bundled here because the fn refreshes the panes

@param interval milliseconds between polls

@return
*/
void e2_filelist_enable_refresh_custom (guint interval)
{
	refresh_refcount--;
#ifdef E2_REFRESH_DEBUG
	printd (DEBUG, "enable refresh, refresh ref now = %d", refresh_refcount);
#endif
	if (refresh_refcount < 0)
	{
		printd (WARN, "The auto-refresh refresh count is < 0");
		gdk_threads_enter ();
		e2_window_output_show (NULL, NULL);
		e2_output_print_error (
		_("Something is wrong with the auto-refresh mechanism"), FALSE);
		gdk_threads_leave ();
	}

	if (refresh_refcount == 0)
	{
		if (e2_option_bool_get ("auto-refresh"))
		{
			curr_view->blocked = FALSE;
			other_view->blocked = FALSE;
		}
//		printd (DEBUG, "refresh enabled, starting auto-refresh timer");
		//this timer is used, even with fam backends
		if (e2_option_bool_get ("auto-refresh-config"))
			app.timers[CONFIG_T] = g_timeout_add (E2_CONFIGCHECK_INTERVAL,
				(GSourceFunc) e2_option_check_config_files, NULL);

		e2_window_enable_status_update (-1);
	}
}
/**
@brief timer destroy function after cancelling filelists checking
@param data UNUSED data specified when the timer was established
@return
*/
static void _e2_filelist_timer_shutdown (gpointer data)
{
	app.timers[REFRESHBEGIN_T] = 0;
}
/**
@brief timer callback function which refreshes filelists as appropriate, as soon as any block is gone
@param data pointerised value of current timer delay
@return FALSE when there's nothing left to refresh
*/
static gboolean _e2_filelist_check_requests (gpointer data)
{
	//parallel filelist updates not supported
	if (!curr_view->completed || !other_view->completed)
	{
		printd (DEBUG, "list(s) refresh deferrred");
		//if appropriate, lengthen the delay between attempts
		if (data < GINT_TO_POINTER (200))
		{
			g_source_remove (app.timers[REFRESHBEGIN_T]);
			app.timers[REFRESHBEGIN_T] = g_timeout_add_full (G_PRIORITY_HIGH, 200,
				(GSourceFunc) _e2_filelist_check_requests, GINT_TO_POINTER (200),
					(GDestroyNotify) _e2_filelist_timer_shutdown);
		}
		return TRUE;
	}
	//blocked flags set when refresh is disabled
	if (curr_view->refresh_requested && !curr_view->blocked)
	{
#ifdef E2_VFSTMP
	//FIXME dir when not mounted local
#else
		printd (DEBUG, "accepting request to refresh %s", curr_view->dir);
#endif
		//refresh function expects BGL closed
		gdk_threads_enter ();
		e2_fileview_refresh_list (curr_view);
		gdk_threads_leave ();
		curr_view->refresh_requested = FALSE;
	}
	if (other_view->refresh_requested && !other_view->blocked)
	{
#ifdef E2_VFSTMP
	//FIXME dir when not mounted local
#else
		printd (DEBUG, "accepting request to refresh %s", other_view->dir);
#endif
		gdk_threads_enter ();
		e2_fileview_refresh_list (other_view);
		gdk_threads_leave ();
		other_view->refresh_requested = FALSE;
	}
	return FALSE;
}
/**
@brief log request, and possibly initiate, filelist(s) refresh if it/they currently show @a dir
If a refresh is initiated, either or both flagged lists will be refreshed
as soon as any in-progress re-list (either or both panes) is completed
@param dir the dir to be refreshed (utf8 string)
@param immediate TRUE to intiate a refresh

@return TRUE if a refresh was initiated
*/
gboolean e2_filelist_request_refresh (gchar *dir, gboolean immediate)
{
	gboolean matched = FALSE;
#ifdef E2_VFSTMP
	//CHECKME v-dir ok
#endif
	if (g_str_equal (curr_view->dir, dir))
	{
		matched = TRUE;
		curr_view->refresh_requested = TRUE;
	}
#ifdef E2_VFSTMP
	//CHECKME v-dir ok
#endif
	if (g_str_equal (other_view->dir, dir))
	{
		matched = TRUE;
		other_view->refresh_requested = TRUE;
	}

	if (matched && immediate)
		//clear any fs report about dirty list, then do the refresh
		e2_filelist_check_dirty (NULL);

	return matched;
}
/**
@brief timer callback function which activates @a focus_wid when filelist refreshing is done
@param focus_wid widget to focus when it's ok
@return TRUE while still waiting for completion
*/
/*static gboolean _e2_filelist_wait_to_focus (GtkWidget *focus_wid)
{
	if (curr_view->refresh_requested || other_view->refresh_requested
		|| !curr_view->completed || !other_view->completed)
		return TRUE;
	gdk_threads_enter ();
	gtk_widget_grab_focus (focus_wid);
	gdk_threads_leave ();
	return FALSE;
} */
/**
@brief focus @a focus_wid when filelist refreshing is done
@param focus_wid widget to focus
@return
*/
/*void e2_filelist_request_focus (GtkWidget *focus_wid)
{
	app.timers[?] = g_timeout_add (330, (GSourceFunc) _e2_filelist_wait_to_focus,
		focus_wid);
}*/
/**
@brief refresh filepane contents (and with FAM, config too) if the corresponding changed is detected

Among other uses, this is the timeout function for filelists auto refresh
When FAM/gamin in use, it also checks for and flags (not processes)
config file updates, as that data comes from the same data stream.
This expects gtk's BGL to be OFF

@param userdata data specified when the timer was initialised (NULL), or when called directly, non-NULL

@return TRUE so the timer keeps working
*/
gboolean e2_filelist_check_dirty (gpointer userdata)
{
#ifdef E2_FAM
	static gboolean cfgdeferred = FALSE;
#endif

	gboolean p1dirty, p2dirty;
#ifdef E2_FAM
	gboolean localcfgdirty;	//use a local copy so that the extern value doesn't get cleared before it's noticed
	extern gboolean cfgdirty;
#endif

#ifdef E2_FAM
	if (userdata != NULL
		 && app.monitor_type != E2_MONITOR_DEFAULT)
			g_usleep (E2_FAMWAIT);	//wait for things to be noticed
#endif
	//this reports 'gone' as 'dirty', and then the list-refesh
	//function handles choosing a replacement dir
	e2_fs_FAM_check_dirty (&p1dirty, &p2dirty
#ifdef E2_FAM
	, &localcfgdirty
#endif
	);

	if (p1dirty)
		app.pane1_view.refresh_requested = TRUE;
	if (p2dirty)
		app.pane2_view.refresh_requested = TRUE;

	if (app.pane1_view.refresh_requested || app.pane2_view.refresh_requested)
	{
		if (app.timers[REFRESHBEGIN_T] > 0)
		{
			//clear any incomplete refresh request
			g_source_remove (app.timers[REFRESHBEGIN_T]);
		}
//		if (_e2_filelist_check_requests (NULL)) can only do this if BGL is always open here
//		{	//immediate refresh was blocked
//			printd (DEBUG, "FILELIST(S) DIRTY, initiating list refresh timer");
			//the delay between attempts may be lengthened later, if appropriate
			app.timers[REFRESHBEGIN_T] = g_timeout_add_full (G_PRIORITY_HIGH, 2,
				(GSourceFunc) _e2_filelist_check_requests, GINT_TO_POINTER (2),
				(GDestroyNotify) _e2_filelist_timer_shutdown);
//			printd (DEBUG, "timer ID = %d", app.timers[REFRESHBEGIN_T]);
//		}
	}

#ifdef E2_FAM
	if (localcfgdirty || cfgdeferred)
	{
		if (!app.rebuild_enabled)
		{
			cfgdeferred = TRUE;
			printd (DEBUG, "config refresh deferred");
		}
		else
		{
			cfgdeferred = FALSE;
			if (localcfgdirty)
			{
				cfgdirty = TRUE;	//set, but don't clear, the main flag
				printd (DEBUG, "config refresh flag set");
			}
		}
	}
#endif	//def E2_FAM

	return TRUE;
}
/**
@brief idle function to clear old liststores

@param user_data UNUSED data specified when the idle was setup

@return FALSE, to stop the callbacks
*/
gboolean e2_filelist_clear_old_stores (gpointer user_data)
{
	GSList *tmp;
#ifdef DEBUG_MESSAGES
	gint debug = g_slist_length (app.used_stores);
#endif
	for (tmp = app.used_stores; tmp != NULL; tmp = tmp->next)
	{
		GtkListStore *store = tmp->data;
		GtkTreeModel *mdl = GTK_TREE_MODEL (store);
		GtkTreeIter iter;
		if (gtk_tree_model_get_iter_first (mdl, &iter))
		{	//it's not empty already
			//clear file info data structs referred to in the store
			//CHECKME need to clear anything else?
			FileInfo *info;
			do
			{
				gtk_tree_model_get (mdl, &iter, FINFO, &info, -1);
#ifdef USE_GLIB2_10
				g_slice_free1 (sizeof(FileInfo), info);
				//DEALLOCATE (FileInfo, info);
#else
				DEALLOCATE (FileInfo, info);
#endif
			} while (gtk_tree_model_iter_next (mdl, &iter));
		}
		//CHECKME clear filtermodel ?
		g_object_unref (G_OBJECT (store));
	}
	g_slist_free (app.used_stores);
	app.used_stores = NULL;
	printd (DEBUG, "%d old liststore(s) cleared", debug);
	return FALSE;
}
/**
@brief create and populate filelist-compatible list store with rows for each item in @a entries

@param entries list of FileInfo's to be processed
@param view pointer to data struct for the view to which the filled store will apply

@return pointer to the liststore, or NULL if a problem occurs
*/
GtkListStore *e2_filelist_fill_store (GList *entries, ViewInfo *view)
{
//	printd (DEBUG, "start store fill");
	GtkListStore *store = e2_filelist_make_store ();
	if (store == NULL)
		return NULL;

	GList *tmp;
	FileInfo *infoptr;
	GtkTreeIter iter;
	struct tm *tm_ptr;
	struct passwd *pwd_buf;
	struct group *grp_buf;
	gchar size_buf[20];	//enough for 999 Tb
	gchar modified_buf[25];
	gchar accessed_buf[25];
	gchar changed_buf[25];
	gchar perm_buf[11];
	gchar uid_buf[20];
	gchar gid_buf[20];
	gchar *buf[NAMEKEY+1];	//pointers to strings to be inserted into each row
	gchar *freeme;
	gchar *__utf__;	//for e2_utf8_from_locale_fast() macro

	//get format parameters

	//format for dates display
	gchar *strf_string;
	switch	(e2_option_int_get ("date-string"))
	{
		case 1:
			strf_string = "%d/%m/%y %H:%M";	//standard
			break;
		case 2:
			strf_string = "%m/%d/%y %H:%M";	//american
			break;
		case 3:
			strf_string = "%Y-%m-%dT%H:%M";	//ISO8601
			break;
		case 4:
/*			{
				strf_string = nl_langinfo (D_T_FMT);
				if (strf_string != NULL && *strf_string != '\0')
					break;
			} */
			strf_string = "%x %X";	//localised
			break;
		default:
			strf_string = "%b %d %H:%M";
			break;
	}
	//get format for size display
	gchar *comma;
	switch	(e2_option_int_get ("size-string"))
	{
		case 1:
		{
			comma = nl_langinfo (THOUSEP);
			if (comma == NULL || *comma == '\0')
				comma = ",";
			break;
		}
		default:
			comma = NULL;	//signal to use the condensed version
			break;
	}

	gboolean caseignore = ! e2_option_bool_get ("namesort-case-sensitive");
	GtkStyle *style = gtk_rc_get_style (curr_view->treeview);
	GdkColor *default_color = &style->text[GTK_STATE_NORMAL];

	GdkColor *link_color = e2_option_color_get ("color-ft-link");
	GdkColor *dir_color = e2_option_color_get ("color-ft-dir");
	GdkColor *dev_color = e2_option_color_get ("color-ft-dev");
	GdkColor *sock_color = e2_option_color_get ("color-ft-socket");
	GdkColor *exec_color = e2_option_color_get ("color-ft-exec");

	//iterate through the listed FileInfo's
	for (tmp = entries; tmp != NULL; tmp = tmp->next)
	{
		infoptr = (FileInfo *) tmp->data;

		//convert to utf if appropriate
		buf[FILENAME] = F_DISPLAYNAME_FROM_LOCALE (infoptr->filename);
		//directories (and links to dirs) get a trailing separator
		//FIXME speed this by using a more efficient process to check for dir
		if (e2_fs_is_dir (infoptr, view))
		{
			freeme = buf[FILENAME];
			buf[FILENAME] = e2_utils_strcat (freeme, "/");
			if (freeme != infoptr->filename)
				g_free (freeme);
		}
		else if (buf[FILENAME] == infoptr->filename)	//no actual conversion, above
			buf[FILENAME] = g_strdup (buf[FILENAME]);

		if (comma == NULL)
		{	//use condensed size format
			if (infoptr->statbuf.st_size < 1024) //less than 1k
			{
			  g_snprintf(size_buf, sizeof(size_buf), "%llu",
					infoptr->statbuf.st_size);
			}
			else if (infoptr->statbuf.st_size < 1048576) //less than a meg
			{
			  g_snprintf(size_buf, sizeof(size_buf), "%.1f%s",
				(gfloat) infoptr->statbuf.st_size / 1024.0, _("k"));
			}
			else  //a meg or more  if (infoptr->statbuf.st_size < 1073741824)
			{
			  g_snprintf(size_buf, sizeof(size_buf), "%.1f%s",
				(gfloat) infoptr->statbuf.st_size / 1048576.0, _("M"));
			}
/*			else //a gig or more
			{
			  g_snprintf(size_buf, sizeof(size_buf), "%.1f%s",
				(gfloat) infoptr->statbuf.st_size / 1073741824.0, _("G"));
			} */
		}
		else
		{	//use actual size, with commas
			g_snprintf(size_buf, sizeof(size_buf), "%llu",
				infoptr->statbuf.st_size);

			guint len = strlen (size_buf);
			guint ths = len-1;  //0-based index
			guint i;
			while (ths > 2 && len < sizeof(size_buf))
			{
				for (i = len-1; i > ths-3; i--)
					size_buf[i+1] = size_buf[i];

				size_buf[i+1] = *comma;
				size_buf[++len] = '\0';
				ths = i;
			}
		}
		buf[SIZE] = g_strdup (size_buf);	//content is ascii, no need to convert to utf

		e2_fs_get_perm_string (perm_buf, sizeof(perm_buf), infoptr->statbuf.st_mode);
//		buf[PERM] = e2_utf8_from_locale_fast (perm_buf);
		buf[PERM] = g_strdup (perm_buf); //content is ascii, no need to convert to utf

		if ((pwd_buf = getpwuid (infoptr->statbuf.st_uid)) == NULL)
		{
		  g_snprintf (uid_buf, sizeof(uid_buf), "%d",
		     (guint) infoptr->statbuf.st_uid);
		  buf[OWNER] = g_strdup (uid_buf); //content is ascii number, no need to convert to utf
		}
		else
		{
		  buf[OWNER] = e2_utf8_from_locale_fast (pwd_buf->pw_name);
		}

		if ((grp_buf = getgrgid (infoptr->statbuf.st_gid)) == NULL)
		{
		  g_snprintf(gid_buf, sizeof(gid_buf), "%d",
		       (guint) infoptr->statbuf.st_gid);
		  buf[GROUP] = g_strdup (gid_buf); //content is ascii number, no need to convert to utf
		}
		else
		{
		  buf[GROUP] = e2_utf8_from_locale_fast (grp_buf->gr_name);
		}
		tm_ptr = localtime (&(infoptr->statbuf.st_mtime));
		strftime (modified_buf, sizeof(modified_buf), strf_string, tm_ptr);
		buf[MODIFIED] = e2_utf8_from_locale_fast (modified_buf);

		tm_ptr = localtime (&(infoptr->statbuf.st_atime));
		strftime (accessed_buf, sizeof(accessed_buf), strf_string, tm_ptr);
		buf[ACCESSED] = e2_utf8_from_locale_fast (accessed_buf);

		tm_ptr = localtime(&(infoptr->statbuf.st_ctime));
		strftime (changed_buf, sizeof(changed_buf), strf_string, tm_ptr);
		buf[CHANGED] = e2_utf8_from_locale_fast (changed_buf);

		if (caseignore)
		{
			freeme = g_utf8_casefold (buf[FILENAME], -1);
#ifdef USE_GTK2_8
			buf[NAMEKEY] = g_utf8_collate_key_for_filename (freeme, -1);
#else
			buf[NAMEKEY] = g_utf8_collate_key (freeme, -1);
#endif
			g_free (freeme);
		}
		else
#ifdef USE_GTK2_8
			buf[NAMEKEY] = g_utf8_collate_key_for_filename (buf[FILENAME], -1);
#else
			buf[NAMEKEY] = g_utf8_collate_key (buf[FILENAME], -1);
#endif
		GdkColor *foreground;
		switch (infoptr->statbuf.st_mode & S_IFMT)
		{
		  case S_IFLNK:
		    foreground = link_color;
		    break;
		  case S_IFDIR:
		    foreground = dir_color;
		    break;
		  case S_IFCHR:
		  case S_IFBLK:
		    foreground = dev_color;
		    break;
		  case S_IFSOCK:
		    foreground = sock_color;
		    break;
		  case S_IFREG:
			  //show as executable if _anyone_ has that permission
		    if ((S_IXUSR & infoptr->statbuf.st_mode)
		  		|| (S_IXGRP & infoptr->statbuf.st_mode)
		  		|| (S_IXOTH & infoptr->statbuf.st_mode))
			  foreground = exec_color;
			else
			{
#ifdef E2_RAINBOW
				//find extension of current item
				//localised text, but assumes . is ascii
				//hashed extension-strings used for comparison have been localised
				foreground = NULL;	//in case there's no '.' at all
				gchar *ext = infoptr->filename + 1;	//+1 in case item is hidden
				while ((ext = strchr (ext,'.')) != NULL)
				{
					ext++;	//pass the '.'
					//use its color, if any
					foreground = g_hash_table_lookup (app.colors, ext);
					if (foreground != NULL)
						break;
				}
				if (foreground == NULL)
					foreground = default_color;
#else
				foreground = default_color;
#endif
			}
		    break;
		  default:
			foreground = default_color;
		    break;
		}
		//gtk >= 2.10 can handle &iter = NULL
		gtk_list_store_insert_with_values (store, &iter, -1,
			FILENAME, buf[FILENAME],
			SIZE, buf[SIZE],
			PERM, buf[PERM],
			OWNER, buf[OWNER],
			GROUP, buf[GROUP],
			MODIFIED, buf[MODIFIED],
			ACCESSED, buf[ACCESSED],
			CHANGED, buf[CHANGED],
			NAMEKEY, buf[NAMEKEY],
			FINFO, infoptr,
			FORECOLOR, foreground,
//			VISIBLE, TRUE,
			-1);

		//cleanup
		gint i;
		for (i = 0; i <= NAMEKEY; i++)
			g_free (buf[i]);

/*
//		gettimeofday (&strt, NULL); //FOR DEBUGGING, check time this load takes
//		gettimeofday (&nd, NULL);
		if (nd.tv_usec < strt.tv_usec)
		{
		    int nsec = (strt.tv_usec - nd.tv_usec) / 1000000 + 1;
    		strt.tv_usec -= 1000000 * nsec;
    		strt.tv_sec += nsec;
		}
		if (nd.tv_usec - strt.tv_usec > 1000000)
		{
			int nsec = (strt.tv_usec - nd.tv_usec) / 1000000;
			strt.tv_usec += 1000000 * nsec;
			strt.tv_sec -= nsec;
		}
		result.tv_sec = result.tv_sec + nd.tv_sec - strt.tv_sec;
		result.tv_usec = result.tv_usec + nd.tv_usec - strt.tv_usec;
		if (result.tv_usec > 1000000)
		{
			int nsec = result.tv_usec / 1000000 + 1;
    		result.tv_usec -= 1000000 * nsec;
    		result.tv_sec += nsec;
		}

//	printd (DEBUG, "populating rows took %f seconds", result.tv_sec + result.tv_usec / 1000000.0 );
*/
	}
	return store;
}
/**
@brief create list store structure

Creates basic structure of list store, for panes filelists
No rows or data are added.

@return the new liststore
*/
GtkListStore *e2_filelist_make_store (void)
{
	GtkListStore *store = gtk_list_store_new (MODEL_COLUMNS,
		G_TYPE_STRING,  //FILENAME
		G_TYPE_STRING,  //SIZE
		G_TYPE_STRING,  //PERM
		G_TYPE_STRING,  //OWNER
		G_TYPE_STRING,  //GROUP
		G_TYPE_STRING,  //MODIFIED
		G_TYPE_STRING,  //ACCESSED
		G_TYPE_STRING,  //CHANGED
		// the rest are not displayed
		G_TYPE_STRING,  //NAMEKEY for i18n name sorts
		G_TYPE_POINTER,  //FINFO pr to FileInfo for the item
		GDK_TYPE_COLOR,  //FORECOLOR line colour
		GDK_TYPE_COLOR  //BACKCOLOR line colour
	//	G_TYPE_BOOLEAN	//VISIBLE whether filtered or not
		);
	return store;
}
