/* $Id: e2_select_image_dialog.c 544 2007-07-20 12:48:20Z tpgww $

Copyright (C) 2003-2007 tooar <tooar@gmx.net>

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
@file src/dialogs/e2_select_image_dialog.c
@brief Select image dialog

This files contains all functions needed to create a select-image
dialog, which is opened when an icon is clicked on a tree-opotion
page in the main configuration dialog
*/

#include "emelfm2.h"
#include <string.h>
#include "e2_dialog.h"
#include "e2_select_image_dialog.h"
#include "e2_config_dialog.h"

GtkWidget *rem_btn;

/**
@brief get the appropriate directory for icons

@param rt pointer to data for sid-dialog

@return newly-allocated string with the path to use, localised with trailing /
*/
static gchar *_e2_sidlg_get_icon_dir (E2_SID_Runtime *rt)
{
	gboolean useiconpath;
	gchar *iconpath, *freeme;
	gpointer dtype = g_object_get_data (G_OBJECT (rt->parent), "dialog-form");
	if (GPOINTER_TO_INT (dtype) == E2_CFGDLG_SINGLE)
	{	//doing a single-page dialog
		iconpath = e2_utils_get_icons_path (TRUE);
	}
	else
	{	//doing a full config dialog
		//so a relevant option may have been changed in this session
		E2_OptionSet *set = e2_option_get ("use-icon-dir");

		if (set->widget != NULL)	//this option is part of the dialog
			useiconpath = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (set->widget));
		else
			useiconpath = e2_option_bool_get_direct (set);
		if (useiconpath)
		{
			set = e2_option_get ("icon-dir");
			if (set->widget != NULL)
			{
				freeme = gtk_editable_get_chars (GTK_EDITABLE (set->widget), 0, -1);
				iconpath = D_FILENAME_TO_LOCALE (freeme);
				g_free (freeme);
				if (!g_str_has_suffix (iconpath, G_DIR_SEPARATOR_S))
				{
					freeme = iconpath;
					iconpath = g_strconcat (freeme, G_DIR_SEPARATOR_S, NULL);
					g_free (freeme);
				}
			}
			else
				iconpath = g_strdup (ICON_DIR G_DIR_SEPARATOR_S);	//localised
		}
		else
			iconpath = g_strdup (ICON_DIR G_DIR_SEPARATOR_S);
	}

	return iconpath;
}

  /*********************/
 /***** callbacks *****/
/*********************/

/**
@brief setup object properties so it's possible to determine the currently selected icon or file

@param dialog the sid-dialog where the response was triggered
@param response the response for the clicked button
@param rt pointer to data for sid-dialog

@return
*/
static void _e2_sidlg_response_cb (GtkDialog *dialog, gint response, E2_SID_Runtime *rt)
{
	if (rt->page == 0)
	{	//working with a custom icon
		gchar *iconpath = _e2_sidlg_get_icon_dir (rt);
		if (response == E2_RESPONSE_REMOVE)
		{
			g_object_set_data_full (G_OBJECT (dialog), "image", g_strdup (""),
				(GDestroyNotify) g_free);
			//set to non-existent file (file can't be "")
			gchar *chooserpath = g_strconcat (iconpath, " ", NULL);
			gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (rt->file_chooser),
				chooserpath);
			g_free (chooserpath);
		}
		else
		{
			gchar *savename;
			gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (rt->file_chooser));
			if (filename == NULL)
				savename = "";
			else if (g_str_has_prefix (filename, iconpath))
			{
				savename = filename + strlen (iconpath);
				while (*savename == G_DIR_SEPARATOR)	//ok to use ascii check
					savename++;
			}
			else
				savename = filename;
			g_object_set_data_full (G_OBJECT (dialog), "image", g_strdup (savename),
				(GDestroyNotify) g_free);
			if (filename != NULL)
				g_free (filename);
		}
		g_free (iconpath);
	}
	else
	{ //working with a stock icon
		GtkTreeIter iter;
		GtkTreeModel *model;
		GtkTreeSelection *selection = gtk_tree_view_get_selection (rt->treeview);
		gboolean sel = gtk_tree_selection_get_selected (selection, &model, &iter);

		if (response == E2_RESPONSE_REMOVE)
		{
			g_object_set_data_full (G_OBJECT (dialog), "image", g_strdup (""),
				(GDestroyNotify) g_free);
			if (sel)
				gtk_tree_selection_unselect_iter (selection, &iter);
		}
		else
		{
			if (sel)
			{
				gchar *icon;
				gtk_tree_model_get (model, &iter, 0, &icon, -1);
				g_object_set_data_full (G_OBJECT (dialog), "image", g_strdup (icon),
					(GDestroyNotify) g_free);
				g_free (icon);
			}
			else
				g_object_set_data_full (G_OBJECT (dialog), "image", g_strdup (""),
					(GDestroyNotify) g_free);
		}
	}
	gtk_widget_set_sensitive (rem_btn, (response != E2_RESPONSE_REMOVE));
}
/**
@brief issue ok response for @a dialog
This is a callback for row-activated signal in stock-icons treeview

@param treeview
@param path
@param tree_column
@param dialog the parent dialog widget

@return
*/
static void _e2_sidlg_activated_cb (GtkTreeView *treeview, GtkTreePath *path,
	GtkTreeViewColumn *tree_column, GtkWidget *dialog)
{
	gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
}
/**
@brief issue ok response for @a dialog
This is a callback for file-activated signal in file-chooser
@param treeview
@param dialog the parent dialog widget

@return
*/
static void _e2_sidlg_file_activated_cb (GtkTreeView *treeview, GtkWidget *dialog)
{
	printd (DEBUG, "file_activated_cb");
	gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
}
/**
@brief update stored notebook page number

@param notebook UNUSED
@param npage UNUSED
@param page the new page no.
@param rt pointer to data for sid-dialog

@return
*/
static void _e2_sidlg_switch_page_cb (GtkNotebook *notebook,
	GtkNotebookPage *npage, guint page, E2_SID_Runtime *rt)
{
	rt->page = page;
}
/**
@brief cleanup sid-dialog data

@param dialog
@param rt pointer to data for sid-dialog

@return
*/
static void _e2_sidlg_destroy_cb (GtkWidget *dialog, E2_SID_Runtime *rt)
{
	g_object_set_data (G_OBJECT (rt->parent), rt->name, NULL);
	g_free (rt->name);
	g_free (rt->icon);
	DEALLOCATE (E2_SID_Runtime, rt);
}
/**
@brief destroy sid-dialog

@param rt pointer to data for sid-dialog

@return
*/
static void _e2_sidlg_destroy_cb2 (E2_SID_Runtime *rt)
{
	gtk_widget_destroy (rt->dialog);
}

/*
//filter backup files from a GtkFileChooser listing
static gboolean image_files_filter (const GtkFileFilterInfo *filter_info,
	gpointer data)
{
	gsize len = filter_info->display_name ? strlen (filter_info->display_name) : 0;
	if (len > 0)
	{
		if (g_str_has_suffix (filter_info->display_name, "~"))
		return FALSE;
	} else
		return TRUE;
}
*/

/**
@brief update the image preview shown in the file-chooser

@param file_chooser the file-chooser showing the image
@param preview

@return
*/
static void _e2_sidlg_update_preview_cb (GtkFileChooser *file_chooser, GtkWidget *preview)
{
	gchar *filename = gtk_file_chooser_get_preview_filename (file_chooser);
	GdkPixbuf *pixbuf = NULL;
	if (filename != NULL)
	{
		pixbuf = gdk_pixbuf_new_from_file_at_size (filename, 64, 64, NULL);
		g_free (filename);
	}
	if (pixbuf != NULL)
	{
		gtk_image_set_from_pixbuf (GTK_IMAGE (preview), pixbuf);
		gdk_pixbuf_unref (pixbuf);
		gtk_file_chooser_set_preview_widget_active (file_chooser, TRUE);
	}
	else
		gtk_file_chooser_set_preview_widget_active (file_chooser, FALSE);
}

  /*****************/
 /***** utils *****/
/*****************/

/**
@brief create liststore and model with all gtk stock icons

@return the treemodel for the store
*/
static GtkTreeModel *_e2_sidlg_create_stock_model (void)
{
	GtkListStore *store = gtk_list_store_new (1, G_TYPE_STRING);

	GSList *ids = gtk_stock_list_ids ();
	ids = g_slist_sort (ids, (GCompareFunc) strcmp);
	GSList *member;
	GtkTreeIter iter;
	for (member = ids; member != NULL; member = g_slist_next (member))
		gtk_list_store_insert_with_values (store, &iter, -1, 0, member->data, -1);

	g_slist_foreach (ids, (GFunc) g_free, NULL);
	g_slist_free (ids);

	return GTK_TREE_MODEL (store);
}
/**
@brief set text for treemodel cell @a cell

@param tree_column UNUSED pointer to column containing @a cell
@param cell pointer to cell whose data is to be set
@param model tree model containing data for @a cell
@param iter pointer to tree iter used for interrogating @a model
@param data UNUSED

@return
*/
static void _e2_sidlg_cell_data_func (GtkTreeViewColumn *tree_column,
	GtkCellRenderer *cell, GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gchar *value;
	gtk_tree_model_get (model, iter, 0, &value, -1);
	if (g_str_has_prefix (value, "gtk-"))
		value += 4;
	g_object_set (GTK_CELL_RENDERER (cell), "text", value, NULL);
}
/**
@brief create scrolled window showing stock icons

@param rt pointer to data for sid-dialog

@return
*/
static GtkWidget *_e2_sidlg_create_stock_icon_browser (E2_SID_Runtime *rt)
{
	rt->model = _e2_sidlg_create_stock_model ();
	rt->treeview = GTK_TREE_VIEW (gtk_tree_view_new_with_model (rt->model));
	gtk_tree_view_set_rules_hint (rt->treeview, TRUE);

	g_object_unref (rt->model);

	g_signal_connect (rt->treeview, "row-activated",
		G_CALLBACK (_e2_sidlg_activated_cb), rt->dialog);

	GtkWidget *sw = e2_widget_get_sw_plain (GTK_POLICY_NEVER,
		GTK_POLICY_AUTOMATIC);

	gtk_container_add (GTK_CONTAINER (sw), GTK_WIDGET (rt->treeview));
	GtkTreeViewColumn *column = gtk_tree_view_column_new ();
	g_object_set (column, "alignment", 0.5, NULL);
	gtk_tree_view_column_set_title (column, _("icons"));

	GtkCellRenderer *cell_renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, cell_renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, cell_renderer, "stock_id", 0,
		NULL);
	g_object_set (cell_renderer, "xpad", E2_PADDING_LARGE, NULL);
	cell_renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, cell_renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column, cell_renderer,
		_e2_sidlg_cell_data_func, NULL, NULL);
	g_object_set (cell_renderer, "xpad", E2_PADDING_LARGE, NULL);
	gtk_tree_view_append_column (rt->treeview, column);

	return sw;
}
/**
@brief create file-chooser widget for the dialog

@param dialog the parent dialog widget

@return the file-chooser widget
*/
static GtkWidget *_e2_sidlg_create_file_chooser (GtkWidget *dialog)
{
//	e2_fs_chdir_local (G_DIR_SEPARATOR_S E2_ERR_NULL());
	GtkWidget *file_chooser = gtk_file_chooser_widget_new
		(GTK_FILE_CHOOSER_ACTION_OPEN);
	GtkFileChooser *chooser = GTK_FILE_CHOOSER (file_chooser);
	gtk_file_chooser_set_local_only (chooser, TRUE);
	gtk_file_chooser_add_shortcut_folder (chooser, ICON_DIR, NULL);	//localised
//	gtk_file_chooser_add_shortcut_folder (chooser,
//#ifdef E2_VFSTMP
			//FIXME dir when not mounted local
//#else
//		app.pane1.path, NULL);
//#endif
//	gtk_file_chooser_add_shortcut_folder (chooser,
//#ifdef E2_VFSTMP
			//FIXME dir when not mounted local
//#else
//		app.pane2.path, NULL);
//#endif

	//add filters
	GtkFileFilter *filter = gtk_file_filter_new ();
	gtk_file_filter_set_name (GTK_FILE_FILTER (filter), _("images"));
/*	gtk_file_filter_add_mime_type (filter, "image/png");
	gtk_file_filter_add_mime_type (filter, "image/x-xpixmap");
	gtk_file_filter_add_mime_type (filter, "image/x-xpm");
	gtk_file_filter_add_mime_type (filter, "image/svg+xml");
	//FIXME: check the mime types
	gtk_file_filter_add_mime_type (filter, "image/jpg");
	gtk_file_filter_add_mime_type (filter, "image/bmp");
	gtk_file_filter_add_mime_type (filter, "image/gif");
*/
	gtk_file_filter_add_mime_type (filter, "image/*");
	gtk_file_chooser_add_filter (chooser, filter);
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name (GTK_FILE_FILTER (filter), _("all"));
	gtk_file_filter_add_pattern (filter, "*");
	gtk_file_chooser_add_filter (chooser, filter);

	GtkWidget *preview = gtk_image_new ();
	gtk_file_chooser_set_preview_widget (chooser, preview);
	g_signal_connect (G_OBJECT (file_chooser), "update-preview",
		G_CALLBACK (_e2_sidlg_update_preview_cb), preview);

	gtk_file_chooser_set_action (chooser, GTK_FILE_CHOOSER_ACTION_OPEN);

//FIXME: read missing location entry to the dialog
//	GtkWidget *entry = gtk_entry_new ();
//	gtk_file_chooser_set_extra_widget (chooser, entry);

	g_signal_connect (G_OBJECT (file_chooser), "file-activated",
		G_CALLBACK (_e2_sidlg_file_activated_cb), dialog);

	return file_chooser;
}
/**
@brief initialize notebook page and line

@param rt pointer to data for sid-dialog

@return
*/
static void _e2_sidlg_init_positions (E2_SID_Runtime *rt)
{
	gchar *icondir, *fullname;
	GtkTreeIter iter;
	if (gtk_tree_model_get_iter_first (rt->model, &iter)
		&& e2_tree_find_iter_from_str_simple (rt->model, 0, rt->icon, &iter, FALSE))
	{	//the selected item is a stock item
		rt->page = 1;	//set notebook page accordingly
		//select & show the corresponding treeview line
		GtkTreePath *path = gtk_tree_model_get_path (rt->model, &iter);
		if (path != NULL)
		{
			GtkTreeViewColumn *column = gtk_tree_view_get_column (rt->treeview, 0);
			gtk_tree_view_set_cursor (rt->treeview, path, column, FALSE);
			gtk_tree_view_scroll_to_cell (rt->treeview, path, column, TRUE, 0.5, 0.5);
			gtk_tree_path_free (path);
		}
		//point the file-chooser to the icons dir, but no specific file
		icondir = _e2_sidlg_get_icon_dir (rt);
		fullname = g_strconcat (icondir, " ", NULL); //file can't be ""
		gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (rt->file_chooser),
			fullname);
		g_free (icondir);
		g_free (fullname);
	}
	else
	{ 	//the selected item is a custom-icon
		rt->page = 0;
		//point to the icon file, if possible
		if ((rt->icon != NULL) && (*rt->icon != '\0'))
		{
			gchar *local = F_FILENAME_TO_LOCALE (rt->icon);
			if (g_path_is_absolute (local))
				fullname = local;
			else
			{
				icondir = _e2_sidlg_get_icon_dir (rt);
				fullname = g_strconcat (icondir, local, NULL);
				g_free (icondir);
			}
			gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (rt->file_chooser),
				fullname);

			if (fullname != local)
				g_free (fullname);
			F_FREE (local);
		}
		else
		{
			icondir = _e2_sidlg_get_icon_dir (rt);
			fullname = g_strconcat (icondir, " ", NULL);
			gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (rt->file_chooser),
				fullname);
			g_free (icondir);
			g_free (fullname);
		}
	}
}

  /******************/
 /***** public *****/
/******************/

/**
@brief create icon-selection dialog

@param parent the main config-dialog widget
@param name dialog title string
@param icon gtk-stock-icon name, or path to icon file, taken from tree-option store
@param response_func callback function for all responses for this dialog
@param set pointer to data for the tree-option to which the dialog relates

@return the dialog widget, or NULL if error occurred
*/
GtkWidget *e2_sid_create (GtkWidget *parent, const gchar *name, gchar *icon,
	gpointer response_func, E2_OptionSet *set)
{
	//get or create and inititialize select image dialog runtime object and
	//ensure that only one runtime object (and only one dialog) exists
	//for every parent widget
	E2_SID_Runtime *rt =  g_object_get_data (G_OBJECT (parent), name);

	if (rt != NULL)
	{	//the dialog already exists
		printd (NOTICE, "select image dialog already exists");
		rt->icon = icon;
		//set up dialog for the possibly new icon
		_e2_sidlg_init_positions (rt);
		//present the window
		gtk_window_present (GTK_WINDOW (rt->dialog));
		gtk_notebook_set_current_page (rt->notebook, rt->page);

		return rt->dialog;
	}
	else
	{
		printd (NOTICE, "creating select image dialog");
		rt = ALLOCATE0 (E2_SID_Runtime);
		CHECKALLOCATEDWARN (rt, return NULL;)
		//set up runtime object
		rt->name = g_strdup (name);
		rt->icon = g_strdup (icon);
		rt->parent = parent;
	}

	//create dialog widgets

	rt->dialog = e2_dialog_create (NULL, NULL, rt->name, _e2_sidlg_response_cb, rt);
	g_signal_connect (rt->dialog, "response", G_CALLBACK (response_func), set);
	//when the dialog is destroyed, free the runtime object
	g_signal_connect (rt->dialog, "destroy", G_CALLBACK (_e2_sidlg_destroy_cb), rt);
	//ensure this dialog is destroyed with the parent widget
	if (parent != NULL)
		g_object_set_data_full (G_OBJECT (parent), rt->name, rt,
			(GDestroyNotify) _e2_sidlg_destroy_cb2);

	//a notebook is used to change between the stock icon selector and the
	//file chooser
	rt->notebook = GTK_NOTEBOOK (e2_widget_add_notebook
		(GTK_DIALOG (rt->dialog)->vbox, TRUE, E2_PADDING, _e2_sidlg_switch_page_cb, rt));

	rt->file_chooser = _e2_sidlg_create_file_chooser (rt->dialog);
	//the file chooser is inside a scrolled window bcause the available size
	//might change
	GtkWidget *sw = e2_widget_get_sw_plain (GTK_POLICY_NEVER, GTK_POLICY_NEVER);
	e2_widget_sw_add_with_viewport (sw, rt->file_chooser);
	GtkWidget *label = gtk_label_new_with_mnemonic (_("_file selection"));
	//add the scrolled window to the notebook
	gtk_notebook_append_page_menu (rt->notebook, sw, label, label);
	//and the stock icon browser is another scrolled window
	GtkWidget *sw2 = _e2_sidlg_create_stock_icon_browser (rt);
	label = gtk_label_new_with_mnemonic (_("_stock icons"));
	gtk_notebook_append_page_menu (rt->notebook, sw2, label, label);

	//show the currently selected notebook page, icon, file
	_e2_sidlg_init_positions (rt);

	rem_btn = e2_dialog_add_defined_button (rt->dialog, &E2_BUTTON_REMOVE);
#ifdef USE_GTK2_12
	gtk_widget_set_tooltip_text (
#else
	e2_widget_set_tooltip (NULL,
#endif
		rem_btn, _("Remove the current icon"));
	if (*icon == '\0')	//nothing to remove at present
		gtk_widget_set_sensitive (rem_btn, FALSE);

	gint page = rt->page;	//need to preserve this, or else it's set to 0, CHECKME how ??

	e2_dialog_show (rt->dialog, rt->parent, 0,
		&E2_BUTTON_APPLY, &E2_BUTTON_CANCEL, &E2_BUTTON_OK, NULL);

	rt->page = page;
	//must set page after the dialog is shown, to get the correct page - CHECKME why ??
	gtk_notebook_set_current_page (rt->notebook, page);
	e2_dialog_resize (rt->dialog, 1.5);

	return rt->dialog;
}
