#include <qstring.h>
#include <stdlib.h>
#include <stdio.h>
#include <qdir.h>
#include <dlfcn.h>
#include <math.h>
#include <qstringlist.h>

#include "ladspaeffect.h"

#ifdef HAVE_LADSPA

LadspaEffectChannel::LadspaEffectChannel(LadspaEffect *e, unsigned long in, unsigned long out) : EffectChannel((Effect *)e, in, out)
{
	e_in_port=in;
	e_out_port=out;
	e_gain=1.0;

	if(e->ports())
		e_port_data=new float[e->ports()];
	else
		e_port_data=NULL;
	e_handle=(*effect()->e_desc->instantiate)(effect()->e_desc, effect()->e_srate);
	if(!e_handle)
		return;
	if(effect()->e_desc->activate)
		(*effect()->e_desc->activate)(e_handle);
	unsigned int i;

	for(i=0;i<effect()->ports();++i)
	{
		if(effect()->isControl(i))
		{
			if(effect()->e_desc->connect_port)
				(*effect()->e_desc->connect_port)(e_handle, i, &e_port_data[i]);
		}
	}
printf("LadspaEffect instance created\n");
}

LadspaEffectChannel::LadspaEffectChannel(LadspaEffectChannel *c, LadspaEffect *e, unsigned long in, unsigned long out) : EffectChannel((EffectChannel *)c, (Effect *)e, in, out)
{
	e_handle=c->e_handle;
	e_in_port=in;
	e_out_port=out;
	e_slave=true;
	e_port_data=0;
	e_effect=e;
	e_gain=1.0;
}

LadspaEffectChannel::~LadspaEffectChannel()
{
	if(e_handle && !e_slave)
	{
		if(effect()->e_desc->deactivate)
			(*effect()->e_desc->deactivate)(e_handle);
		if(effect()->e_desc->cleanup)
			(*effect()->e_desc->cleanup)(e_handle);
	}
	if(e_port_data)
		delete e_port_data;
}

void LadspaEffectChannel::setBuffer(float *data)
{
	if(!e_handle)
		return;

	e_data=data;
	if(effect()->e_desc->connect_port)
	{
		(*effect()->e_desc->connect_port)(e_handle, e_in_port, data);
		(*effect()->e_desc->connect_port)(e_handle, e_out_port, data);
	}
}

void LadspaEffectChannel::process(unsigned long nsamples)
{
	if(!e_handle)
		return;

	if(e_slave)
		return;

	(*effect()->e_desc->run)(e_handle, nsamples);
	if(e_gain != 1.0)
	{
		unsigned long i;

		for(i=0;i<nsamples;++i)
			e_data[i]*=e_gain;
	}
}

void LadspaEffectChannel::setGain(float gain)
{
	e_gain=gain;
}

void LadspaEffectChannel::setControl(unsigned long port, float data)
{
	if(e_slave)
		return;

	if(port >= e_effect->ports())
		return;
	if(!effect()->isControl(port))
		return;
	e_port_data[port]=data;
}

LadspaEffect *LadspaEffectChannel::effect()
{
	return (LadspaEffect *)e_effect;
}

LadspaEffect::LadspaEffect()
{
	dl_handle=NULL;
	e_desc=NULL;
}

LadspaEffect::~LadspaEffect()
{
	if(e_left)
		delete e_left;
	if(e_right)
		delete e_right;
	e_left=e_right=0;
	if(dl_handle)
		dlclose(dl_handle);
}

bool LadspaEffect::loadEffect(unsigned long samplerate, unsigned long id, const char *name)
{
	e_srate=samplerate;

	QString ladspa_path=getenv("LADSPA_PATH");
	if(ladspa_path == QString::null || ladspa_path == "")
		ladspa_path="/usr/lib/ladspa:/usr/local/lib/ladspa";
	
	QStringList path_list=QStringList::split(':', ladspa_path, false);

	unsigned i;

	QString id_string;

	id_string.sprintf("*.so*");
//	id_string.sprintf("*_%lu.so*", id);
	for(i=0;i<path_list.count();++i)
	{
		QDir d(path_list[i], id_string, QDir::Name | QDir::IgnoreCase, QDir::Files);
		QStringList entries=d.entryList();
		while(entries.count() > 0)
		{
			if(0 && entries.count() > 1)
			{
				printf("Error: more than one matching entry in LADSPA plugin list by ID\n");
				return false;
			}
//			printf("Open file %s\n", (const char *)entries[0]);
			QFileInfo info(path_list[i]+"/"+entries[0]);
			QCString abs=info.absFilePath().local8Bit();

			void *handle=dlopen(abs, RTLD_LAZY);
			if(handle == NULL)
			{
				entries.remove(entries.first());
				continue;
			}
			LADSPA_Descriptor_Function ladspa_descriptor=(LADSPA_Descriptor_Function)dlsym(handle, "ladspa_descriptor");
			if(ladspa_descriptor == NULL)
			{
				entries.remove(entries.first());
				dlclose(handle);
				continue;
			}

			int j;

			for(j=0;j<500;++j)
			{
				const LADSPA_Descriptor *desc=(*ladspa_descriptor)(j);
				if(!desc)
					break;
				
//				printf("Descriptor %ld name %s\n", desc->UniqueID, desc->Label);
				if(desc->UniqueID == id)
				{
					if(!strcmp(desc->Label, name))
					{
						dl_handle=handle;
						e_desc=desc;

						unsigned long k;
						int ai=0, ao=0;
						unsigned long in_ports[ports()];
						unsigned long out_ports[ports()];
						for(k=0;k<ports();++k)
						{
printf("Port %lu: %s\n", k, desc->PortNames[k]);
							if(isControl(k))
								continue;
							if(isInput(k))
							{
								in_ports[ai]=k;
								++ai;
							}
							else
							{
								out_ports[ao]=k;
								++ao;
							}
						}
						if(!LADSPA_IS_HARD_RT_CAPABLE(desc->Properties) ||
						   LADSPA_IS_INPLACE_BROKEN(desc->Properties))
						{
							entries.remove(entries.first());
							printf("Plugin found, but not suitable, prop %d\n", desc->Properties);
							dlclose(handle);
							e_desc=NULL;
							dl_handle=NULL;
							continue;
						}
						if(ai == 1 && ao == 1)
						{
							e_left=(EffectChannel *)new LadspaEffectChannel(this, in_ports[0], out_ports[0]);
							e_right=(EffectChannel *)new LadspaEffectChannel(this, in_ports[0], out_ports[0]);
							setupPorts();
							return true;
						}
						if(ai == 2 && ao == 2)
						{
							e_left=(EffectChannel *)new LadspaEffectChannel(this, in_ports[0], out_ports[0]);
							e_right=(EffectChannel *)new LadspaEffectChannel((LadspaEffectChannel *)left(), this, in_ports[1], out_ports[1]);
							setupPorts();
							return true;
						}
						printf("LadspaEffect has more than two channels, %d in, %d out\n", ai, ao);
						e_desc=NULL;
					}
				}
			}
			dlclose(handle);
			entries.remove(entries.first());
		}
	}
	dl_handle=NULL;
	return false;
}

unsigned long LadspaEffect::ports()
{
	if(!e_desc)
		return 0;
	return e_desc->PortCount;
}

bool LadspaEffect::isInput(unsigned long port)
{
	if(port >= ports())
		return false;
	return(LADSPA_IS_PORT_INPUT(e_desc->PortDescriptors[port]));
}

bool LadspaEffect::isControl(unsigned long port)
{
	if(port >= ports())
		return false;
	return(LADSPA_IS_PORT_CONTROL(e_desc->PortDescriptors[port]));
}

void LadspaEffect::setControl(unsigned long port, float data)
{
	left()->setControl(port, data);
	right()->setControl(port, data);
}

void LadspaEffect::setGain(float gain)
{
	left()->setGain(gain);
	right()->setGain(gain);
}

float LadspaEffect::portMin(unsigned long port)
{
	if(port >= ports())
		return 0;
	if(!isControl(port))
		return 0;
	const LADSPA_PortRangeHint *hint=&e_desc->PortRangeHints[port];

	if(LADSPA_IS_HINT_BOUNDED_BELOW(hint->HintDescriptor))
	{
		if(LADSPA_IS_HINT_SAMPLE_RATE(hint->HintDescriptor))
			return hint->LowerBound*(float)e_srate;
		else
			return hint->LowerBound;
	}
	return -HUGE_VAL;
}

float LadspaEffect::portMax(unsigned long port)
{
	if(port >= ports())
		return 0;
	if(!isControl(port))
		return 0;
	const LADSPA_PortRangeHint *hint=&e_desc->PortRangeHints[port];

	if(LADSPA_IS_HINT_BOUNDED_ABOVE(hint->HintDescriptor))
	{
		if(LADSPA_IS_HINT_SAMPLE_RATE(hint->HintDescriptor))
			return hint->UpperBound*(float)e_srate;
		else
			return hint->UpperBound;
	}
	return HUGE_VAL;
}

float LadspaEffect::portDefault(unsigned long port)
{
	if(port >= ports())
		return 0;
	if(!isControl(port))
		return 0;
	const LADSPA_PortRangeHint *hint=&e_desc->PortRangeHints[port];

	if(!LADSPA_IS_HINT_HAS_DEFAULT(hint->HintDescriptor))
		return 0;
	if(LADSPA_IS_HINT_DEFAULT_MINIMUM(hint->HintDescriptor))
		return portMin(port);
	if(LADSPA_IS_HINT_DEFAULT_MAXIMUM(hint->HintDescriptor))
		return portMax(port);
	if(LADSPA_IS_HINT_DEFAULT_0(hint->HintDescriptor))
		return 0.0;
	if(LADSPA_IS_HINT_DEFAULT_1(hint->HintDescriptor))
		return 1.0;
	if(LADSPA_IS_HINT_DEFAULT_100(hint->HintDescriptor))
		return 100.0;
	if(LADSPA_IS_HINT_DEFAULT_440(hint->HintDescriptor))
		return 440.0;
	if(LADSPA_IS_HINT_DEFAULT_LOW(hint->HintDescriptor))
		return (portMax(port)-portMin(port))/4+portMin(port);
	if(LADSPA_IS_HINT_DEFAULT_HIGH(hint->HintDescriptor))
		return (portMax(port)-portMin(port))/4*3+portMin(port);
	if(LADSPA_IS_HINT_DEFAULT_MIDDLE(hint->HintDescriptor))
		return (portMax(port)-portMin(port))/2+portMin(port);
	return 0.0;
}

void LadspaEffect::setupPorts()
{
	unsigned long k;

	for(k=0;k<ports();++k)
		if(isControl(k))
			setControl(k, portDefault(k));
}

const char *LadspaEffect::portName(unsigned long port)
{
	if(port >= ports())
		return NULL;

	return e_desc->PortNames[port];
}

float LadspaEffect::control(unsigned long port)
{
	if(port >= ports())
		return 0.0;
	if(!isControl(port))
		return 0.0;
	
	return left()->e_port_data[port];
}

LadspaEffectChannel *LadspaEffect::left()
{
	return (LadspaEffectChannel *)e_left;
}

LadspaEffectChannel *LadspaEffect::right()
{
	return (LadspaEffectChannel *)e_right;
}

#endif /* HAVE LADSPA */
