/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/logger/file_logger.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::File_Logger
 *
 * $Id: file_logger.t.cpp,v 1.15 2005/06/23 09:54:21 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/logger/file_logger.hpp>

// used components
#include <diagnostics/frame/record.hpp>
#include <diagnostics/frame/logging_config_exception.hpp>

// test support
#include <diagnostics/util/to_string.hpp>

#include <diagnostics/util/dummy_values_for_tid_pid.ts.hpp>

#include <sstream>

#define TEST_COMPONENT_NAME File_Logger
#define TEST_COMPONENT_NAMESPACE diagnostics

/**
 * @brief two records to play with -- assembly_cstr
 */
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
#  define RECORD_1 \
    Record r1(LEVEL_TEST,TYPE_TESTCASE_ENTER,0,"what","base","file",1, \
              Dummy_Values_For_Pid::value2,Dummy_Values_For_Tid::value3,4,5,"host")
#  define RECORD_2 \
    Record r2(LEVEL_AUDIT,TYPE_TESTCASE_EXIT,1,"what1","base1","file1",2, \
              Dummy_Values_For_Pid::value3,Dummy_Values_For_Tid::value2,5,6,"host1")
#else 
#  define RECORD_1 \
    Record r1(LEVEL_TEST,TYPE_TESTCASE_ENTER,0,"what","base","file",1)
#  define RECORD_2 \
    Record r2(LEVEL_AUDIT,TYPE_TESTCASE_EXIT,1,"what1","base1","file1",2)
#endif

#define TEST_FILENAME "test/scratch/file_logger.log"

DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace ::diagnostics::unittest;

void all_methods(Test_Data & data)
{
    using ::diagnostics::internal::to_string;

    RECORD_1;
    RECORD_2;

    int const BUFFER_SIZE(1024);
    char buffer[BUFFER_SIZE];

    // write 
    {
		// construct with overwrite to ensure independence
		File_Logger logger(TEST_FILENAME, ::std::ios::out);
	
		// log two records
		logger.log(r1);
		logger.log(r2);
	
		// close
		logger.close();
    }
    
    // read
    {
		::std::ifstream logfile(TEST_FILENAME);
		// the file exists
		TEST_ASSERT(logfile.good());
	
		// the first line is the first record
		logfile.getline(buffer,BUFFER_SIZE);
		TEST_ASSERT(buffer==to_string(r1));
	
		// the second line is the second record
		logfile.getline(buffer,BUFFER_SIZE);
		TEST_ASSERT(buffer==to_string(r2));

		// check for eof
		logfile.get();
		TEST_ASSERT(logfile.eof());
    }
    
    // append
    {
		File_Logger logger(TEST_FILENAME);
	
		// log two records
		logger.log(r1);
		logger.log(r2);
	
		// close
		logger.close();
    }

    // read
    {
		::std::ifstream logfile(TEST_FILENAME);
		// the file exists
		TEST_ASSERT(logfile.good());
	
		// the first line is the first record
		logfile.getline(buffer,BUFFER_SIZE);
		TEST_ASSERT(buffer==to_string(r1));
	
		// the second line is the second record
		logfile.getline(buffer,BUFFER_SIZE);
		TEST_ASSERT(buffer==to_string(r2));

		// the third line is the first record
		logfile.getline(buffer,BUFFER_SIZE);
		TEST_ASSERT(buffer==to_string(r1));
	
		// the fourth line is the second record
		logfile.getline(buffer,BUFFER_SIZE);
		TEST_ASSERT(buffer==to_string(r2));

		// check for eof
		logfile.get();
		TEST_ASSERT(logfile.eof());
    }
    

}


void file_does_not_exist(Test_Data & data)
{
    TEST_THROWING_BLOCK_ENTER;
    File_Logger logger("nonextisting_directory/so_this_file_does_not_exist");
    TEST_THROWING_BLOCK_EXIT(Logging_Config_Exception);
}

void log_1024_1024(Test_Data & data)
{
    RECORD_1;
    
    {
		File_Logger logger(TEST_FILENAME, ::std::ios::out);
	
		for(int i=0;i<1024*1024;++i) logger.log(r1);
		logger.close();
    }
    // clean up
    File_Logger logger(TEST_FILENAME, ::std::ios::out);
    logger.log(r1);
    logger.close();
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&all_methods,LEVEL_PROD);
TEST_NORMAL_CASE(&file_does_not_exist,LEVEL_PROD);
TEST_NONFCT_CASE(&log_1024_1024,30);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
