#!/usr/bin/ruby -w

=begin
    Ruby support library for dhelp database access

    Copyright (C) 2005  Esteban Manchado Velzquez

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
=end

require 'bdb'

# Dhelp-related classes
module Dhelp
    # C struct wrapper
    class CStructWrapper
        # Class methods
        class <<self
            # Get/Set the pack format
            def pack_fmt(fmt = nil)
                @fmt = fmt if fmt
                @fmt
            end
            # Get/Set the field list. It can be set as an array, or as several
            # parameters
            def field_list(*args)
                case args.size
                when 0
                    # Just get the value, do nothing
                when 1
                    first = args.first
                    @fieldList = first.kind_of?(Array) ? first : [first]
                else
                    @fieldList = args
                end
                @fieldList.map {|f| f.to_sym}
            end
        end

        def initialize(data)
            case data
            when String
                @data = data
            when Hash
                @data = keys.map {|f| data[f]}.pack(pack_fmt)
            else
                raise ArgumentError, "Argument must be either String or Hash"
            end
        end

        # Handy shortcut methods
        def keys;     self.class.field_list; end
        def pack_fmt; self.class.pack_fmt;   end

        def get_field(f)
            i = keys.index(f.to_sym)
            if i
                @data.unpack(pack_fmt)[i]
            else
                raise ArgumentError, "Unknown field '#{f}'"
            end
        end

        # Returns a Hash object with all the C struct fields
        def to_hash
            h = {}
            keys.each do |f|
                h[f] = get_field(f)
            end
            h
        end

        # Returns the data in C format
        def to_raw_data
            @data
        end

        # Catches missing methods, to get the field values
        def method_missing(meth, *args)
            if keys.include? meth.to_sym
                get_field(meth)
            else
                super
            end
        end
    end


    # Key data entry
    class KeyData < CStructWrapper
        pack_fmt    'Z100 Z100 Z100'
        field_list  %w(file dir name)
    end


    # Value data entry
    class ValueData < CStructWrapper
        pack_fmt    'Z1000'
        field_list  %w(descrip)
    end


    # Title database key data entry
    class TitleKeyData < CStructWrapper
        pack_fmt    'Z1000'
        field_list  %w(dir)
    end


    # Title database value data entry
    class TitleValueData < CStructWrapper
        pack_fmt    'Z1000'
        field_list  %w(dtitle)
    end


    # Item data entry
    class ItemData < CStructWrapper
        pack_fmt    'Z100 Z100 Z100 Z100 Z1000'
        field_list  %w(dir dtitle name file descrip)
    end


    # Dhelp database
    class Database < BDB::Btree
        STD_COMPARISON = lambda {|a,b|
                                    dataA, dataB = KeyData.new(a), KeyData.new(b)
                                    r = dataA.dir <=> dataB.dir
                                    r == 0 ? (dataA.name <=> dataB.name) : r
                                }

        def Database.open(flags   = BDB::RDONLY,
                          options = {},
                          mode    = 0644,
                          # name    = 'dhelpdbase',
                          name    = '/var/lib/dhelp/dbase',
                          subname = nil)
            defaultOptions = {"flags"      => 0,
                              "cachesize"  => 10000,
                              "minkeypage" => 0,
                              "psize"      => 0,
                              "compare"    => STD_COMPARISON,
                              "prefix"     => nil,
                              "lorder"     => 0}
            super(name, subname, flags, mode, defaultOptions.merge(options))
        end

        # Writes an ItemData object to the database
        def write(data)
            key = KeyData.new(:file => data.file, :dir => data.dir,
                              :name => data.name)
            value = ValueData.new(:descrip => data.descrip)
            put(key.to_raw_data, value.to_raw_data)
        end

        def del(data)
            key = KeyData.new(:file => data.file, :dir => data.dir,
                              :name => data.name)
            delete(key.to_raw_data)
        end

        # Hide delete method, always use the high-level one
        protected :delete

        # Traverse entire BD, passing each item to the block
        def each
            super do |k,v|
                key, value = KeyData.new(k), ValueData.new(v)
                # Note: missing dtitle field
                yield ItemData.new(:file    => key.file,
                                   :dir     => key.dir,
                                   :name    => key.name,
                                   :descrip => value.descrip)
            end
        end

        # Traverse each _item_, collecting their categories, and pass the
        # category and item list to the given block
        def each_category
            itemList        = {}
            each do |item|
                itemList[item.dir] ||= []
                itemList[item.dir] << item
            end

            orderedCategories = itemList.keys.sort {|a,b|
                # Order subcategories first
                case tmp = a.scan('/').size <=> b.scan('/').size
                when 0
                    a <=> b
                else
                    tmp * -1
                end
            }
            orderedCategories.each do |cat|
                yield cat, itemList[cat].sort {|a,b| a.name <=> b.name}
            end
        end
    end


    # Dhelp titles database
    class TitleDatabase < BDB::Hash
        def TitleDatabase.open(flags   = BDB::RDONLY,
                                options = {},
                                mode    = 0644,
                                name    = '/var/lib/dhelp/titles',
                                # name    = 'dhelptitles',
                                subname = nil)
            defaultOptions = {"ffactor"   => 8,
                              "nelem"     => 1,
                              "cachesize" => 5000,
                              "hash"      => nil,
                              "lorder"    => 0}
            super(name, subname, flags, mode, defaultOptions.merge(options))
        end

        # Traverse entire BD, passing each item to the block
        def each
            super do |k,v|
                key, value = KeyData.new(k), ValueData.new(v)
                # Note: missing dtitle field
                yield ItemData.new(:file    => key.file,
                                   :dir     => key.dir,
                                   :name    => key.name,
                                   :descrip => value.descrip)
            end
        end

        def title_for(dir)
            valueData = get(TitleKeyData.new(:dir => dir).to_raw_data)
            valueData ? TitleValueData.new(valueData).dtitle : nil
        end

        # Writes an ItemData object to the database
        def write(data)
            key = TitleKeyData.new(:dir => data.dir)
            value = TitleValueData.new(:dtitle => data.dtitle)
            put(key.to_raw_data, value.to_raw_data)
        end
    end
end
