/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   Copyright (C) 2000 Trolltech AS.  All rights reserved.                *
*   info@trolltech.com                                                    *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include "sqlformwizardimpl.h"

#include "datatable.h"
#include "asciivalidator.h"
#include "databaseconnection.h"

#include <kconfig.h>
#include <kmessagebox.h>
#include <kurldrag.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <kaction.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <kdebug.h>
#include <kinputdialog.h>
#include <kicondialog.h>

#include "project.h"
#include "sqlformwizardimpl.h"

#include <qlistbox.h>
#include <qwidget.h>
#include <qcheckbox.h>
#include <qlineedit.h>

#include <qlabel.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qpushbutton.h>
#include <qmultilineedit.h>
#include <qlistview.h>
#include <qfeatures.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <limits.h>

#include <qdatatable.h>
#include <qdatabrowser.h>
#include <qdataview.h>
#include <qsqleditorfactory.h>
#include <qsqlindex.h>
#include <qsqlcursor.h>
#include <qsqldatabase.h>

static bool m_blockChanges = false;

SqlFormWizard::SqlFormWizard( Project *project, DataTable *table, DataTable *parentTable, bool configure )
        : SqlFormWizardBase( table ),
        m_project( project ),
        m_list( table ),
        m_parentTable( parentTable ),
        m_configure( configure ),
        mode( None )
{
    mode = Table;

    setMinimumSize( 800, 600 );
    setCaption( i18n("Data Table Wizard") );
    setAppropriate( sortPage, true );
    setFinishEnabled( sortPage, true );
    setAppropriate( relationPage, false );

    wizard1->setPixmap( UserIcon( "wizard_table_1" ) );
    wizard2->setPixmap( UserIcon( "wizard_table_2" ) );
    wizard3->setPixmap( UserIcon( "wizard_table_3" ) );
    wizard4->setPixmap( UserIcon( "wizard_table_4" ) );
    wizard5->setPixmap( UserIcon( "wizard_table_2" ) );

    buttonIcon->setPixmap( SmallIcon( "table", 32 ) );
    buttonFieldUp->setPixmap( UserIcon( "up" ) );
    buttonFieldDown->setPixmap( UserIcon( "down" ) );
    buttonAddSortField->setPixmap( UserIcon( "right" ) );
    buttonRemoveSortField->setPixmap( UserIcon( "left" ) );
    buttonUpSortField->setPixmap( UserIcon( "up" ) );
    buttonDownSortField->setPixmap( UserIcon( "down" ) );
    buttonReSortField->setPixmap( UserIcon( "re-sort" ) );

    connect( nextButton(), SIGNAL( clicked() ), SLOT( nextPageClicked() ) );

    setupFirstPage();
}

SqlFormWizard::~SqlFormWizard()
{}

/****************connectionPage methods****************/
void SqlFormWizard::doConnect()
{
    DatabaseConnection * conn = new DatabaseConnection();
    conn->setName( editName->text() );
    conn->setDriver( comboDriver->lineEdit() ->text() );
    conn->setDatabase( editDatabase->text() );
    conn->setUsername( editUsername->text() );
    conn->setPassword( editPassword->text() );
    conn->setHostname( editHostname->text() );
    conn->setPort( editPort->value() );
    if ( conn->refreshCatalog() )
    {
        if ( listBoxConnection->currentText().isEmpty() )
        {
            m_project->addDatabaseConnection( conn );
            listBoxConnection->insertItem( conn->name() );
            listBoxConnection->setCurrentItem( listBoxConnection->count() - 1 );
        }
        else if ( m_project->databaseConnection( listBoxConnection->currentText() ) )
        {
            m_project->removeDatabaseConnection( listBoxConnection->currentText() );
            m_project->addDatabaseConnection( conn );
        }
    }
    else
    {
        QMessageBox::warning( NULL, i18n( "Connection" ),
                                QString( i18n("Could not connect to the database.\n"
                                        "Please ensure that the database server is running "
                                        "and that all the connection information is correct.\n") +
                                        "[ " + conn->lastError() + " ]" ) );
        delete conn;
    }
}

void SqlFormWizard::newConnection()
{
    m_blockChanges = true;
    enableAllConnectionPage( true );
    QString n( "(default)" );
    if ( m_project->databaseConnection( n ) )
    {
        n = "connection";
        int i = 2;
        while ( m_project->databaseConnection( n + QString::number( i ) ) )
            ++i;
        n = n + QString::number( i );
    }
    editName->setText( n );
    listBoxConnection->clearSelection();
    buttonConnect->setDefault( true );
    editName->setFocus();
    m_blockChanges = false;
}

void SqlFormWizard::deleteConnection()
{
    if ( listBoxConnection->currentItem() == -1 )
        return ;
    m_project->removeDatabaseConnection( listBoxConnection->currentText() );
    delete listBoxConnection->item( listBoxConnection->currentItem() );
    if ( listBoxConnection->count() )
    {
        listBoxConnection->setCurrentItem( 0 );
        currentConnectionChanged( listBoxConnection->currentText() );
    }
    else
    {
        enableAllConnectionPage( false );
    }
}

void SqlFormWizard::connectionNameChanged( const QString &s )
{
    if ( listBoxConnection->currentItem() == 0 || m_blockChanges )
        return ;
    listBoxConnection->changeItem( s, listBoxConnection->currentItem() );
}

void SqlFormWizard::currentConnectionChanged( const QString &s )
{
    DatabaseConnection * conn = m_project->databaseConnection( s );
    m_blockChanges = true;
    enableAllConnectionPage( conn != 0 );
    editName->setEnabled( false );
    m_blockChanges = false;
    if ( !conn )
        return ;
    m_list->setConnection( s );
    m_blockChanges = true;
    editName->setText( conn->name() );
    m_blockChanges = false;
    comboDriver->lineEdit() ->setText( conn->driver() );
    editDatabase->setText( conn->database() );
    editUsername->setText( conn->username() );
    editPassword->setText( conn->password() );
    editHostname->setText( conn->hostname() );
    editPort->setValue( conn->port() );

    setNextEnabled( connectionPage, true );
}

void SqlFormWizard::enableAllConnectionPage( bool b )
{
    editName->setEnabled( b );
    editName->setText( "" );
    comboDriver->setEnabled( b );
    comboDriver->lineEdit() ->setText( "" );
    editDatabase->setEnabled( b );
    editDatabase->setText( "" );
    editUsername->setEnabled( b );
    editUsername->setText( "" );
    editPassword->setEnabled( b );
    editPassword->setText( "" );
    editHostname->setEnabled( b );
    editHostname->setText( "" );
    editPort->setEnabled( b );
    editPort->setValue( -1 );
    buttonDelete->setEnabled( b );
    buttonConnect->setEnabled( b );
}

/****************relationPage methods****************/
void SqlFormWizard::parentKeyChanged()
{
    comboChildTable->clear();
    m_list->setParentKey( listBoxParentKey->currentText() );
    comboChildTable->setEnabled( true );
    comboChildTable->insertItem( i18n("<select child table>") );
    comboChildTable->insertStringList( m_project->databaseTableList( m_list->connection() ) );
}

void SqlFormWizard::childTableActivated( const QString &s )
{
    m_list->setTableName( s );
    comboChildKey->clear();
    comboChildKey->setEnabled( true );
    comboChildKey->insertItem( i18n("<select foreign key>") );
    comboChildKey->insertStringList( m_project->databaseFieldList( m_list->connection(), m_list->tableName() ) );
}

void SqlFormWizard::childKeyActivated( const QString &s )
{
    m_list->setForeignKey( s );

    comboChildRelation->clear();
    comboChildRelation->setEnabled( true );
    comboChildRelation->insertItem( i18n("<select relationship>") );
    comboChildRelation->insertItem( i18n("one-to-one") );
    comboChildRelation->insertItem( i18n("one-to-many") );
    comboChildRelation->insertItem( i18n("many-to-one") );
    comboChildRelation->insertItem( i18n("many-to-many") );
}

void SqlFormWizard::childRelationActivated( const QString &s )
{
    if ( s == i18n("one-to-one") )
        m_list->setRelationToParent( DataTable::OneToOne );
    else if ( s == i18n("one-to-many") )
        m_list->setRelationToParent( DataTable::OneToMany );
    else if ( s == i18n("many-to-one") )
        m_list->setRelationToParent( DataTable::ManyToOne );
    else if ( s == i18n("many-to-many") )
        m_list->setRelationToParent( DataTable::ManyToMany );
    else
        return;

    setNextEnabled( relationPage, true );
}

void SqlFormWizard::enableAllRelationPage( bool )
{}

/****************tablePage methods****************/
void SqlFormWizard::tableSelected( const QString &name )
{
    editTable->blockSignals( true );
    editTable->setText( name );
    editTable->blockSignals( false );

    if ( m_list->name() == name )
        return ;

    m_list->setName( m_project->uniqueDataTableName( name ) );
    m_list->setTableName( listBoxTable->currentText() );

    setNextEnabled( tablePage, true );
}

void SqlFormWizard::tableLabelChanged( const QString &label )
{
    m_list->setName( m_project->uniqueDataTableName( label ) );
}

void SqlFormWizard::tableIconChanged()
{
    QString iconName = KIconDialog::getIcon( KIcon::NoGroup, KIcon::Any, false, 32, true, this, i18n("Select Icon for This Table") );
    buttonIcon->setPixmap( SmallIcon( iconName, 32 ) );
    m_list->setIconName( iconName );
}

void SqlFormWizard::enableAllTablePage( bool b )
{
    editTable->setEnabled( b );
    buttonIcon->setEnabled( b );
    checkBoxSorting->setEnabled( b );
    checkBoxReadOnly->setEnabled( b );
    checkBoxConfirmInserts->setEnabled( b );
    checkBoxConfirmUpdates->setEnabled( b );
    checkBoxConfirmDeletes->setEnabled( b );
    checkBoxConfirmCancels->setEnabled( b );
}

/****************fieldPage methods****************/
void SqlFormWizard::fieldUp()
{
    if ( listBoxField->currentItem() <= 0 ||
            listBoxField->count() < 2 )
        return ;
    int index = listBoxField->currentItem() - 1;
    QListBoxItem *i = listBoxField->item( listBoxField->currentItem() );
    listBoxField->takeItem( i );
    listBoxField->insertItem( i, index );
    listBoxField->setCurrentItem( i );

    sortDataFields();
}

void SqlFormWizard::fieldDown()
{
    if ( listBoxField->currentItem() == -1 ||
            listBoxField->currentItem() == ( int ) listBoxField->count() - 1 ||
            listBoxField->count() < 2 )
        return ;
    int index = listBoxField->currentItem() + 1;
    QListBoxItem *i = listBoxField->item( listBoxField->currentItem() );
    listBoxField->takeItem( i );
    listBoxField->insertItem( i, index );
    listBoxField->setCurrentItem( i );

    sortDataFields();
}

void SqlFormWizard::sortDataFields()
{
    int k = -1;
    for ( uint j = 0; j < listBoxField->count(); ++j )
    {
        DataField *field = findField( listBoxField->item( j ) );
        if ( !field->hidden() )
            field->setNumber( ++k );
        else
            field->setNumber( -1 );
    }
}

void SqlFormWizard::removeField()
{
    int i = listBoxField->currentItem();
    if ( i != -1 )
    {
        listBoxField->insertItem( listBoxField->currentText() );
        listBoxField->removeItem( i );
    }
}

void SqlFormWizard::addField()
{
    int i = listBoxField->currentItem();
    if ( i == -1 )
        return ;
    QString f = listBoxField->currentText();
    if ( !f.isEmpty() )
        listBoxField->insertItem( f );
    listBoxField->removeItem( i );
}

void SqlFormWizard::currentFieldChanged( QListBoxItem *i )
{
    DataField * field = findField( i );
    if ( !i || !field )
    {
        editField->setEnabled( false );
        checkBoxForeign->setEnabled( false );
        checkBoxHidden->setEnabled( false );
        checkBoxResizable->setEnabled( false );
        checkBoxSortable->setEnabled( false );

        comboTargetTable->setEnabled( false );
        comboTargetKey->setEnabled( false );
        comboTargetField->setEnabled( false );
        return ;
    }

    editField->setEnabled( true );
    checkBoxForeign->setEnabled( true );
    checkBoxHidden->setEnabled( true );
    checkBoxResizable->setEnabled( true );
    checkBoxSortable->setEnabled( true );

    editField->setText( field->label() );
    checkBoxForeign->setChecked( field->foreign() );
    checkBoxHidden->setChecked( field->hidden() );
    checkBoxResizable->setChecked( field->resizable() );
    checkBoxSortable->setChecked( field->sortable() );

    comboTargetTable->setEnabled( false );
    comboTargetKey->setEnabled( false );
    comboTargetField->setEnabled( false );

    comboTargetTable->clear();
    comboTargetKey->clear();
    comboTargetField->clear();

    foreignToggled( field->foreign() );
}

void SqlFormWizard::fieldLabelChanged( const QString &label )
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field )
        return ;

    field->setLabel( label );
}

void SqlFormWizard::foreignToggled( bool toggled )
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field )
        return ;

    if ( !toggled )
    {
        comboTargetTable->setEnabled( false );
        comboTargetKey->setEnabled( false );
        comboTargetField->setEnabled( false );
        comboTargetTable->clear();
        comboTargetKey->clear();
        comboTargetField->clear();
        if ( field->relation() )
        {
            DataRelation *relation = field->relation();
            delete relation;
            relation = 0L;
        }
    }

    field->setForeign( toggled );
    listBoxField->blockSignals( true );

    if ( field->foreign() )
    {
        if ( !field->relation() )
            m_list->addDataRelation( field->name(), QString::null, QString::null,
                                     QString::null, QString::null, QVariant::Type( 0 ), 0, 0 );

        DatabaseConnection *conn = m_project->databaseConnection( m_list->connection() );

        comboTargetTable->setEnabled( true );
        comboTargetTable->insertItem( i18n("<select target table>") );
        comboTargetTable->insertStringList( conn->tables() );

        if ( !field->relation() ->table().isEmpty() )
        {
            comboTargetTable->setCurrentText( field->relation() ->table() );
            targetTableActivated(); //Goto the next step
        }
        if ( !field->primary() )
            listBoxField->changeItem( UserIcon( "foreignkey" ), listBoxField->currentText(), listBoxField->currentItem() );
    }
    else if ( field->hidden() && !field->primary() )
        listBoxField->changeItem( UserIcon( "hidden" ), listBoxField->currentText(), listBoxField->currentItem() );
    else if ( !field->primary() )
        listBoxField->changeItem( UserIcon( "visible" ), listBoxField->currentText(), listBoxField->currentItem() );

    listBoxField->blockSignals( false );
}

void SqlFormWizard::targetTableActivated()
{
    DatabaseConnection * conn = m_project->databaseConnection( m_list->connection() );
    comboTargetKey->clear();
    comboTargetField->clear();
    comboTargetKey->setEnabled( true );
    comboTargetField->setEnabled( true );
    comboTargetKey->insertItem( i18n("<select target key>") );
    comboTargetField->insertItem( i18n("<select target field>") );
    QStringList fields = conn->fields( comboTargetTable->currentText() );
    comboTargetKey->insertStringList( fields );
    comboTargetField->insertStringList( fields );

    DataField *field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || comboTargetTable->currentText() == i18n("<select target table>") )
    {
        comboTargetKey->setEnabled( false );
        comboTargetField->setEnabled( false );
        comboTargetKey->clear();
        comboTargetField->clear();
        return ;
    }

    field->relation() ->setTable( comboTargetTable->currentText() );

    if ( !field->relation() ->key().isEmpty() &&
         fields.contains( field->relation() ->key() ) )
    {
        comboTargetKey->setCurrentText( field->relation() ->key() );
        targetKeyActivated(); //Goto the next step
    }
    if ( !field->relation() ->field().isEmpty() &&
         fields.contains( field->relation() ->field() ) )
    {
        comboTargetField->setCurrentText( field->relation() ->field() );
        targetFieldActivated(); //Goto the next step
    }
}

void SqlFormWizard::targetKeyActivated()
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || comboTargetKey->currentText() == i18n("<select target key>") )
        return ;

    field->relation() ->setKey( comboTargetKey->currentText() );
}

void SqlFormWizard::targetFieldActivated()
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || comboTargetField->currentText() == i18n("<select target field>") )
        return ;

    field->relation() ->setField( comboTargetField->currentText() );

    DatabaseConnection *conn = m_project->databaseConnection( m_list->connection() );
    QVariant::Type fieldType = conn->getFieldType( field->relation() ->table(), field->relation() ->field() );
    field->relation() ->setType( fieldType );
}

void SqlFormWizard::hiddenToggled( bool toggled )
{
    DataField * field = findField( listBoxField->selectedItem() );
    if ( !field )
        return ;

    if ( toggled && !field->hidden() && field->isRequired() )
        KMessageBox::information( this, i18n( "This is a required field for inserting new records. If you set this to hidden then it must either be generated by the SQL server ( triggers, auto increment, ... ) OR you must first setup how it is to be calculated.  If you do not, then INSERT statements will not work correctly." ) );

    field->setHidden( toggled );

    if ( field->primary() || field->foreign() )
    {
        sortDataFields();
        return ;
    }

    listBoxField->blockSignals( true );

    if ( field->hidden() )
        listBoxField->changeItem( UserIcon( "hidden" ), listBoxField->currentText(), listBoxField->currentItem() );
    else
        listBoxField->changeItem( UserIcon( "visible" ), listBoxField->currentText(), listBoxField->currentItem() );

    listBoxField->blockSignals( false );

    sortDataFields();
}

void SqlFormWizard::resizableToggled( bool toggled )
{
    DataField * field = findField( listBoxField->selectedItem() );
    if ( !field )
        return ;

    field->setResizable( toggled );
}

void SqlFormWizard::sortableToggled( bool toggled )
{
    DataField * field = findField( listBoxField->selectedItem() );
    if ( !field )
        return ;

    field->setSortable( toggled );
}

void SqlFormWizard::enableAllFieldPage( bool b )
{
    editField->setEnabled( b );
    checkBoxHidden->setEnabled( b );
    checkBoxResizable->setEnabled( b );
    checkBoxSortable->setEnabled( b );
    checkBoxForeign->setEnabled( b );
}

/****************sqlPage methods****************/
void SqlFormWizard::addSortField()
{
    int i = listBoxSortField->currentItem();
    if ( i == -1 )
        return ;
    QString f = listBoxSortField->currentText();
    if ( !f.isEmpty() )
        listBoxSortedField->insertItem( f + " ASC" );
}

void SqlFormWizard::reSortSortField()
{
    int i = listBoxSortedField->currentItem();
    if ( i != -1 )
    {
        QString text = listBoxSortedField->currentText();
        if ( text.mid( text.length() - 3 ) == "ASC" )
            text = text.mid( 0, text.length() - 3 ) + "DESC";
        else if ( text.mid( text.length() - 4 ) == "DESC" )
            text = text.mid( 0, text.length() - 4 ) + "ASC";
        listBoxSortedField->removeItem( i );
        listBoxSortedField->insertItem( text, i );
        listBoxSortedField->setCurrentItem( i );
    }
}

void SqlFormWizard::removeSortField()
{
    int i = listBoxSortedField->currentItem();
    if ( i != -1 )
    {
        listBoxSortedField->removeItem( i );
    }
}

void SqlFormWizard::sortFieldUp()
{
    if ( listBoxSortedField->currentItem() <= 0 ||
            listBoxSortedField->count() < 2 )
        return ;
    int index = listBoxSortedField->currentItem() - 1;
    QListBoxItem *i = listBoxSortedField->item( listBoxSortedField->currentItem() );
    listBoxSortedField->takeItem( i );
    listBoxSortedField->insertItem( i, index );
    listBoxSortedField->setCurrentItem( i );
}

void SqlFormWizard::sortFieldDown()
{
    if ( listBoxSortedField->currentItem() == -1 ||
            listBoxSortedField->currentItem() == ( int ) listBoxSortedField->count() - 1 ||
            listBoxSortedField->count() < 2 )
        return ;
    int index = listBoxSortedField->currentItem() + 1;
    QListBoxItem *i = listBoxSortedField->item( listBoxSortedField->currentItem() );
    listBoxSortedField->takeItem( i );
    listBoxSortedField->insertItem( i, index );
    listBoxSortedField->setCurrentItem( i );
}

void SqlFormWizard::enableAllSortPage( bool b )
{
    listBoxSortField->setEnabled( b );
    listBoxSortedField->setEnabled( b );
    buttonAddSortField->setEnabled( b );
    buttonRemoveSortField->setEnabled( b );
    buttonUpSortField->setEnabled( b );
    buttonDownSortField->setEnabled( b );
    buttonReSortField->setEnabled( b );
}

/****************misc methods****************/

void SqlFormWizard::nextPageClicked()
{
    if ( currentPage() == tablePage )
    {
        setupTablePage();
        enableAllTablePage( true );
    }
    else if ( currentPage() == fieldPage )
    {
        setupFieldPage();
        enableAllFieldPage( true );
    }
    else if ( currentPage() == sortPage )
    {
        setupSortPage();
        enableAllSortPage( true );
    }
}

void SqlFormWizard::setupFirstPage()
{
    setNextEnabled( connectionPage, false );
    setNextEnabled( relationPage, false );
    setNextEnabled( tablePage, false );
    //     setNextEnabled( fieldPage, false );

    if ( !m_parentTable )
        setupConnectionPage();
    else
        setupRelationPage();
}

void SqlFormWizard::setupConnectionPage()
{
    listBoxConnection->clear();
    comboDriver->insertStringList( QSqlDatabase::drivers() );
    listBoxConnection->insertStringList( m_project->databaseConnectionList() );

    if ( m_configure )
        listBoxConnection->setCurrentItem( listBoxConnection->findItem( m_list->connection() ) );
    else if ( listBoxConnection->count() )
        listBoxConnection->setCurrentItem( 0 );
}

void SqlFormWizard::setupRelationPage()
{
    listBoxParentKey->clear();
    removePage( connectionPage );
    setAppropriate( relationPage, true );

    QStringList lst = m_project->databaseFieldList( m_parentTable->connection(), m_parentTable->tableName() );
    listBoxParentKey->insertStringList( lst );
    m_list->setConnection( m_parentTable->connection() );

    if ( m_configure )
    {
        listBoxParentKey->setCurrentItem( listBoxParentKey->findItem( m_list->parentKey() ) );

        comboChildTable->setCurrentText( m_list->tableName() );
        childTableActivated( m_list->tableName() );

        comboChildKey->setCurrentText( m_list->foreignKey() );
        childKeyActivated( m_list->foreignKey() );

        switch ( m_list->relationToParent() )
        {
        case DataTable::OneToOne:
            comboChildRelation->setCurrentText( i18n("one-to-one") );
            childRelationActivated( "one-to-one" );
            break;
        case DataTable::OneToMany:
            comboChildRelation->setCurrentText( i18n("one-to-many") );
            childRelationActivated( "one-to-many"  );
            break;
        case DataTable::ManyToOne:
            comboChildRelation->setCurrentText( i18n("many-to-one") );
            childRelationActivated( "many-to-one" );
            break;
        case DataTable::ManyToMany:
            comboChildRelation->setCurrentText( i18n("many-to-many") );
            childRelationActivated( "many-to-many" );
            break;
        default:
            break;
        }
    }
}

void SqlFormWizard::setupTablePage()
{
    listBoxTable->clear();
    DatabaseConnection *conn = m_project->databaseConnection( m_list->connection() );

    listBoxTable->blockSignals( true );
    if ( !m_parentTable )
        listBoxTable->insertStringList( conn->tables() );
    else
        listBoxTable->insertItem( m_list->tableName() );
    listBoxTable->blockSignals( false );

    if ( m_configure || m_parentTable )
    {
        editTable->blockSignals( true );
        listBoxTable->blockSignals( true );

        editTable->setText( m_list->name() );
        listBoxTable->setCurrentItem( listBoxTable->findItem( m_list->tableName() ) );

        editTable->blockSignals( false );
        listBoxTable->blockSignals( false );

        buttonIcon->setPixmap( SmallIcon( m_list->iconName(), 32 ) );
        setNextEnabled( tablePage, true );
    }
}

void SqlFormWizard::setupFieldPage()
{
    listBoxField->clear();
    DatabaseConnection *conn = m_project->databaseConnection( m_list->connection() );

    connect( comboTargetTable, SIGNAL( activated( int ) ), this, SLOT( targetTableActivated() ) );
    connect( comboTargetKey, SIGNAL( activated( int ) ), this, SLOT( targetKeyActivated() ) );
    connect( comboTargetField, SIGNAL( activated( int ) ), this, SLOT( targetFieldActivated() ) );

    if ( m_configure )
    {
        DataFieldList fields = m_list->fieldList();
        for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
        {
            if ( ( *it ) ->primary() )
                listBoxField->insertItem( new QListBoxPixmap( UserIcon( "primarykey" ), ( *it ) ->name() ), ( *it ) ->number() );
            else if ( ( *it ) ->foreign() )
                listBoxField->insertItem( new QListBoxPixmap( UserIcon( "foreignkey" ), ( *it ) ->name() ), ( *it ) ->number() );
            else if ( ( *it ) ->hidden() )
                listBoxField->insertItem( new QListBoxPixmap( UserIcon( "hidden" ), ( *it ) ->name() ), ( *it ) ->number() );
            else if ( !( *it ) ->isVirtual() )
                listBoxField->insertItem( new QListBoxPixmap( UserIcon( "visible" ), ( *it ) ->name() ), ( *it ) ->number() );
        }
        return ;
    }

    QStringList lst = conn->fields( m_list->tableName() );

    conn->open( true );
    QSqlIndex pIdx = conn->connection() ->primaryIndex( m_list->tableName() );
    for ( uint i = 0; i < pIdx.count(); i++ )
    {
        QString fieldName = pIdx.field( i ) ->name();
        QVariant::Type fieldType = pIdx.field( i ) ->type();
        bool isRequired = conn->getFieldIsRequired( m_list->tableName(), fieldName );

        m_list->addDataField(
            fieldName,                      //name
            fieldType,                      //type
            isRequired,                     //required
            false,                          //calculated
            false,                          //virtual
            QString::null,                  //equation
            QString::null,                  //label
            -1,                             //number
            -1,                             //width
            false,                          //foreign
            true,                           //primary
            !isRequired,                    //hidden
            false,                          //resizable
            false                           //sortable
        );
        listBoxField->insertItem( new QListBoxPixmap( UserIcon( "primarykey" ), fieldName ) );
        lst.remove( pIdx.field( i ) ->name() );
    }
    for ( QStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
    {
        QString fieldName = ( *it );

        QVariant::Type fieldType = conn->getFieldType( m_list->tableName(), fieldName );
        bool isRequired = conn->getFieldIsRequired( m_list->tableName(), fieldName );

        m_list->addDataField(
            fieldName,                      //name
            fieldType,                      //type
            isRequired,                     //required
            false,                          //calculated
            false,                          //virtual
            QString::null,                  //equation
            QString::null,                  //label
            -1,                             //number
            -1,                             //width
            false,                          //foreign
            false,                          //primary
            !isRequired,                    //hidden
            true,                           //resizable
            true                            //sortable
        );
        if ( isRequired )
            listBoxField->insertItem( new QListBoxPixmap( UserIcon( "visible" ), fieldName ) );
        else
            listBoxField->insertItem( new QListBoxPixmap( UserIcon( "hidden" ), fieldName ) );
    }
    sortDataFields();
    conn->close();
}

void SqlFormWizard::setupSortPage()
{
    listBoxSortField->clear();
    listBoxSortedField->clear();

    DataFieldList fields = m_list ->fieldList();
    DataFieldList::Iterator it = fields.begin();
    for ( ; it != fields.end(); ++it )
    {
        if ( !( *it ) ->isVirtual() )
            listBoxSortField->insertItem( ( *it ) ->name() );
    }

    if ( m_configure )
    {
        listBoxSortedField->insertStringList( m_list->sort() );
    }
}

DataField *SqlFormWizard::findField( QListBoxItem * i )
{
    if ( !i )
        return 0;

    DataFieldList fields = m_list->fieldList();
    for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
    {
        if ( ( *it ) ->name() == i->text() )
            return * it;
    }

    return 0;
}

void SqlFormWizard::accept()
{
    m_list->dataTableEdit() ->setReadOnly( checkBoxReadOnly->isChecked() );
    m_list->dataTableEdit() ->setConfirmInsert( checkBoxConfirmInserts->isChecked() );
    m_list->dataTableEdit() ->setConfirmUpdate( checkBoxConfirmUpdates->isChecked() );
    m_list->dataTableEdit() ->setConfirmDelete( checkBoxConfirmDeletes->isChecked() );
    m_list->dataTableEdit() ->setConfirmCancels( checkBoxConfirmCancels->isChecked() );
    m_list->dataTableView() ->setSorting( checkBoxSorting->isChecked() );

    QStringList sort;
    QListBoxItem *item = listBoxSortedField->item( 0 );
    while ( item )
    {
        sort.append( item->text() );
        item = item->next();
    }
    m_list->setSort( sort );

    if ( !m_configure )
    {
        m_list->initialize();
        m_project->addDataTable( m_list );
    }
    else
    {
        m_list->initializeFields( true );
        m_list->initializeEditorFields();
    }
    SqlFormWizardBase::accept();
}

#include "sqlformwizardimpl.moc"
