#!/usr/bin/env python


################################################################################
# Chris's Lame Filebrowser 4 (The file mover dialog)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
 #   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##################################################################

import os, os.path, sys, shutil
import gtk, pygtk, pango
import thread, commands, string


class move:
	def __init__(self, move_type, x, y, moving_message, start_location, end_location):
		'''This is the file mover dialog. It copies a file from one location to another.
		Pass (move_type, x, y, moving_message,  start_location, end_location), where the
		move_type is "copy" or "move", x and y are the location you want the dialog to appear, 
		the moving message is the message to display (moving file to ...), the start location 
		is where the file is now, the end location is where it is to be moved to i.e.
		('/home/be/text.txt', '/mnt/fedora/') '''

		# Create the main window and title it
		self.progress_window= gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.progress_window.connect('destroy', self.kill)
		self.progress_window.set_title("Chris's Lame  File Mover")

		# Place the progress window at the desired location
		self.progress_window.move(x, y)     

		# Create a box for the dialog (in case I want to add more stuff later)
		self.main_table = gtk.Table(1, 2, gtk.FALSE)
		self.main_table.show()
		self.progress_window.add(self.main_table)

		# Create a Label for the Dialog (to hold the file name)
		self.mainLabel = gtk.Label(moving_message)
		self.mainLabel.modify_font(pango.FontDescription("sans bold 12"))
		self.main_table.attach(self.mainLabel, 0,1,0,1)

		# Create the progress bar and add it to the box
		self.progress_bar = gtk.ProgressBar()
		self.progress_bar.show()
		self.main_table.attach(self.progress_bar, 0,1,1,2)

		# Set the window size and show all
		self.progress_window.resize(self.progress_window.allocation.width, 50)
		self.progress_window.show_all()

		# Find out if this is a directory or a file 
		if os.path.isdir(start_location):
			self.__moveDirectory__(move_type, start_location, end_location, moving_message)
		else:
			# Start the file moving
			self.__moveFile__(move_type, start_location, end_location)
		self.kill()

	###################################################################
	# This function opens the file and moves it piece by piece, showing the progress
	# in the progress bar
	###################################################################
	def __moveFile__(self, move_type, start_location, end_location):

		# Resize the window so the name fits
		width, height = self.progress_bar.size_request()
		self.progress_window.set_size_request(width, 25)	

		# Get the size of the file to move
		filesize = os.path.getsize(start_location)

		# If the file size is zero
		if filesize == 0:
			return "file_is_empty"
	
		if move_type == "copy":	
			thread.start_new_thread(shutil.copy, (start_location, end_location))
		else:
			thread.start_new_thread(shutil.move, (start_location, end_location + "/" + os.path.basename(start_location)))

		while not os.path.exists(end_location + "/" + os.path.basename(start_location)):
			pass

		while filesize > os.path.getsize(end_location + "/" + os.path.basename(start_location)):
			new_filesize = float(os.path.getsize(end_location + "/" + os.path.basename(start_location)))
			adjustment =  float(new_filesize/filesize)
			
			# This function allows the gui to update in case the machine is too busy
			self.__updategui__()
				
			# Set the progress bar progress amount	
 			self.progress_bar.set_fraction(adjustment)	
 		
			if new_filesize  < 1000:
				self.progress_bar.set_text(str(int(new_filesize)) +" bytes of " + str(filesize) + "bytes")
			else:
				self.progress_bar.set_text(str(int(new_filesize/1024)) +" Kbs of " + str(filesize/1024) + "Kbs")


	###################################################################
	# This function moves the contents of directories
	###################################################################
	def __moveDirectory__(self, move_type, start_location, end_location, moving_message):

		# Resize the window so the name fits
		width, height = self.progress_bar.size_request()
		self.progress_window.set_size_request(width, 25)	

		# If the directory is empty, then just move it and return
		if len(os.listdir(start_location)) == 0:
			if move_type =="copy":
				shutil.copytree(start_location, end_location + "/" + os.path.basename(start_location))
			else:
				shutil.move(start_location, end_location + "/" + os.path.basename(start_location))
			self.kill()
			return 0

		# Get the original file size 
		status, tree_size = commands.getstatusoutput("du -bxs " + self.__returnClean__(start_location))
		tree_size = float(string.split(tree_size)[0])

		# Start the move or copy (threaded, of course)
		if move_type == "copy":
			try:
				thread.start_new_thread(shutil.copytree, (start_location, end_location + "/" + os.path.basename(start_location)))	
				return 0
			except:
				print "ERROR: COPY FAILED"
				return "cannot_copy"				
		elif move_type == "move":
			try:
				thread.start_new_thread(shutil.move, (start_location, end_location + "/" + os.path.basename(start_location)))
			except:
				print "ERROR: MOVE FAILED"
				return "cannot_move"

		while not os.path.exists(end_location + "/" + os.path.basename(start_location)):
			pass

		# Use the new and old directory tree sizes to calculate the progress bar
		new_tree_size = 0
		while tree_size > new_tree_size and os.path.exists(start_location):
			new_tree_size = commands.getoutput("du -bs " + self.__returnClean__(end_location + "/" + os.path.basename(start_location)))
			new_tree_size = float(string.split(new_tree_size)[0])		
			adjustment = float(new_tree_size/tree_size)

			# This function allows the gui to update in case the machine is too busy
			self.__updategui__()

			# Set the progress bar progress amount	
 			self.progress_bar.set_fraction(adjustment)

			if new_tree_size  < 1000:
				self.progress_bar.set_text(str(int(new_tree_size)) +" bytes of " + str(int(tree_size)) + "bytes")
			else:
				self.progress_bar.set_text(str(int(new_tree_size/1024)) +" Kbs of " + str(int(tree_size/1024)) + "Kbs")


	###################################################################
	# This function makes all current processes end before allowing it to continue
	# This allows the GUI to update even when the processor is peaked
	###################################################################
	def __updategui__(self):
		while  gtk.events_pending() :
			gtk.main_iteration()
		return


	###################################################################
	# This function destroys the widget when we are finished
	###################################################################
	def kill(self, button=None):
		self.progress_window.destroy()


	########################################################################
	# This function returns BASH friendly text
	########################################################################		
	def __returnClean__(self, text):
	
		# Here are the bad characters
		badChars = [" ","\'","&","(",")"]
		
		# Here are the replacement characters
		goodChars=["\ ","\\'","\&","\(","\)"]
		
		# Replace the bad with the good
		for x in range(len(badChars)):
			text = string.replace(text, badChars[x], goodChars[x])		
		
		# Return the "cleaned", BASH friendly text
		return text

