#!/usr/bin/env python

########################################################################
# Chris's Lame Filebrowser 4 (New Page and Icon Layout)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################################################

import gtk, pygtk, pango
import os, os.path, shutil, string, sys
import xml_reader, layout, listview, menu, mimetype
from xml.dom import minidom


IMAGE_DIRECTORY = os.path.abspath(string.replace(sys.argv[0],"claw4.py","") + "Images")

class create:
	def __init__(self, Notebook, location):
		'''This class creates the contents for a new tab in the main notebook.
		It handles common functions of the detail's view and the icon view as
		well as tab specific callbacks'''

		# This is where we attempt to determine where we shall open to
		try:
			os.chdir(location)	
		except:
			os.chdir(os.environ['HOME'])
			location = os.environ['HOME']

		# Get the theme from the preferences.xml file
		xml_reader.read()
		preferences_file = minidom.parse(os.environ['HOME'] + "/.claw/preferences.xml")
		node = preferences_file.getElementsByTagName("theme")
		theme = node[0].childNodes[0].data

		# Create the pixmaps for all the mimetypes we have in the icon directory
		icons = os.listdir(IMAGE_DIRECTORY + "/Icons/" + theme)

		# make a global list to hold the mime pixmaps
		self.pixbuf_list = {}
		for icon in icons:
			if not os.path.isdir(IMAGE_DIRECTORY +"/Icons/" + theme + "/" + icon):
				new_icon = gtk.gdk.pixbuf_new_from_file(IMAGE_DIRECTORY +"/Icons/" + theme + "/" + icon)
				if theme == "gorilla":
					new_icon = new_icon.scale_simple(48, 48, gtk.gdk.INTERP_BILINEAR)
				self.pixbuf_list[icon] = new_icon
		# Set some common variables
		self.main_notebook = Notebook.main_notebook
		self.Notebook = Notebook
		self.file_descriptor = -1
		self.view = "icon"

		# We use one global tooltips list
		self.tool_tips = gtk.Tooltips()
		self.tool_tips.enable()

		# Set some default preferences
		self.play_audio_previews = True
		self.show_picture_preview = True
		self.inline_picture_preview = True
		self.font = pango.FontDescription("sans 11")
		self.font_color = gtk.gdk.Color(0, 0, 0, 0)
		self.image_max_size = 10000
		self.bg_color = gtk.gdk.Color(65000, 65000, 65000, 0)
		self.archive_command = "bzip2"
		self.theme = "default"
		self.single_click = True
		self.show_hidden = False

		# Create the tab for the notebook and set the tooltip
		name_label = gtk.Label(os.path.basename(os.path.realpath(location)))

		close_button = gtk.Button()
		close_image = gtk.Image()
		close_image.set_from_file(IMAGE_DIRECTORY + "/Toolbar/close.png")
		close_button.add(close_image)
		close_button.connect("clicked", self.__tabCloseClicked__, Notebook)
		self.tool_tips.set_tip(close_button, "Close This Tab")
		
		pack_box = gtk.HBox()
		pack_box.pack_start(name_label, padding=5)
		pack_box.pack_start(close_button)
		
		event_box = gtk.EventBox()
		event_box.add(pack_box)
		event_box.set_name(os.getcwd())
		event_box.show_all()

		event_box.set_events(gtk.gdk.ALL_EVENTS_MASK)
		event_box.drag_dest_set(gtk.DEST_DEFAULT_ALL, [("text/plain", 0, 80)], gtk.gdk.ACTION_COPY)
		event_box.connect("drag-motion", self.__dragToTab__, Notebook)
		event_box.connect("button-press-event",self. __tabClicked__, Notebook)
		self.tool_tips.set_tip(event_box, os.getcwd())

		# Create the scrolled window that will contain the detail or icon view widget
		self.scroll = gtk.ScrolledWindow()
		self.scroll.set_policy(gtk.POLICY_AUTOMATIC,gtk.POLICY_AUTOMATIC)
		self.scroll.show()

		self.main_notebook.append_page(self.scroll, event_box)
		self.main_notebook.set_current_page(self.main_notebook.page_num(event_box))

		# Create the history for the layout
		self.history = []
		self.history.append(os.environ['HOME'])	
		
		# Set the preferences from the preferences file
		self.__setPreferences__()		
		
		# Finally, add either the detail (list) view or the icon view to the page
		if len(os.listdir(location)) > 500:
			self.child = listview.create(self)
			self.view = "list"
		else:
			self.child = layout.create(self)
			self.view = "icon"

	def __populate__(self, location):
		'''This function is simply a way of handling the _populate_ from the notebook which is called
		form the toolbar'''
		self.child.__populate__(location)

	############################ LOADING DIALOG ###############################


	def __loadingDialog__(self, event):
		'''This function handles a dialog while loading a directory'''

		# Create the dialog
		if event == True:
			self.window = gtk.Window(gtk.WINDOW_POPUP)
			self.label = gtk.Label("Loading " + os.path.basename(os.getcwd()))
			self.label.modify_font(pango.FontDescription("sans bold 20"))
			self.window.add(self.label)
			try:
				x,y = self.__getMousePosition__()
				self.window.move(x, y)
			except:
				self.window.set_position(gtk.WIN_POS_CENTER)
			self.window.show_all()

			while gtk.events_pending():
				gtk.main_iteration()

		elif event == False:
			self.window.destroy()
		else:
			self.label.set_text(event)


	############################ TAB CALLBACKS ###############################


	def __tabClicked__(self, event_box, event, Notebook):
		'''This button handles clicks on a tab'''

		# A double-click
		if event.button == 1 and event.type == gtk.gdk._2BUTTON_PRESS:
			if self.main_notebook.get_n_pages() > 1:
				Notebook.__destroyPage__(self)
		else:
			os.chdir(event_box.get_name())

	def __tabCloseClicked__(self, button, Notebook):
		'''This function is the callback for the close button on the tab'''
		if self.main_notebook.get_n_pages() > 1:
			Notebook.__destroyPage__(self)

	def changeTabLabel(self, location):
		event_box = self.main_notebook.get_tab_label(self.scroll)
		event_box.get_child().get_children()[0].set_text(os.path.basename(location))
		event_box.set_name(location)
		self.tool_tips.set_tip(event_box, location)


	def __dragToTab__(self, event_box, drag_context, x, y, timestamp, Notebook):
		'''This function handles switching tabs during a drag event'''
		
		num_pages =  self.main_notebook.get_n_pages()
		for page_num in range(num_pages):
			child = self.main_notebook.get_nth_page(page_num)
			if self.main_notebook.get_tab_label(child) == event_box:
				self.main_notebook.set_current_page(page_num)
				os.chdir(event_box.get_name())


	############################  ICONVIEW TO LISTVIEW ###############################

	def _toggleView_(self):
		'''This function, which is called by the notebook, destroys the
		existing view and replaces it with the other kind'''

		self.scroll.get_child().remove(self.scroll.get_child().get_child())
		self.child._destroy_()	
		del self.child

		if self.view == "icon":
			self.child = listview.create(self)
			self.view = "list"
		else:
			self.child = layout.create(self)
			self.view = "icon"


	############################  ICONVIEW TO LISTVIEW ###############################
	def setInfoBarText(self, text, style):
	
		self.Notebook.info_bar.set_text(text)
		if style == "ERROR":
			self.Notebook.info_bar.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(40000, 0, 0, 0))
		elif style == "INFO":
			self.Notebook.info_bar.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color(0, 40000, 0, 0))


	############################ FUNCTIONS ###############################
	
	def  __organizeContents__(self, directory):
		'''This function returns the contents of a directory in alphabetical
		order with the folders listed first. It returns a full path list as well.
		- directory is the address of the directory to list
		- show_hidden is either True or False'''
	
		if not os.path.isdir(directory):
			print "ERROR:", location, "is not a directory"
			return [],[]

		files = os.listdir(directory)
	
		# Filter out the hidden files and folders if desired
		if self.show_hidden == False:
			files =  filter(lambda x : x[0] != '.', files)

		# Sort the list alphabetically
		files.sort(lambda x,y : cmp(string.lower(x), string.lower(y)))
	
		# Filter out the folders
		folders =  filter(lambda file: os.path.isdir(directory + "/" + file), files)	
	
		# Place the folders at the beginning of the list in the same order they were removed
		for z in range(len(folders)):	
			files.remove(folders[z])
			files.insert(z, folders[z])

		if directory != "/" and not directory.endswith("/"):
			locations = [directory + "/" + file for file in files]
		else:
			locations = [directory + file for file in files]
		return locations

	def __getPixbuf__(self, location, size, mime_type=None):
		'''This function creates an icon and label and returns them'''

		if mime_type == None:
			mime_type = mimetype.getTypes(None, location)[location]

		image_types = ["x-xpixmap","x-ms-bmp","bmp","gif", "jpeg", "png", "svg"] 

		# Get the mime type for the location
		catagory = mime_type[0]
		mime = mime_type[1]
	
		# Create the pixbufs (but not image previews)
		image_pic = mime + ".png"
		if image_pic in self.pixbuf_list:
			pixbuf = self.pixbuf_list[image_pic]

		elif catagory == "image" and mime in image_types:
				image_pic = "image.png"
				pixbuf = self.pixbuf_list[image_pic]
		else:
			image_pic = "plain.png"
			pixbuf = self.pixbuf_list[image_pic]

		if size == "small":
			pixbuf = pixbuf.scale_simple(22, 22, gtk.gdk.INTERP_HYPER)

		return pixbuf

	def _deleteFile_(self, location):
		'''This function deletes a file or directory passed to it'''

		if os.path.islink(location):
			os.path.unlink(location)
		elif os.path.isdir(location):
			shutil.rmtree(location)
		else:
			os.remove(location)

	def __setPreferences__(self):
		'''The function handles setting the preferences'''
		preferences = xml_reader.read()
		xml_reader.set(self, preferences)

		if self.view == "icon":
			try:
				self.child.main_layout.modify_bg(gtk.STATE_NORMAL, self.bg_color)
				self.child.main_layout.modify_bg(gtk.STATE_ACTIVE, self.bg_color)
			except:
				pass
	
		self.scroll.modify_bg(gtk.STATE_ACTIVE, self.bg_color)
		self.scroll.modify_bg(gtk.STATE_NORMAL, self.bg_color)

	def __getMousePosition__(self):
		'''This functions returns the current mouse location
		relative to the upper left corner of the main window.'''
		
		main_window = self.main_notebook.get_toplevel()
		x, y = main_window.get_position()
		offset_x ,offset_y = main_window.get_pointer()
		return x + offset_x, y+offset_y


def __returnClean__(text):
	'''This function returns Bash script command line friendly text'''	
					
	# Here are the bad characters
	badChars = [" ","\'","&","(",")"]
			
	# Here are the replacement characters
	goodChars=["\ ","\\'","\&","\(","\)"]
			
	# Replace the bad with the good
	for x in range(len(badChars)):
		text = string.replace(text, badChars[x], goodChars[x])		
		
	# Return the "cleaned", BASH friendly text
	return text
		

	
