/* eucl_math.c    euclidean geometry utility routines 
Ken Stephenson, 8/91 */

/* cc eucl_math.c -o eucl_math.o -c -g -lm */

#include "cp_types.h"
extern int s_pt_to_vec();

double e_area(double r1,double r2,double r3)
     /* return area of triangle given radii */
{
  double s;
	
  s=(r1+r2+r3);
  return (sqrt(s*(s-r1-r2)*(s-r1-r3)*(s-r2-r3)));
} /* e_area */

int e_compcenter(complex z1,complex z2,complex *z3,
		 double e1,double e2,double *e3,
		 double o1,double o2,double o3)
     /* given centers/radii of circles 1 and 2 and rad of 3, and 
cosines oj opposite zj, find eucl center of 3. (e3 is ptr only 
for compat reasons.) */ 
{
  double l1,l2,l3,s,cosang,sq23;
  complex v,w;
  extern double cAbs();
	
  l3=e1*e1+e2*e2+2*e1*e2*o3;
  l2=e1*e1+(*e3)*(*e3)+2*e1*(*e3)*o2;
  l1=e2*e2+(*e3)*(*e3)+2*e2*(*e3)*o1;
  if ((sq23=2.0*sqrt(l2*l3))<okerr) return 0;
  cosang=(l2+l3-l1)/sq23;
  v.re=sqrt(l2)*cosang;
  v.im=sqrt(l2-v.re*v.re);
  w.re=z2.re-z1.re;
  w.im=z2.im-z1.im;
  if ((s=cAbs(w))<okerr) return 0;
  w.re = w.re/s; w.im = w.im/s;
  z3->re=z1.re+w.re*v.re-w.im*v.im;
  z3->im=z1.im+w.re*v.im+w.im*v.re;
  return 1;
} /* e_compcenter */
	
double e_cos_overlap(double e1,double e2,double e3,
		     double t1,double t2,double t3,int *flag)
     /* given three eucl radii, cos's of opp overlap angles, return 
cos of angle at e1. If radii and overlaps incompatible, return 
value (for 0 or Pi, depending) and set flag. Note: tj <= 1 for 
overlap, tj > 1 for inv distance. */
{
  double l1,l2,l3,sqrlen;

  l3=e1*e1+e2*e2+2*e1*e2*t3; 
  l2=e1*e1+e3*e3+2*e1*e3*t2;
  l1=e2*e2+e3*e3+2*e2*e3*t1;
  sqrlen=(l2+l3-l1)/(2*sqrt(l2*l3));
  if (sqrlen < -1.000001){(*flag)++; return -1.0;} /* error? */
  if (sqrlen < -1.0) return -1.0; /* roundoff error? */
  if (sqrlen > 1.000001){(*flag)++; return 1.0;} /* error? */ 
  if (sqrlen > 1.0) return 1.0; /* roundoff error? */
  return sqrlen;
} /* e_cos_overlap */

double R3_dist(complex z,complex w)
     /* distance in R^3 between sph. pts z and w */
{
  double Z[3],W[3];
  
  s_pt_to_vec(z,Z);
  s_pt_to_vec(w,W);
  
  Z[0]=Z[0]-W[0];
  Z[1]=Z[1]-W[1];
  Z[2]=Z[2]-W[2];
  
  return (vec_norm(Z));
} /* R3_dist */

double e_cos_overlap_special(double e1,double e2,double e3,
			     double t1,double t2,double t3,
			     int *flag,double *incompat_err)
     /* given three eucl radii, cos's of opp overlap angles, return 
cos of angle at e1. If radii and overlaps incompatible, return 
value (for 0 or Pi, depending) and set flag. Note: tj <= 1 for 
overlap, tj > 1 for inv distance. */
{
  double l1,l2,l3,sqrlen;

  l3=e1*e1+e2*e2+2*e1*e2*t3; 
  l2=e1*e1+e3*e3+2*e1*e3*t2;
  l1=e2*e2+e3*e3+2*e2*e3*t1;
  sqrlen=(l2+l3-l1)/(2*sqrt(l2*l3));
  if (sqrlen < -1.000001)
    {
      (*flag)++;
      *incompat_err += fabs(sqrlen);
      return -1.0;
    } /* error? */
  if (sqrlen < -1.0) return -1.0; /* roundoff error? */
  if (sqrlen > 1.000001)
    {
      (*flag)++;
      *incompat_err += sqrlen;
      return 1.0;
    } /* error? */ 
  if (sqrlen > 1.0) return 1.0; /* roundoff error? */
  return sqrlen;
} /* e_cos_overlap_special */

double Fx(double x,double y,double z)
     /* deriv in eucl setting for angle at x w.r.t radius x.*/
{
  double a,b;

  a=x*(x+y+z); b=y*z;
  return ( (-b)*(2*x+y+z)/((a+b)*sqrt(a*b)) );
}	 
	
double Fy(double x,double y,double z)
     /* deriv in eucl setting for angle at x w.r.t radius y.*/
{
  double a,b;

  a=x*(x+y+z); b=y*z;
  return (x*y*(x+y)/((a+b)*sqrt(a*b)));
}

double Fz(double x,double y,double z)
     /* deriv in eucl setting for angle at x w.r.t radius y.*/
{
  return (Fy(x,z,y));
}	 
	
double inv_dist(complex z1,double r1,complex z2,double r2)
     /* compute inversive distance between two eucl circles. Return 
negative value if overlap more than PI/2. Return default 1.0 if 
situation not yet covered. */
{
  double ab;

  if (r1>0 && r2>0)
    {
      ab=cAbs(csub(z1,z2));
      return ((ab*ab-(r1*r1+r2*r2))/(2.0*r1*r2));
    }
  /* need to handle: one plane, r=0, or one negative r */
  return 1.0;
} /* inv_dist */
