/**
  \struct CVideoOptions
  \brief Options per video device
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tracer.h"

#include "CamStreamApp.h"
#include "VideoOptions.h"

TR_MODULE("VideoOptions");

static const char *option[][2] = {
   { "basename",     "snapshot" },
   { "textfont",     "Helvetica/10/100/F" },
   { "textcolor",    "#ffff00" },
   { "timeinimage",  "false" },
   { "fileformat",   "PNG" },
   { "savetodisk",   "true" },
   { "ftptoserver",  "false" },
//   { "saveoption",   "1" },
   { "maxsequence",  "1000" },
   { "sequence",     "0" },
   { "snapinterval", "30" },
   { "ftpserver",    "" },
   { "ftpport",      "21" },
   { "ftppath",      "" },
   { "ftpuser",      "" },
   { "ftppass",      "" },
   { "ftppassive",   "false" },
   { "ftpunique",    "true" },
   { "runcommand",   "false" },
   { "command",      "" },
   { "usermessage",  "" },
   { "audio_device", "" },
   { "audio_node",   "" },
   { "middlepart",   "1" },
   { NULL,           NULL }
};


CVideoOptions::CVideoOptions()
{
   TR_ENTER();
   DeclareVariables();
   TR_LEAVE();
}


// private

// protected

void CVideoOptions::DeclareVariables()
{
   int i;

   TR_ENTER();
   for (i = 0; option[i][0] != NULL; i++)
      DeclareVariable(option[i][0], option[i][1]);
   TR_LEAVE();
}

// public

void CVideoOptions::SetXML(const QDomNode &node)
{
  QDomElement e;

  COptionsXML::SetXML(node);

  e = node.cloneNode().toElement();
  if (!e.isNull())
  {
    m_Name = e.attribute("name");
    m_Node = e.attribute("node");
  }
}

void CVideoOptions::GetXML(QDomNode &node) const
{
  QDomElement e;

  COptionsXML::GetXML(node);

  e = node.toElement();
  if (!e.isNull())
  {
    e.setAttribute("name", m_Name);
    e.setAttribute("node", m_Node);
  }
}


QString CVideoOptions::GetDeviceName() const
{
   return m_Name;
}

void CVideoOptions::SetDeviceName(const QString &name)
{
   m_Name = name;
}

QString CVideoOptions::GetNodeName() const
{
   return m_Node;
}

void CVideoOptions::SetNodeName(const QString &node)
{
   m_Node = node;
}

QString CVideoOptions::GetAudioDeviceName() const
{
   return Get("audio_device");
}

void CVideoOptions::SetAudioDeviceName(const QString &name)
{
   Set("audio_device", name);
}

QString CVideoOptions::GetAudioNode() const
{
   return Get("audio_node");
}

void CVideoOptions::SetAudioNode(const QString &number)
{
   Set("audio_node", number);
}

QString CVideoOptions::GetBaseName() const
{
   return Get("basename");
}

void CVideoOptions::SetBaseName(const QString &basename)
{
   Set("basename", basename);
}

CVideoOptions::MiddlePartOption CVideoOptions::GetMiddlePart() const
{
  return (MiddlePartOption)GetInt("middlepart");
}

void CVideoOptions::SetMiddlePart(MiddlePartOption option)
{
   Set("middlepart", option);
}

QColor CVideoOptions::GetTextColor() const
{
   return QColor(Get("textcolor"));
}

void CVideoOptions::SetTextColor(const QColor &color)
{
   Set("textcolor", color.name());
}

void CVideoOptions::SetTextColor(const QString &color_name)
{
   Set("textcolor", color_name);
}

QFont CVideoOptions::GetTextFont() const
{
   QFont f("Helvetica", 12, QFont::Normal, false);
   QString fontname;
   QStringList FontProperties;

   fontname = Get("textfont");
   FontProperties = QStringList::split('/', fontname, true);
   if (FontProperties.count() >= 4)
   {
     f.setFamily(FontProperties[0]);
     f.setPointSize(FontProperties[1].toInt());
     f.setWeight(FontProperties[2].toInt());
     f.setItalic(FontProperties[3] == "T");
   }
   return f;
}

void CVideoOptions::SetTextFont(const QFont &font)
{
   QString fontname;

   // Assemble a string with the most important font properties
   fontname.sprintf("%s/%d/%d/%c", font.family().ascii(), font.pointSize(), font.weight(), font.italic() ? 'T' : 'F');
   Set("textfont", fontname);
}

bool CVideoOptions::GetTimeInImage() const
{
   return GetBool("timeinimage");
}

void CVideoOptions::SetTimeInImage(bool value)
{
   Set("timeinimage", value);
}

QString CVideoOptions::GetFileFormat() const
{
   return Get("fileformat");
}

void CVideoOptions::SetFileFormat(const QString &format)
{
   Set("fileformat", format);
}

/** Get current sequence number */
int CVideoOptions::GetSequence() const
{
   return GetInt("sequence");
}

void CVideoOptions::SetSequence(int value)
{
   Set("sequence", value);
}

int CVideoOptions::GetMaxSequence() const
{
   return GetInt("maxsequence");
}

void CVideoOptions::SetMaxSequence(int max)
{
   Set("maxsequence", max);
}

bool CVideoOptions::GetSaveToDisk() const
{
   return GetBool("savetodisk");
}

void CVideoOptions::SetSaveToDisk(bool value)
{
   Set("savetodisk", value);
}

int CVideoOptions::GetSnapInterval() const
{
   return GetInt("snapinterval");
}

void CVideoOptions::SetSnapInterval(int value)
{
   Set("snapinterval", value);
}

bool CVideoOptions::GetFTPToServer() const
{
   return GetBool("ftptoserver");
}

void CVideoOptions::SetFTPToServer(bool value)
{
   Set("ftptoserver", value);
}

int CVideoOptions::GetFTPPort() const
{
   return GetInt("ftpport");
}

void CVideoOptions::SetFTPPort(int port_number)
{
   Set("ftpport", port_number);
}


QString CVideoOptions::GetFTPServer() const
{
   return Get("ftpserver");
}

void CVideoOptions::SetFTPServer(const QString &server)
{
   Set("ftpserver", server);
}

QString CVideoOptions::GetFTPPath() const
{
   return Get("ftppath");
}

void CVideoOptions::SetFTPPath(const QString &path)
{
   Set("ftppath", path);
}

QString CVideoOptions::GetFTPUser() const
{
   return Get("ftpuser");
}

void CVideoOptions::SetFTPUser(const QString &user)
{
   Set("ftpuser", user);
}

QString CVideoOptions::GetFTPPass() const
{
   return Get("ftppass");
}

void CVideoOptions::SetFTPPass(const QString &pass)
{
   Set("ftppass", pass);
}

bool CVideoOptions::GetFTPPassive() const
{
   return GetBool("ftppassive");
}

void CVideoOptions::SetFTPPassive(bool passive)
{
   Set("ftppassive", passive);
}

bool CVideoOptions::GetFTPUnique() const
{
   return GetBool("ftpunique");

}

void CVideoOptions::SetFTPUnique(bool unique)
{
   Set("ftpunique", unique);
}

bool CVideoOptions::GetRunCommand() const
{
   return GetBool("runcommand");
}

void CVideoOptions::SetRunCommand(bool value)
{
   Set("runcommand", value);
}

QString CVideoOptions::GetCommand() const
{
   return Get("command");
}

void CVideoOptions::SetCommand(const QString &command)
{
   Set("command", command);
}

QString CVideoOptions::GetUserMessage() const
{
   return Get("usermessage");
}

void CVideoOptions::SetUserMessage(const QString &message)
{
   Set("usermessage", message);
}

