
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

//#define DEBUG

#include <math.h>

#include "bristol.h"
#include "bristolmidiapi.h"

extern int midiNoteOn();
extern int midiNoteOff();
extern bristolGlobalController(Baudio *, u_char, u_char, float);

int buildCurrentTable(Baudio *, float);

bristolVoice *
findVoice(bristolVoice *voice, int key, int chan)
{
	if (voice == (bristolVoice *) NULL)
		return(NULL);

//printf("findVoice(%i, %i, %i) [%x]\n", voice->index,key,chan,voice->baudio);

	if ((voice->key.key == key)
		&& (voice->baudio != (Baudio *) NULL)
		&& (voice->baudio->midichannel == chan))
		return(voice);

	return(findVoice(voice->next, key, chan));
}

Baudio *
findBristolAudioByChan(Baudio *baudio, int chan)
{
#ifdef DEBUG
	printf("findBristolAudioByChan(%x, %i)\n", baudio, chan);
#endif

	if (baudio == (Baudio *) NULL)
		return(NULL);

	if (baudio->midichannel == chan)
		return(baudio);

	return(findBristolAudioByChan(baudio->next, chan));
}

/*
 * This would probably be a lot faster as a while() loop.
 */
Baudio *
findBristolAudio(Baudio *baudio, int id, int channel)
{
#ifdef DEBUG
	printf("findBristolAudio(%x, %i, %i)\n", baudio, id, channel);
#endif

	if (baudio == (Baudio *) NULL)
		return(NULL);

//	if ((baudio->controlid == id)
//		&& (baudio->midichannel == channel))
	if (baudio->sid == id)
		return(baudio);

	return(findBristolAudio(baudio->next, id, channel));
}

static int midiPolyPressure(audioMain *audiomain, bristolMidiMsg *msg)
{
	bristolVoice *voice;

#ifdef DEBUG
	printf("midiPolyPressure(%i, %i)\n",
		msg->params.pressure.key, msg->params.pressure.pressure);
#endif

	/*
	 * Find this note, and adjust its pressure settings.
	 */
	if ((voice = findVoice(audiomain->playlist, msg->params.pressure.key,
		msg->channel)) != NULL)
	{
		voice->pressure.pressure = msg->params.pressure.pressure;
		voice->press = ((float) msg->params.pressure.pressure) / 127;
		return;
	}
}

static int midiControl(audioMain *audiomain, bristolMidiMsg *msg)
{
	int c_id;
	float c_val;
	Baudio *baudio = audiomain->audiolist;

	c_id = msg->params.controller.c_id;
	c_val = (float) msg->params.controller.c_val;

#ifdef DEBUG
	printf("midiControl(%i, %f)\n", c_id, c_val);
#endif

	if ((msg->command >= MIDI_ALL_NOTES_OFF)
		&& (msg->command <= MIDI_POLY_OFF))
	{
		/*
		 * For now just turn off ALL notes. This should arguably be done on
		 * the specified channel, but we are effectively only one multitimbral
		 * synth.
		 */
		allNotesOff(audiomain, msg->channel);
		return;
	}

	if ((c_id >= MIDI_CONTROLLER_COUNT) || (c_id < 0))
		return;

	while ((baudio = findBristolAudioByChan(baudio, msg->channel)) != NULL)
	{
		baudio->contcontroller[c_id] = ((float) c_val) / 127;
		baudio = baudio->next;
	}
}

static int midiProgram(audioMain *audiomain, bristolMidiMsg *msg)
{
	int p_id = msg->params.program.p_id;

#ifdef DEBUG
	printf("midiProgram(%i)\n", p_id);
#endif

	/*
	 * Ah, the engine receives this message, probably from rawmidi. What to do
	 * next is not selfevident: the engine has no memories - these are
	 * all handled by the GUI. These (and other) messages should be passed
	 * through to the GUI, or the GUI should register for a midi pipe as well
	 * and do the other necessary things. I think I actually prefer the latter,
	 * since having a passthrough here would only be one special case.
	 */
}

static int midiChannelPressure(audioMain *audiomain, bristolMidiMsg *msg)
{
	bristolVoice *voice = audiomain->playlist;
	Baudio *baudio = audiomain->audiolist;

#ifdef DEBUG
	printf("midiChannelPressure()\n");
#endif
#warning - no channel pressure support yet
return;
	while (baudio != NULL)
	{
		if (baudio->midichannel == msg->channel)
		{
			baudio->chanPress.pressure = msg->params.channelpress.pressure;
			baudio->chanpressure = ((float) baudio->chanPress.pressure) / 127;
		}

		baudio = baudio->next;
	}

	while (voice->next != NULL)
	{
		voice->chanpressure = baudio->chanpressure;
		voice = voice->next;
	}
}

static void
doPitchWheel(Baudio *baudio)
{
	float note = 1.0;
	int i;

	if (baudio->midi_pitchfreq >= 0)
	{
		for (i = 0; i < baudio->midi_pitch; i++)
			note *= baudio->note_diff;

		baudio->midi_pitchfreq = 1.0 + (note - 1.0) * baudio->midi_pitchfreq;
	} else {
		for (i = 0; i < baudio->midi_pitch; i++)
			note /= baudio->note_diff;

		baudio->midi_pitchfreq = 1.0 - (note - 1.0) * baudio->midi_pitchfreq;
	}
}

/*
 * Correctly speaking we should calculate all the interpolative values. We can
 * do this by having a parameter for the spread of the pitchwheel, and taking
 * that octave fraction for any given value. Hm... FFS.
 */
static int midiPitchWheel(audioMain *audiomain, bristolMidiMsg *msg)
{
	float pitch;
	Baudio *baudio = audiomain->audiolist;

	pitch = (float) (msg->params.pitch.lsb + (msg->params.pitch.msb << 7));
	pitch = (pitch - 8192);
	if (pitch > 0)
		pitch += 128;

	baudio->pitchwheel = (pitch + 8192) / CONTROLLER_RANGE;

	while ((baudio = findBristolAudioByChan(baudio, msg->channel)) != NULL)
	{
		if (baudio->gtune == 0)
			baudio->gtune = 1.0;

		/*
		 * Put it in a range of -1 to 1.
		 */
		baudio->midi_pitchfreq = pitch / 8192;

#ifdef DEBUG
		printf("midiPitchWheel(%f)\n", pitch);
#endif

		doPitchWheel(baudio);

		buildCurrentTable(baudio, baudio->midi_pitchfreq * baudio->gtune);
		alterAllNotes(baudio);

		baudio = baudio->next;
	}
}

static int midiSystem(audioMain *audiomain, bristolMidiMsg *msg)
{
	float adjusted;
	Baudio *baudio;

	adjusted = ((float) (msg->params.bristol.valueLSB
		+ (msg->params.bristol.valueMSB << 7))) / (CONTROLLER_RANGE - 1);

	if (adjusted > 1.0)
		adjusted = 1.0;
	else if (adjusted < 0)
		adjusted = 0;

#ifdef DEBUG
	printf("midiSystem(%i, %i, %i, %i)\n",
		msg->params.bristol.SysID,
		msg->params.bristol.L,
		msg->params.bristol.a,
		msg->params.bristol.b);
#endif

	/*
	 * First take a peek to see if this is a "SLab" message.
	 */
	if (strncmp(&msg->params.bristol.SysID, "SLab", 4) == 0)
	{
//		bristolOPParams *params;

		/*
		 * If so we can look for index (should be zero for now) operator,
		 * controller and value.
		 */
#ifdef DEBUG
		printf("	SLab[%i] %i, %i, %i/%i, %f\n",
			msg->params.bristol.channel,
			msg->params.bristol.operator,
			msg->params.bristol.controller,
			msg->params.bristol.valueMSB,
			msg->params.bristol.valueLSB,
			adjusted);
#endif

		if (msg->params.bristol.operator == BRISTOL_SYSTEM)
		{
			bristolSystem(audiomain, msg);
			return;
		}

		if (audiomain->atStatus == -1)
			/*
			 * If we do not have an active audiothread, then we will only
			 * accept messages on operator 127 - the bristol system control
			 * operator channel
			 */
			return;

		/*
		 * Bristol Sysex messages do contain midi channel information, but 
		 * since we are doing system operation we need to make sure we take the
		 * single bristolAudio associated with this config interface, ie, we
		 * need to consider where the message came from, and look for its ID.
		 * The rest of the midi messages search for midichannel, and may get
		 * several hits when we are working with multitimbral configurations.
		 */
		baudio = findBristolAudio(audiomain->audiolist,
			msg->params.bristol.channel, 0);

		if (baudio == (Baudio *) NULL)
			return;

		if (msg->params.bristol.operator < baudio->soundCount) {
			/*
			 * Find out if this is a float val or what? Alternatively, find
			 * out which algo this is, and call the "param()" routing
			 * associated with it - let it sort out its parameter range!
			 *
			 * These are called with operator, parameter and local:
			 */
			audiomain->palette[baudio->sound[
				msg->params.bristol.operator]->index]->param(
					audiomain->palette[
					baudio->sound[msg->params.bristol.operator]->index],
					baudio->sound[msg->params.bristol.operator]->param,
					msg->params.bristol.controller,
					(float) adjusted);
		} else {
			/*
			 * Pass the event on to any global controller registered by
			 * this bristolSound
			 */
			if (baudio->param != NULL)
				baudio->param(baudio, msg->params.bristol.operator,
					msg->params.bristol.controller, adjusted);
		}
	}
}

/*
 * We have a default table which consists of the correct note steps for a 
 * tuned synth. We also have a global tuning, and this applies alterations into
 * the "currentTable". This table is used to direct the frequencies applied to
 * all the oscillators.
 */
int
buildCurrentTable(register Baudio *baudio, register float gtune)
{
	register int i;

	if (gtune == 0)
		return;

	for (i = 0; i < DEF_TAB_SIZE; i++)
		baudio->ctab[i] = defaultTable[i].defnote * gtune;

	/*
	 * We should also consider retuning any active notes at this point, but
	 * since this code should not be concerned with the actual synthesis 
	 * algorithm, it is left to the caller to make such changes. See the code
	 * for the audioEngine, where this routine is called.
	 */

#ifdef DEBUG
	printf("min 0(%.3f) key L 36(%.3f) key H 96(%.3f) max 127(%.3f) [%f]\n",
		baudio->ctab[0], baudio->ctab[36],
		baudio->ctab[96], baudio->ctab[127], gtune);
	printf("min 0(%.3f) key L 36(%.3f) key H 96(%.3f) max 127(%.3f) [%f]\n",
		defaultTable[0].defnote, defaultTable[36].defnote,
		defaultTable[96].defnote, defaultTable[127].defnote, gtune);
#endif
}

/*
 * Calculate a default frequency table for the 127 MIDI keys.
 *
 * This is the rate at which we should step through the 1024 samples of wave
 * table.
 *
 * Taken from the table:
 *
 * A  220.000 57 44=110 32=55 20=27.5 8=13.75
 * A# 233.082 58
 * B  246.942 59
 * C  261.626 60 = Midi Key, Middle C 
 * C# 277.183 61
 * D  293.665 62
 * D# 311.127 63
 * E  329.626 64
 * F  349.228 65
 * F# 369.994 66
 * G  391.995 67
 * G# 415.305 68
 *
 * We can work on putting full calculation into here for other frequency 
 * tables. For correct operation, each semitone is
 *	previous frequency * (2^^(1/12))
 * Since A is constand whole numbers we can calcuate each octave from A.
 */
int
initFrequencyTable(int rate)
{
	int i, j = 1, k;
	float gain_diff, accum = 1.0;

/*
 * This is a seriously wrong set of numbers, but is a better algo.
 *
	defaultTable[8] = 13.75;
	defaultTable[20] = 27.5;
	defaultTable[32] = 55.0;
	defaultTable[44] = 110.0;
	defaultTable[56] = 220.0;
	defaultTable[68] = 440.0;
	defaultTable[80] = 880.0;
	defaultTable[92] = 1760.0;
	defaultTable[104] = 1760.0;
	defaultTable[116] = 1760.0;
	defaultTable[128] = 1760.0;

	for (i = 8; i < DEF_TAB_SIZE; i += 12)
	{
		defaultTable[i] = j * 13.75;
		for (k = 2; k <= 12; k++)
		{
			defaultTable[i + k] = ((float) 1024.0)
				/ (((float) rate)
					/ (defaultTable[i + k - 1] * pow(2, ((float) k) / 12)));
		}
		j++;
	}
*/
	/*
	 * For any given frequency, we need given number of cycles per second, and
	 * this is equal to rate/f samples per cycle. We have 1024 samples, so we
	 * can use 1024/(rate/f) to get step rates.
	 */
	defaultTable[57].defnote = ((float) 1024.0) / (rate / 220.000);
	defaultTable[58].defnote = ((float) 1024.0) / (rate / 233.082);
	defaultTable[59].defnote = ((float) 1024.0) / (rate / 246.942);
	defaultTable[60].defnote = ((float) 1024.0) / (rate / 261.626);
	defaultTable[61].defnote = ((float) 1024.0) / (rate / 277.183);
	defaultTable[62].defnote = ((float) 1024.0) / (rate / 293.665);
	defaultTable[63].defnote = ((float) 1024.0) / (rate / 311.127);
	defaultTable[64].defnote = ((float) 1024.0) / (rate / 329.626);
	defaultTable[65].defnote = ((float) 1024.0) / (rate / 349.228);
	defaultTable[66].defnote = ((float) 1024.0) / (rate / 369.994);
	defaultTable[67].defnote = ((float) 1024.0) / (rate / 391.995);
	defaultTable[68].defnote = ((float) 1024.0) / (rate / 415.305);

	/*
	 * The lower keys
	 */
	for (i = 56; i >= 0; i--)
		defaultTable[i].defnote = defaultTable[i + 12].defnote / 2;

	/*
	 * The higher keys
	 */
	for (i = 69; i < DEF_TAB_SIZE; i++)
		defaultTable[i].defnote = defaultTable[i - 12].defnote * 2;

	/*
	 * We now have to build in a gain table, which will be logarithmic,
	 * and a multipication list for attack/decay rates.
	 */
	gain_diff = pow((double) 13, ((double) 1)/DEF_TAB_SIZE);

	for (i = 1; i <= DEF_TAB_SIZE; i++)
	{
		defaultTable[i - 1].gain = (accum *= gain_diff);
	}
	gain_diff = pow((double) CONTROLLER_RANGE, ((double) 1)/CONTROLLER_RANGE);
	accum = 1.0;
	gainTable[0].gain = (accum *= gain_diff) / CONTROLLER_RANGE;
	for (i = 1; i < CONTROLLER_RANGE; i++)
	{
		gainTable[i].gain = (accum *= gain_diff) / CONTROLLER_RANGE -
			gainTable[0].gain;
	}
	gainTable[0].gain = 0;
	gainTable[CONTROLLER_RANGE - 1].gain = 
		gainTable[CONTROLLER_RANGE - 2].gain;

	gain_diff = pow((double) 2, ((double) 1)/CONTROLLER_RANGE);
	accum = 1.0;
	for (i = 0; i < CONTROLLER_RANGE; i++)
		gainTable[i].rate = pow((double) 13, ((double) 1)/(i * 20));
}

int
initMidiRoutines(audioMain *audiomain, midiHandler midiRoutines[])
{
#ifdef DEBUG
	printf("initMidiRoutines()\n");
#endif

	midiRoutines[0] = midiNoteOff;
	midiRoutines[1] = midiNoteOn;
	midiRoutines[2] = midiPolyPressure;
	midiRoutines[3] = midiControl;
	midiRoutines[4] = midiProgram;
	midiRoutines[5] = midiChannelPressure;
	midiRoutines[6] = midiPitchWheel;
	midiRoutines[7] = midiSystem;

	initFrequencyTable(audiomain->samplerate);
}

