/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "xslfofootnote.h"
#include "usfmtools.h"
#include "constants.h"
#include "gwrappers.h"


XslFoFootnote::XslFoFootnote (const Usfm& usfm, bool show)
// Stores the properties for all the footnote related styles.
// If "show" is true, then printing of the footnotes is done when the 
// "print" property of the \f style is true.
{
  // Store and initialize variables.
  myshow = show;
  NoteNumberingType note_numbering_type = nntNumerical;
  note_numbering_restart = nnrtPage;
  ustring note_numbering_user_sequence;
  bool spacious = false;
  standardparagraph = NULL;
  extraparagraph = NULL;
  
  // Go through all the styles.
  for (unsigned int i = 0; i < usfm.styles.size(); i++) {
    if (usfm.styles[i].type == stFootEndNote) {
      // Check subtype.
      FootEndNoteType footnotetype = (FootEndNoteType) usfm.styles[i].subtype;
      switch (footnotetype) {
        case fentFootnote:
        {
          // Store data for the markers and the anchor.
          opening_marker = usfm_get_full_opening_marker (usfm.styles[i].marker);
          closing_marker = usfm_get_full_closing_marker (usfm.styles[i].marker);
          anchor_fontpercentage = usfm.styles[i].fontpercentage;
          anchor_italic = usfm.styles[i].italic;
          anchor_bold = usfm.styles[i].bold;
          anchor_underline = usfm.styles[i].underline;
          anchor_smallcaps = usfm.styles[i].smallcaps;
          anchor_superscript = usfm.styles[i].superscript;
          note_numbering_type = (NoteNumberingType) usfm.styles[i].userint1;
          note_numbering_restart = (NoteNumberingRestartType) usfm.styles[i].userint2;
          note_numbering_user_sequence = usfm.styles[i].userstring1;
          spacious = usfm.styles[i].userbool2;
          if (myshow) myshow = usfm.styles[i].print;
          break;
        }
        case fentEndnote:
        {
          // Endnotes not dealt with here.
          break;
        }
        case fentStandardContent:
        {
          // Standard content.
          if (!standardparagraph)
            standardparagraph = new XslFoFootnoteParagraph (
              usfm.styles[i].marker, usfm.styles[i].fontsize, 
              usfm.styles[i].italic, usfm.styles[i].bold, usfm.styles[i].underline, usfm.styles[i].smallcaps,
              usfm.styles[i].justification, 
              usfm.styles[i].spacebefore, usfm.styles[i].spaceafter, 
              usfm.styles[i].leftmargin, usfm.styles[i].rightmargin, usfm.styles[i].firstlineindent, 
              usfm.styles[i].userbool1);
          footnote_markers.insert (usfm.styles[i].marker);
          break;
        }
        case fentContent:
        case fentContentWithEndmarker:
        {
          // Store data for the footnote body.
          content_marker.push_back (usfm_get_full_opening_marker (usfm.styles[i].marker));
          content_apocrypha.push_back (usfm.styles[i].userbool1);
          footnote_markers.insert (usfm.styles[i].marker);
          break;
        }
        case fentParagraph:
        {
          // Store relevant data for a paragraph marker.
          if (!extraparagraph)
            extraparagraph = new XslFoFootnoteParagraph (
              usfm.styles[i].marker, usfm.styles[i].fontsize, 
              usfm.styles[i].italic, usfm.styles[i].bold, usfm.styles[i].underline, usfm.styles[i].smallcaps,
              usfm.styles[i].justification, 
              usfm.styles[i].spacebefore, usfm.styles[i].spaceafter, 
              usfm.styles[i].leftmargin, usfm.styles[i].rightmargin, usfm.styles[i].firstlineindent, 
              usfm.styles[i].userbool1);
          footnote_markers.insert (usfm.styles[i].marker);
          break;
        }
      }
    }
  }
  // Ensure that both of the paragraph styles are there.
  if (!standardparagraph)
    standardparagraph = new XslFoFootnoteParagraph ("ft", 11, OFF, OFF, OFF, OFF, JUSTIFIED, 0, 0, 3, 0, 0, false);
  if (!extraparagraph)
    extraparagraph =    new XslFoFootnoteParagraph ("fp", 11, OFF, OFF, OFF, OFF, JUSTIFIED, 0, 0, 3, 0, 3, false);
  // Create footnote caller object.
  footnotecaller = new NoteCaller (note_numbering_type, note_numbering_user_sequence, spacious);
}


XslFoFootnote::~XslFoFootnote ()
{
  delete footnotecaller;
  delete standardparagraph;
  delete extraparagraph;
}


void XslFoFootnote::new_book ()
{
  if (note_numbering_restart == nnrtBook)
    footnotecaller->reset ();
  new_chapter ();
}


void XslFoFootnote::new_chapter ()
{
  if (note_numbering_restart == nnrtChapter)
    footnotecaller->reset ();
}


void XslFoFootnote::transform (XmlFoBlock * xmlfoblock, ustring& line)
// Replace all footnote related content with corresponding xslfo code.
{
  // If no opening marker in stylesheet, bail out.
  if (opening_marker.empty())
    return;
  
  // Variables.
  size_t opening_position;
  
  // Look for footnotes, but only deal with them if they have the endmarker too.
  opening_position = line.find (opening_marker);
  while (opening_position != string::npos) {
    
    // Look for the endmarker.
    size_t closing_position;
    closing_position = line.find (closing_marker, opening_position);    
    if (closing_position == string::npos) {
      gw_warning ("Missing endmarker: " + line);
      return;
    }

    // Take out this bit of the line, transform it, and insert it again.
    ustring footnote;
    footnote = line.substr (opening_position + opening_marker.length(), closing_position - opening_position - closing_marker.length());
    line.erase (opening_position, closing_position - opening_position + closing_marker.length());
    if (myshow) {
      footnote = transform_main_parts (xmlfoblock, footnote);
      line.insert (opening_position, footnote);
    }
    
    // Search for another footnote.
    opening_position = line.find (opening_marker, opening_position);
  }
}


ustring XslFoFootnote::transform_main_parts (XmlFoBlock * xmlfoblock, const ustring& line)
{
  // Variables.
  ustring xslfo_code;
  
  // Work on a copy.
  ustring footnote (line);
  
  // Add first bit of code.
  xslfo_code.append ("<fo:footnote>");
  
  // Extract the footnote caller.    
  ustring caller_in_text;
  ustring caller_in_note;
  if (footnote.length() > 0) {
    caller_in_text = footnote.substr (0, 1);
    caller_in_text = trim (caller_in_text);
    if (caller_in_text == "+") {
      caller_in_text = footnotecaller->get_caller();
      if (note_numbering_restart == nnrtPage)
        if (footnotecaller->get_spacious())
          caller_in_text = FOOTNOTE_CALLER_NUMBERING_PER_PAGE_TEXT_SPACIOUS;
        else
          caller_in_text = FOOTNOTE_CALLER_NUMBERING_PER_PAGE_TEXT;
    } else if (caller_in_text == "-") {
      caller_in_text.clear();
    }
    footnote.erase (0, 1);
    footnote = trim (footnote);
  }
  caller_in_note = caller_in_text;
  if (caller_in_note == FOOTNOTE_CALLER_NUMBERING_PER_PAGE_TEXT)
    caller_in_note = FOOTNOTE_CALLER_NUMBERING_PER_PAGE_NOTE;
  if (caller_in_note == FOOTNOTE_CALLER_NUMBERING_PER_PAGE_TEXT_SPACIOUS)
    caller_in_note = FOOTNOTE_CALLER_NUMBERING_PER_PAGE_NOTE_SPACIOUS;
  // Insert the xslfo code.
  xslfo_code.append (XmlFoInlineText (caller_in_text, xmlfoblock, anchor_fontpercentage, anchor_italic, anchor_bold, anchor_underline, anchor_smallcaps, anchor_superscript, 0));

  // We now come to the footnote body.
  xslfo_code.append ("<fo:footnote-body>");
  xslfo_code.append ("<fo:list-block provisional-label-separation=\"0pt\" provisional-distance-between-starts=\"18pt\">");
  xslfo_code.append ("<fo:list-item>");
  
  // Insert the caller in the footnote body too.
  xslfo_code.append ("<fo:list-item-label end-indent=\"label-end()\">");
  vector<ustring> lines;
  {
    XmlFoBlock block (&lines, standardparagraph->fontsize, 100,
                      standardparagraph->italic, standardparagraph->bold,
                      standardparagraph->underline, standardparagraph->smallcaps,
                      standardparagraph->justification, 
                      standardparagraph->spacebefore, standardparagraph->spaceafter, 
                      0, 0, 0, false, false);
    if (!caller_in_note.empty()) {
      lines.push_back (XmlFoInlineText (caller_in_note, &block, anchor_fontpercentage, anchor_italic, anchor_bold, anchor_underline, anchor_smallcaps, false, 0));
    }
  }
  xslfo_code.append ("\n");
  for (unsigned int i = 0; i < lines.size(); i++) {
    xslfo_code.append (lines[i]);
    xslfo_code.append ("\n");
  }
  xslfo_code.append ("</fo:list-item-label>");

  // Insert some code for this footnote into the body.
  xslfo_code.append ("<fo:list-item-body start-indent=\"body-start()\">");
  
  // Divide the footnote into paragraphs, if there are more than one.
  vector<ustring> paragraphs;
  size_t pos;
  pos = footnote.find (extraparagraph->marker_open);
  while (pos != string::npos) {
    paragraphs.push_back (footnote.substr (0, pos));
    footnote.erase (0, pos + extraparagraph->marker_open.length());
    pos = footnote.find (extraparagraph->marker_open);
  }
  paragraphs.push_back (footnote);

  // Deal with each paragraph.
  for (unsigned int i = 0; i < paragraphs.size(); i++) {
    // Choose the right paragraph.
    XslFoFootnoteParagraph * paragraph;
    if (i == 0)
      paragraph = standardparagraph;
    else
      paragraph = extraparagraph;
    // Format actual text, and comply with footnote nesting, see USFM standard.
    paragraphs[i] = usfm_notes_handle_nesting (paragraphs[i], 
      standardparagraph->marker_open, standardparagraph->marker_close, footnote_markers);
    // Insert the paragraph as another block.
    lines.clear();
    {
      XmlFoBlock block (&lines, paragraph->fontsize, 100,
                        paragraph->italic, paragraph->bold,
                        paragraph->underline, paragraph->smallcaps,
                        paragraph->justification, 
                        paragraph->spacebefore, paragraph->spaceafter, 
                        paragraph->leftmargin, paragraph->rightmargin, 
                        paragraph->firstlineindent, false, false);
      lines.push_back (paragraphs[i]);
    }
    xslfo_code.append ("\n");
    for (unsigned int i = 0; i < lines.size(); i++) {
      xslfo_code.append (lines[i]);
      xslfo_code.append ("\n");
    }
  }    
  
  // Close footnote and foot notebody.
  xslfo_code.append ("</fo:list-item-body>");
  xslfo_code.append ("</fo:list-item>");
  xslfo_code.append ("</fo:list-block>");
  xslfo_code.append ("</fo:footnote-body>");
  xslfo_code.append ("</fo:footnote>");

  // Return the code.
  return xslfo_code;  
}
