/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Application.h"
#include "OperationLog.h"
#include "TrayIcon.h"
#include "TrayMenu.h"
#include "AboutDialog.h"
#include "LogDialog.h"
#include "TwoPhaseLogRecord.h"
#include "BasicConfigDialog.h"
#include "AdvancedConfigWindow.h"
#include "FilterConfigWindow.h"
#include "ForwardingConfigWindow.h"
#include "RequestLogWindow.h"
#include "Conf.h"
#include "ConfigFile.h"
#include "ForwardingConfigFile.h"
#include "UrlsFile.h"
#include "ContentFilters.h"
#include "GlobalState.h"
#include "WorkerThreadPoolExSingleton.h"
#include "ScopedIncDec.h"
#include "FileOps.h"
#include "cache/ObjectStorage.h"
#ifdef DEBUG
#include "DebugWindow.h"
#endif
#include "binreloc/prefix.h"
#include <glibmm/miscutils.h>
#include <glibmm/fileutils.h>
#include <gtkmm/window.h>
#include <gdk/gdk.h> // for gdk_notify_startup_complete()
#include <fstream>
#include <string>
#include <stddef.h>
#include <locale.h>

using namespace std;

namespace GtkGUI
{

class Application::WorkerThreadPoolActivator
{
public:
	WorkerThreadPoolActivator() {
		// starts accepting and processing connections
		WorkerThreadPoolExSingleton::instance()->activate();
	}
	
	~WorkerThreadPoolActivator() {
		// finishes worker threads
		WorkerThreadPoolExSingleton::instance()->deactivate();
	}
};


Application* Application::m_spInstance = 0;

Application::Application(int argc, char* argv[])
:	m_isExiting(false),
	m_commandNestLevel(0),
	m_gtkMain(argc, argv),
	m_commandQueue(COMMAND_QUEUE_CAPACITY, *this),
	m_networkActivityHandler(m_commandQueue),
	m_filterJsLogHandler(m_commandQueue),
	m_requestLogHandler(m_commandQueue)
#ifdef DEBUG
	, m_debugAgent(m_commandQueue)
#endif
{
	// Some locales (tr_TR for example) are not ASCII-compatible
	// with regards to case conversion, so we set LC_COLLATE and LC_CTYPE
	// to C locale.
	{
		// The locale is set by Gtk::Main constructor, but for some
		// reason, creating the first window sets it again.
		Gtk::Window wnd;
	}
	setlocale(LC_COLLATE, "C");
	setlocale(LC_CTYPE, "C");
	
	m_userConfDir = Glib::build_filename(Glib::get_home_dir(), ".bfilter");
	m_globalConfDir = string(SYSCONFDIR) + "/bfilter";
	m_userCacheDir = Glib::build_filename(m_userConfDir, "cache");
	
	m_spInstance = this;
	
	m_ptrConfigFile.reset(new ConfigFile(m_userConfDir+"/config"));
	
	m_ptrFwdConfigFile.reset(new ForwardingConfigFile(
		m_userConfDir+"/forwarding.xml"
	));
	
	m_ptrUrlsFile.reset(
		new UrlsFile(UrlsFile::URLS, m_globalConfDir+"/urls")
	);
	m_ptrUrlsLocalFile.reset(
		new UrlsFile(UrlsFile::URLS_LOCAL, m_userConfDir+"/urls.local")
	);
	m_ptrContentFilters.reset(
		new ContentFilters(m_userConfDir+"/filters", m_globalConfDir+"/filters")
	);
	
	m_ptrTrayIcon.reset(new TrayIcon);
	m_ptrTrayMenu.reset(new TrayMenu);
#ifdef DEBUG
	m_ptrDebugWindow.reset(new DebugWindow);
#endif
	m_dispatcher.connect(
		sigc::mem_fun(*this, &Application::processCommands)
	);
}

Application::~Application()
{
	m_destroySignal.emit();
}

void
Application::run()
{
	bool errors = false;
	errors |= !createMissingUserConfigs();
	errors |= !m_ptrConfigFile->loadAndForceApply();
	
	{
		// !!! assumes config is already loaded !!!
		ObsoleteForwardingInfo const fwd_fallback(
			GlobalState::ReadAccessor()->config()
			.getObsoleteForwardingInfo()
		);
		m_ptrFwdConfigFile->loadAndApply(fwd_fallback);
	}
	
	m_ptrUrlsFile->loadAndForceApply();
	m_ptrUrlsLocalFile->loadAndForceApply();
	m_ptrContentFilters->loadAndApply();
	loadCache();
	if (errors) {
		showLogDialog();
	}
	
	{
		WorkerThreadPoolActivator activator;
		m_gtkMain.run();
	}
}

void
Application::entryPoint(int argc, char* argv[])
{
	//Glib::thread_init();
	// we don't really need it since we've moved from
	// Glib::signal_idle to Glib::Dispatcher.
	
	Application app(argc, argv);
	gdk_notify_startup_complete();
	app.run();
}

void
Application::showTrayMenu(guint button, guint32 activate_time)
{
	m_ptrTrayMenu->popup(button, activate_time);
}

void
Application::showAboutDialog()
{
	AboutDialog::showWindow();
}

void
Application::showLogDialog()
{
	LogDialog::showWindow();
}

void
Application::showBasicConfigDialog()
{
	BasicConfigDialog::showWindow();
}

void
Application::showAdvancedConfigWindow()
{
	AdvancedConfigWindow::showWindow();
}

void
Application::showFilterConfigWindow()
{
	FilterConfigWindow::showWindow();
}

void
Application::showForwardingConfigWindow()
{
	ForwardingConfigWindow::showWindow();
}

void
Application::showRequestLogWindow()
{
	RequestLogWindow::showWindow();
}

#ifdef DEBUG
void
Application::showDebugWindow()
{
	m_ptrDebugWindow->present();
}
#endif

void
Application::requestAppExit()
{
	// If we just call m_gtkMain.quit(), we may get a deadlock
	// while waiting for worker threads to finish. This happens if
	// a worker thread is blocked while sending a command through
	// m_commandQueue.
	if (m_commandQueue.close() == 0) {
		// command queue is empty
		m_gtkMain.quit();
	} else {
		// Wait for the pending commands to be processed.
		// The actual quit will be done in notifyQueueEmpty().
		m_isExiting = true;
	}
}

void
Application::handleNetworkActivity()
{
	m_ptrTrayIcon->displayNetworkActivity();
}

void
Application::notifyCommandsPending()
{
	m_dispatcher.emit();
}

// Note: this function is called from the main thread only.
void
Application::notifyQueueEmpty()
{
	if (m_isExiting) {
		m_gtkMain.quit();
	}
}

void
Application::processCommands()
{
	if (m_commandNestLevel) {
		// If a command does things that invoke a recursive main loop,
		// we can end up here.
		return;
	}
	
	ScopedIncDec<int> nest_level_manager(m_commandNestLevel);
	
	unsigned max_commands = m_commandQueue.getCapacity();
	/*
	We limit the number of iterations to give the GUI thread
	a chance to draw something if other threads keep posting commands.
	*/
	for (; max_commands > 0; --max_commands) {
		InterthreadCommandQueue::CommandPtr command = m_commandQueue.pop();
		if (!command) {
			return;
		}
		(*command)();
	}
	m_dispatcher.emit(); // there could be more commands in the queue
}

bool
Application::loadCache()
{
	
	Log* log = OperationLog::instance();
	TwoPhaseLogRecord log_record(log,
		StyledText("Loading cache ... ", log->getDefaultStyle()),
		StyledText("done", log->getSuccessStyle())
	);
	
	if (GlobalState::ReadAccessor()->config().getCacheSize() == 0) {
		log_record.setSecondPhaseMessage(
			StyledText("disabled", log->getDefaultStyle())
		);
		return true;
	}
	
	if (!HttpCache::ObjectStorage::instance()->setCacheDir(m_userCacheDir)) {
		log_record.setSecondPhaseMessage(
			StyledText("failed", log->getErrorStyle())
		);
		return false;
	}
	
	return true;
}

bool
Application::createMissingUserConfigs()
{
	std::string const config_fname(
		Glib::build_filename(m_userConfDir, "config")
	);
	std::string const user_forwarding_xml_fname(
		Glib::build_filename(m_userConfDir, "forwarding.xml")
	);
	std::string const global_forwarding_xml_fname(
		Glib::build_filename(m_globalConfDir, "forwarding.xml")
	);
	std::string const urls_local_fname(
		Glib::build_filename(m_userConfDir, "urls.local")
	);
	std::string const filters_dirname(
		Glib::build_filename(m_userConfDir, "filters")
	);
	bool const conf_dir_exists = fileExists(m_userConfDir);
	bool const cache_dir_exists = fileExists(m_userCacheDir);
	bool const config_exists = fileExists(config_fname);
	bool const user_forwarding_xml_exists = fileExists(user_forwarding_xml_fname);
	bool const global_forwarding_xml_exists = fileExists(global_forwarding_xml_fname);
	bool const urls_local_exists = fileExists(urls_local_fname);
	bool const filters_dir_exists = fileExists(filters_dirname);
	if (conf_dir_exists && cache_dir_exists && config_exists &&
	    urls_local_exists && filters_dir_exists &&
	    (user_forwarding_xml_exists || !global_forwarding_xml_exists)) {
		return true;
	}
	
	Log* log = OperationLog::instance();
	log->appendRecord("Creating user configuration ... ");
	size_t num_records = log->getNumRecords();
	bool errors = false;
	
	if (!conf_dir_exists) {
		errors |= !FileOps::createDir(m_userConfDir);
	}
	if (!cache_dir_exists) {
		errors |= !FileOps::createDir(m_userCacheDir);
	}
	if (!config_exists) {
		string const source(Glib::build_filename(m_globalConfDir, "config"));
		errors |= !copyFile(source, config_fname);
	}
	if (!user_forwarding_xml_exists && global_forwarding_xml_exists) {
		errors |= !copyFile(global_forwarding_xml_fname, user_forwarding_xml_fname);
	}
	if (!urls_local_exists) {
		string const source(Glib::build_filename(m_globalConfDir, "urls.local"));
		errors |= !copyFile(source, urls_local_fname);
	}
	if (!filters_dir_exists) {
		errors |= !FileOps::createDir(filters_dirname);
	}
	
	if (!errors && num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	return !errors;
}

bool
Application::copyFile(std::string const& source, std::string const& target)
{
	Log* log = OperationLog::instance();
	ifstream in(source.c_str());
	if (!in.good()) {
		log->appendRecord(
			"Could not open " + source,
			log->getErrorStyle()
		);
		return false;
	}
	
	ofstream out(target.c_str());
	if (!out.good()) {
		log->appendRecord(
			"Could not create " + target,
			log->getErrorStyle()
		);
		return false;
	}
	
	out << in.rdbuf();
	if (in.bad()) {
		log->appendRecord(
			"Error reading from " + source,
			log->getErrorStyle()
		);
		return false;
	}
	if (out.bad()) {
		log->appendRecord(
			"Error writing to " + target,
			log->getErrorStyle()
		);
		return false;
	}
	
	return true;
}

std::string
Application::getConfigFileName() const
{
	return Glib::build_filename(m_userConfDir, "config");
}

std::string
Application::getConfigDefaultFileName() const
{
	return Glib::build_filename(m_globalConfDir, "config.default");
}

std::string
Application::getStandardUrlPatternsFileName() const
{
	return Glib::build_filename(m_globalConfDir, "urls");
}

std::string
Application::getLocalUrlPatternsFileName() const
{
	return Glib::build_filename(m_userConfDir, "urls.local");
}

std::string
Application::getGlobalFiltersDir() const
{
	return Glib::build_filename(m_globalConfDir, "filters");
}

std::string
Application::getUserFiltersDir() const
{
	return Glib::build_filename(m_userConfDir, "filters");
}

bool
Application::fileExists(std::string const& path)
{
	return Glib::file_test(path, Glib::FILE_TEST_EXISTS);
}

bool
Application::fileExists(std::string const& dir, std::string const& file)
{
	return fileExists(Glib::build_filename(dir, file));
}

} // namespace GtkGUI
