/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FilteringServer.h"
#include "URI.h"
#include "HttpRequestMetadata.h"
#include "HttpRequestLine.h"
#include "HttpStatusLine.h"
#include "AbstractRequestHandler.h"
#include "AbstractResponseHandler.h"
#include "FilterTryList.h"
#include "CraftedResponse.h"
#include "ErrorCodes.h"
#include "ErrorFactory.h"
#include "ErrorDescriptor.h"
#include "ImmediateResponse.h"
#include "ReplacementImage.h"
#include "ReplacementFlash.h"
#include "ReplacementHtml.h"
#include "ReplacementJs.h"
#include "ResponseFilterChain.h"
#include "RegexFilterDescriptor.h"
#include "SubstitutionRequestParser.h"
#include "State.h"
#include "GlobalState.h"
#include "Conf.h"
#include "CombinedUrlPatterns.h"
#include "AvailableFiltersOrdered.h"
#include "BString.h"
#include "RequestTag.h"
#include "ArraySize.h"
#include <string>

using namespace std;

class FilteringServer::ImageSubstitutionParser : public SubstitutionRequestParser
{
public:
	std::auto_ptr<ImmediateResponse> retrieveResponse() {
		return m_ptrResponse;
	}
private:
	virtual void handleResult(
		unsigned int width, unsigned int height,
		URI const& url, BString const& orig_path);
	
	std::auto_ptr<ImmediateResponse> m_ptrResponse;
};


class FilteringServer::FlashSubstitutionParser : public SubstitutionRequestParser
{
public:
	std::auto_ptr<ImmediateResponse> retrieveResponse() {
		return m_ptrResponse;
	}
private:
	virtual void handleResult(
		unsigned int width, unsigned int height,
		URI const& url, BString const& orig_path);
	
	std::auto_ptr<ImmediateResponse> m_ptrResponse;
};


FilteringServer::FilteringServer(
	ServiceContext& context, AbstractServer& delegate)
:	m_rContext(context),
	m_rDelegate(delegate)
{
}

FilteringServer::~FilteringServer()
{
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::submitRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	IntrusivePtr<AbstractRequestHandler> res;
	
	typedef IntrusivePtr<AbstractRequestHandler> (FilteringServer::*ProcessFunc)(
		ConstRequestPtr const&, RequestTag const&,
		IntrusivePtr<AbstractResponseHandler> const&);
	
	static ProcessFunc const process_funcs[] = {
		&FilteringServer::processConnectRequest,
		&FilteringServer::processSubstitutionRequest,
		&FilteringServer::processUrlPatterns,
		&FilteringServer::processAnalyzeRequest,
		&FilteringServer::processGenericRequest
	};
	
	ProcessFunc const* f = process_funcs;
	ProcessFunc const* const f_end = f + ARRAY_SIZE(process_funcs);
	for (; f != f_end && !res; ++f) {
		res = (this->*(*f))(request_metadata, request_tag, handler);
	}
	
	return res;
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::submitRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler,
	std::auto_ptr<ImmediateResponse> response)
{
	return m_rDelegate.submitRequest(
		request_metadata, request_tag, handler, response
	);
}

bool
FilteringServer::isIdle() const
{
	return m_rDelegate.isIdle();
}

bool
FilteringServer::isOverloaded() const
{
	return m_rDelegate.isOverloaded();
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processConnectRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	if (request_metadata->requestLine().getMethod() == BString("CONNECT")) {
		URI const& url = request_metadata->requestLine().getURI();
		UrlPatterns::Substitution subst =
			GlobalState::ReadAccessor()->urlPatterns()
			.getSubstitutionFor(url);
		if (subst == UrlPatterns::NO_SUBSTITUTION) {
			return m_rDelegate.submitRequest(
				request_metadata, request_tag, handler
			);
		} else {
			return m_rDelegate.submitRequest(
				request_metadata, request_tag, handler,
				ErrorFactory::errUrlForbidden(
					ErrorCodes::URL_FORBIDDEN,
					std::string(), url
				)->retrieveResponse()
			);
		}
	}
	
	return IntrusivePtr<AbstractRequestHandler>();
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processSubstitutionRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	BString const si_prefix("bf-si-");
	BString const sf_prefix("bf-sf-");
	URI const& uri = request_metadata->requestLine().getURI();
	
	ImageSubstitutionParser image_parser;
	if (image_parser.parse(uri, si_prefix)) {
		request_tag->flags().set(RequestTag::REQUEST_AD_SUBST);
		return m_rDelegate.submitRequest(
			request_metadata, request_tag, handler,
			image_parser.retrieveResponse()
		);
	}
	
	FlashSubstitutionParser flash_parser;
	if (flash_parser.parse(uri, sf_prefix)) {
		request_tag->flags().set(RequestTag::REQUEST_AD_SUBST);
		return m_rDelegate.submitRequest(
			request_metadata, request_tag, handler,
			flash_parser.retrieveResponse()
		);
	}
	
	return IntrusivePtr<AbstractRequestHandler>();
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processUrlPatterns(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	URI const& url = request_metadata->requestLine().getURI();
	UrlPatterns::Substitution subst = GlobalState::ReadAccessor()->urlPatterns().getSubstitutionFor(url);
	
	if (subst == UrlPatterns::AUTO_SUBST) {
		BString const& path = request_metadata->requestLine().getURI().getRawPath();
		BString const swf(".swf");
		if (StringUtils::ciEndsWith(path.begin(), path.end(), swf.begin(), swf.end())) {
			subst = UrlPatterns::EMPTY_FLASH;
		} else {
			subst = UrlPatterns::EMPTY_IMAGE;
		}
	}
	
	std::auto_ptr<ImmediateResponse> response;
	switch (subst) {
		case UrlPatterns::FORBID_HTML: {
			response = ErrorFactory::errUrlForbidden(
				ErrorCodes::URL_FORBIDDEN,
				std::string(), url
			)->retrieveResponse();
			break;
		}
		case UrlPatterns::EMPTY_IMAGE: {
			response = ReplacementImage::createHttpResponse(
				1, 1, // width, height
				HttpStatusLine::SC_FORBIDDEN
			);
			break;
		}
		case UrlPatterns::EMPTY_FLASH: {
			response = ReplacementFlash::createHttpResponse(
				1, 1, // width, height
				url.toString()
			);
			break;
		}
		case UrlPatterns::EMPTY_HTML: {
			response = ReplacementHtml::createHttpResponse();
			break;
		}
		case UrlPatterns::EMPTY_JS: {
			response = ReplacementJs::createHttpResponse();
			break;
		}
		default: {
			return IntrusivePtr<AbstractRequestHandler>();
		}
	}
	
	return m_rDelegate.submitRequest(
		request_metadata, request_tag, handler, response
	);
	
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processAnalyzeRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	IntrusivePtr<AbstractRequestHandler> res;
	URI const& uri = request_metadata->requestLine().getURI();
	BString const prefix("bf-analyze");
	BString const is_str("-is");
	BString path = uri.getRawPath();
	bool ignore_size = false;
	if (StringUtils::startsWith(path.begin(), path.end(),
	                            prefix.begin(), prefix.end())) {
		path.trimFront(prefix.size());
		if (StringUtils::startsWith(path.begin(), path.end(),
		                            is_str.begin(), is_str.end())) {
			ignore_size = true;
			path.trimFront(is_str.size());
		}
		if (!path.empty()) {
			if (path[0] == '/') {
				path.trimFront(1);
			} else {
				return res;
			}
		}
		URI new_uri(uri);
		new_uri.setAbsoluteRawPath(path);
		RequestPtr req(new HttpRequestMetadata(*request_metadata));
		req->requestLine().setURI(new_uri);
		std::auto_ptr<FilterTryList> filter_try_list(new FilterTryList);
		if (isFilteringEnabledForURL(*GlobalState::ReadAccessor(), new_uri)) {
			filter_try_list->append(FilterTryList::FilterConstructorPtr(
				new FilterTryList::FilterConstructor(
					sigc::bind(
						sigc::ptr_fun(&FilterTryList::tryAnalyzeFilter),
						ignore_size
					)
				)
			));
			request_tag->flags().set(RequestTag::REQUEST_ANALYZE);
		}
		
		IntrusivePtr<AbstractResponseHandler> const filter_chain(
			new ResponseFilterChain(
				m_rContext, req, request_tag,
				filter_try_list, handler
			)
		);
		res = m_rDelegate.submitRequest(req, request_tag, filter_chain);
	}
	
	return res;
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processGenericRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	URI const& uri = request_metadata->requestLine().getURI();
	std::auto_ptr<FilterTryList> filters(new FilterTryList);
	
	{
		GlobalState::ReadAccessor global_state; // hold the lock
		if (isFilteringEnabledForURL(*global_state, uri)) {
			applyStandardFilters(*filters, uri);
		}
		if (global_state->config().isClientCompressionEnabled()) {
			filters->append(FilterTryList::FilterConstructorPtr(
				new FilterTryList::FilterConstructor(
					sigc::ptr_fun(&FilterTryList::tryCompressorFilter)
				)
			));
		}
	}
	
	IntrusivePtr<AbstractResponseHandler> const filter_chain(
		new ResponseFilterChain(
			m_rContext, request_metadata,
			request_tag, filters, handler
		)
	);
	return m_rDelegate.submitRequest(request_metadata, request_tag, filter_chain);
}

void
FilteringServer::applyStandardFilters(FilterTryList& filters, URI const& request_url)
{
	BString const url_str(request_url.toBString());
	
	filters.append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::tryTextContentValidator)
		)
	));

	// Note: HtmlResponseFilter depends on the presence of HtmlContentValidator
	filters.append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::tryHtmlContentValidator)
		)
	));
	
	GlobalState::ReadAccessor state; // hold the read lock
	
	AvailableFiltersOrdered const& avail_filters = state->contentFilters();
	AvailableFiltersOrdered::iterator it(avail_filters.begin());
	AvailableFiltersOrdered::iterator const end(avail_filters.end());
	for (; it != end && (*it)->order() < 0; ++it) {
		filters.tryAppendRegexFilter(url_str, *it);
	}
	filters.append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::tryHtmlFilter)
		)
	));
	for (; it != end; ++it) {
		filters.tryAppendRegexFilter(url_str, *it);
	}
}

bool
FilteringServer::isFilteringEnabledForURL(State const& state, URI const& url)
{
	if (!state.isFilteringEnabled()) {
		return false;
	}
	
	return !state.urlPatterns().isFilteringDisabled(url);
}


void
FilteringServer::ImageSubstitutionParser::handleResult(
	unsigned int width, unsigned int height,
	URI const& url, BString const& orig_path)
{
	m_ptrResponse = ReplacementImage::createHttpResponse(
		width, height, HttpStatusLine::SC_OK
	);
}

void
FilteringServer::FlashSubstitutionParser::handleResult(
	unsigned int width, unsigned int height,
	URI const& url, BString const& orig_path)
{
	URI orig_url(url);
	orig_url.setAbsoluteRawPath(orig_path);
	m_ptrResponse = ReplacementFlash::createHttpResponse(
		width, height, orig_url.toString()
	);
}
