// bsload.cc for bbsload - a tool to display the load in X11.
//
//  Copyright (c) 1998-2000 by John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// (See the included file COPYING / GPL-2.0)
//

#include "bbsload.hh"

#define NETBUFSIZE 255
#define BYTESINAKILOBIT 125.0	// Number of bytes in a Kilobit (float)

CheckLoad::CheckLoad(ToolWindow *toolwindow) {
  bbtool=toolwindow;
  timer=new BTimer(bbtool->getCurrentScreenInfo()->getBaseDisplay(),this) ;
  timer->setTimeout(bbtool->getResource()->report.check_delay*1000);
  timer->start();
  task.running=0;
  task.total=0;
  broken=False;
  load.average1m=0;
  load.average5m=0;
  load.average15m=0;
}

void CheckLoad::Reconfigure() {
  timeout();
}

void CheckLoad::timeout()
{
    FILE *fp;

    char loadstr[28];
    char buffer[128];
    bool redraw=False;
    struct stat file_status;
    static time_t last_mtime;
    static unsigned long old_user=0;
    static unsigned long old_nice=0;
    static unsigned long old_system=0;
    static unsigned long old_idle=0;
    broken=False;

    if (!(fp = fopen("/proc/loadavg", "r")))
        broken=True;
    else
    {
        fstat(fileno(fp),&file_status);
        if  (file_status.st_mtime != last_mtime)
        {
            fgets(loadstr, 28, fp);
            sscanf(loadstr,"%e %e %e %d/%d",&load.average1m,
                    &load.average5m,&load.average15m,&task.running,
                    &task.total);
            redraw=true;
        }
    }
    fclose(fp);
     if (!(fp = fopen("/proc/meminfo", "r")))
        broken=True;
    else
    {
        fstat(fileno(fp),&file_status);
        if  (file_status.st_mtime != last_mtime)
        {
            /* should work for now */
            fgets(buffer,128,fp);
            fgets(buffer,128,fp);
            sscanf(buffer,"Mem:  %lu %lu %lu %lu %lu %lu",&mem.total,
                     &mem.used,&mem.free,&mem.shared,&mem.buffers,
                     &mem.cached);
            fgets(buffer,128,fp);
            sscanf(buffer,"Swap: %lu %lu %lu",
                   &swap.total,&swap.used,&swap.free);
            
            norm.total_used=(double)(mem.used+swap.used)/
                            (double)(mem.total+swap.total);
            norm.mem_used=(double)mem.used/(double)mem.total;
            if (swap.total!=0)
              norm.swap_used=(double)swap.used/(double)swap.total;
                    
            redraw=true;
        }
    }
    fclose(fp);

    if (!(fp = fopen("/proc/stat", "r")))
        broken=True;
    else
    {
        fstat(fileno(fp),&file_status);
        if  (file_status.st_mtime != last_mtime)
        {
           unsigned long user,nice,system,idle,total;
            /* should work for now */
            fgets(buffer,128,fp);
            sscanf(buffer,"cpu  %lu %lu %lu %lu",&user,
                     &nice,&system,&idle);
            total=(user-old_user)+(nice-old_nice)+
                  (system-old_system)+(idle-old_idle);

            if ((total!=0)&(old_user!=0))
            {
              cpu.user=(user-old_user)/(double)total;
              cpu.nice=(nice-old_nice)/(double)user;
              cpu.system=(system-old_system)/(double)total;
              cpu.idle=(idle-old_idle)/(double)total;
            }
            else
              cpu.user=cpu.nice=cpu.system=cpu.idle=0;

            old_user=user;
            old_nice=nice;
            old_system=system;
            old_idle=idle;
            redraw=true;
        }
    }
    fclose(fp);
    
    char s[NETBUFSIZE], *ch;
#ifdef  _GNU_SOURCE
    unsigned long long totalin=0,totalin_old=0,totalout=0,totalout_old=0;
#else
    unsigned long totalin=0,totalin_old=0,totalout=0,totalout_old=0;
#warning "Use the GNU Compiler Collection for better accuracy in your integers."
#endif
    
    // Read the network interface from /proc/net/dev
    // This is GNU/Linux specific code. I have access to SUN and
    // FreeBSD kernels, so I will hopefully update this to be
    // cross-platform soon. Thanks to the fluxbox team for the
    // code which this is based on. Data in /proc/net/dev is cumulative,
    // so we must subtract the last reading. "Sam Halliday" <fommil@yahoo.ie>
    totalin_old=net.intotal;
    totalout_old=net.outtotal;
    
    if (!(fp = fopen("/proc/net/dev", "r")))
    {
     fprintf(stderr, "Open /proc/net/dev failed. Is a Linux /proc mounted?");
     broken=True;
    }
    else
    {
     while (!feof(fp))
     {
      fgets (s, NETBUFSIZE-1, fp);
      if (strstr(s, bbtool->getResource()->report.netname))
      {
       ch = strchr(s,':');
#ifdef  _GNU_SOURCE
       if(ch) sscanf(++ch, "%Lu %*d %*d %*d %*d %*d %*d %*d %Lu", &net.intotal,
       &net.outtotal);
#else
       if(ch) sscanf(++ch, "%u %*d %*d %*d %*d %*d %*d %*d %u", &net.intotal,
       &net.outtotal);
#endif
       redraw=true;
       break;
      }
     }
    }
    fclose(fp);
    // We need to consider the difference in bytes, network card speed and our
    // delay in checking this value.
    // BUG: the first reading will always be random as net.intotal and
    // net.outtotal will be random numbers on startup.
    // BUG: the 'total' numbers get quite big, and will eventually
    // overrun the integer limit. This will only mess up one reading
    // and will fix itself on the next 'check_delay'
    net.inchange=(float)(net.intotal-totalin_old)/
                 ((bbtool->getResource()->report.netspeed)*BYTESINAKILOBIT
		 *(bbtool->getResource()->report.check_delay));
    net.outchange=(float)(net.outtotal-totalout_old)/
                 ((bbtool->getResource()->report.netspeed)*BYTESINAKILOBIT
		 *(bbtool->getResource()->report.check_delay));
    // printf("DEBUG: %d\t%d\t%f\t%f\n",net.intotal,net.outtotal,net.inchange,net.outchange);
           
    if (redraw)
      bbtool->Redraw();
}


ToolWindow::ToolWindow(int argc,char **argv,struct CMDOPTIONS *options):
  Basewindow(argc,argv,options)
{
    /* initialize xrm database */
    XrmInitialize();
    resource = new Resource(this);

    load = new GEOM [resource->gauge.number_of_bars];
    MakeWindow(False);
    check_load = new CheckLoad(this);
    check_load->Reconfigure();
    Redraw();
    eventLoop();
}

ToolWindow::~ToolWindow()
{
    XUnmapWindow(getXDisplay(),framewin);

    /* destroy pixmaps */
    if (pixmap.frame) getImageControl()->removeImage(pixmap.frame);
    if (pixmap.label) getImageControl()->removeImage(pixmap.label);
    if (pixmap.loadbar_active)
        getImageControl()->removeImage(pixmap.loadbar_active);
    if (pixmap.loadbar_inactive)
        getImageControl()->removeImage(pixmap.loadbar_inactive);

    if (frameGC) XFreeGC(getXDisplay(),frameGC);
    XDestroyWindow(getXDisplay(),labelwin);
    XDestroyWindow(getXDisplay(),framewin);
    delete resource;
}


void ToolWindow::reconfigure(void)
{

    /* destroy pixmaps */
    getImageControl()->removeImage(pixmap.frame);
    getImageControl()->removeImage(pixmap.label);
    getImageControl()->removeImage(pixmap.loadbar_active);
    getImageControl()->removeImage(pixmap.loadbar_inactive);
    delete [] load;
    resource->Reload();

    load = new GEOM [resource->gauge.number_of_bars];
    memset(load,0,sizeof(GEOM)*resource->gauge.number_of_bars);

    MakeWindow(True);

    if (resource->show.label)
      XClearWindow(getXDisplay(), labelwin);
    XClearWindow(getXDisplay(), framewin);
    check_load->Reconfigure();
    Redraw();
}

void ToolWindow::MakeWindow(bool reconfigure)
{
  XSetWindowAttributes attrib;
  XWMHints wmhints;
  int i;

  unsigned long create_mask = CWBackPixmap|CWCursor|CWEventMask;

  if (!resource->show.vertical)
    frame.height=resource->frame.font->ascent+resource->frame.font->descent+
               4*resource->frame.bevelWidth;
  else
    frame.height=(resource->frame.font->ascent+resource->frame.font->descent+
               4*resource->frame.bevelWidth)*2;

  label.seperator_width=label.space_width=0;
  for (i=0;i<15;i++) label.width[i]=0;
  int extra_space=0;
  int label_counter=0;

  if (resource->show.label) {
    if (resource->show.running_counter) {
      label.width[0] = XTextWidth(resource->label.font,"00",strlen("00"));
      label_counter++;
    }
    if (resource->show.task_counter) {
      label.width[1] =  XTextWidth(resource->label.font,"00",strlen("00"));
            label_counter++;
    }
    if (resource->show.label_load1m) {
      label.width[2] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_load5m) {
      label.width[3] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_load15m) {
      label.width[4] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_totalused) {
      label.width[6] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_memused) {
      label.width[7] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_swapused) {
      label.width[8] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_usercpu) {
      label.width[9] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_nicecpu) {
      label.width[10] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_systemcpu) {
      label.width[11] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_idlecpu) {
      label.width[12] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_networkin) {
      label.width[13] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    if (resource->show.label_networkout) {
      label.width[14] =  XTextWidth(resource->label.font,"0.00",strlen("0.00"));
      label_counter++;
    }
    
    label.seperator_width = 
                       XTextWidth(resource->label.font,"/",strlen("/"));
    label.space_width = XTextWidth(resource->label.font," ",strlen(" ")) +
                       resource->frame.bevelWidth;
    if (!resource->show.vertical) 
      label.height=frame.height-2*resource->frame.bevelWidth;
    else
       label.height=(frame.height-2*resource->frame.bevelWidth)/2;
     
    extra_space=resource->frame.bevelWidth;
  }
  label.total_width=(label_counter-1)*label.seperator_width+2*label.space_width;
  for (i=0;i<15;i++) 
    label.total_width+=label.width[i];
    
  int number_of_loadbars=0;
  if (resource->show.load1m) number_of_loadbars++;
  if (resource->show.load5m) number_of_loadbars++;
  if (resource->show.load15m) number_of_loadbars++;
  if (resource->show.idlecpu) number_of_loadbars++;
  if (resource->show.nicecpu) number_of_loadbars++;
  if (resource->show.usercpu) number_of_loadbars++;
  if (resource->show.networkin) number_of_loadbars++;
  if (resource->show.networkout) number_of_loadbars++;
  if (resource->show.systemcpu) number_of_loadbars++;
  if (resource->show.totalused) number_of_loadbars++;
  if (resource->show.swapused) number_of_loadbars++;
  if (resource->show.memused) number_of_loadbars++;

  if (!resource->show.vertical) 
    frame.width =label.total_width + extra_space+ 
                resource->frame.bevelWidth*2 + 
                (resource->frame.bevelWidth+ resource->gauge.width)*
                number_of_loadbars;
  else {
    if ((label.total_width+extra_space)>
        (resource->frame.bevelWidth+ resource->gauge.width)*number_of_loadbars)
      frame.width=label.total_width+extra_space + 
                  resource->frame.bevelWidth*2;
    else
      frame.width=resource->frame.bevelWidth*2 + 
                  (resource->frame.bevelWidth+ resource->gauge.width)*
                   number_of_loadbars;
  }
  int label_x_position;
  
  if ((number_of_loadbars!=0)&(!resource->show.vertical))
    label_x_position= 2*resource->frame.bevelWidth+
                     (resource->frame.bevelWidth+
                      resource->gauge.width)*number_of_loadbars;
  else
    label_x_position= resource->frame.bevelWidth+extra_space/2;
     
  if (resource->position.mask & XNegative)
    resource->position.x = getCurrentScreenInfo()->getWidth()+
                           resource->position.x-frame.width;

  if (resource->position.mask & YNegative)
    resource->position.y = getCurrentScreenInfo()->getHeight()+
                           resource->position.y-frame.height;

	if (!withdrawn) {
		frame.x=resource->position.x;
		frame.y=resource->position.y;
	}

  
  attrib.background_pixmap = ParentRelative;

  pixmap.frame = getImageControl()->renderImage(frame.width, frame.height, 
                                  &resource->frame.texture);

  if (withdrawn)
    wmhints.initial_state = WithdrawnState;
  else
    wmhints.initial_state = NormalState;

  attrib.cursor = getSessionCursor();
  attrib.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
                      FocusChangeMask | KeyPressMask | StructureNotifyMask;

  if	(!reconfigure)
    framewin = XCreateWindow(getXDisplay(),
                             getCurrentScreenInfo()->getRootWindow(), 
                             resource->position.x,resource->position.y, 
                             frame.width,frame.height, 0, 
                             getCurrentScreenInfo()->getDepth(),InputOutput,
                             getCurrentScreenInfo()->getVisual(), 
                             create_mask, &attrib);
  else if (!withdrawn)
    XMoveResizeWindow(getXDisplay(),framewin,resource->position.x,
                      resource->position.y, frame.width,
                       frame.height);
  else
    XResizeWindow(getXDisplay(),framewin,frame.width,
                      frame.height);

 	wmhints.flags = IconWindowHint | StateHint;
	wmhints.icon_window = framewin;

  XTextProperty windowname;
  XClassHint classhints;
  
  char *name=BBTOOL;
  XSizeHints sizehints;

  classhints.res_name=BBTOOL;
  classhints.res_class="bbtools";
          
  sizehints.x=resource->position.x;
  sizehints.y=resource->position.y;
  sizehints.max_width=sizehints.min_width=frame.width;
  sizehints.max_height=sizehints.min_height=frame.height;
  sizehints.flags=USPosition|PMinSize|PMaxSize;

  XStringListToTextProperty(&name,1,&windowname);
  XSetWMProperties(getXDisplay(),framewin,&windowname,NULL,getArgv(),getArgc(),
                  &sizehints,&wmhints,&classhints);
  Atom wmproto[2];
  wmproto[0]=wm_delete_window;
  wmproto[1]=getBlackboxStructureMessagesAtom();
  XSetWMProtocols(getXDisplay(), framewin,wmproto, 2);

 if (!decorated&&!withdrawn) {
    BlackboxHints bb_hints;
    bb_hints.decoration= DecorNone;
    bb_hints.attrib= AttribOmnipresent;
    bb_hints.flags= AttribDecoration|AttribOmnipresent;
    XChangeProperty(getXDisplay(), framewin, getBlackboxHintsAtom(),
                getBlackboxHintsAtom(), 32, PropModeReplace,
                (unsigned char *) &bb_hints,  PropBlackboxHintsElements);
  }

  if (!shape)
    XSetWindowBackgroundPixmap(getXDisplay(), framewin, pixmap.frame);


  if (resource->show.label) {
    pixmap.label =
            getImageControl()->renderImage(label.total_width, label.height,
                                       &resource->label.texture);

    int label_y_position;
    if (!resource->show.vertical)
      label_y_position=resource->frame.bevelWidth;
    else
      label_y_position=frame.height/2+resource->frame.bevelWidth;
    
    if (!reconfigure)
      labelwin = XCreateWindow(getXDisplay(), framewin,label_x_position, 
                               label_y_position,
                               label.total_width,label.height, 0,
                               getCurrentScreenInfo()->getDepth(),InputOutput,
                               getCurrentScreenInfo()->getVisual(),create_mask, 
                               &attrib);
    else
      XMoveResizeWindow(getXDisplay(),labelwin,
                        label_x_position,
                        label_y_position,
                        label.total_width,label.height);
    if (!resource->label.transparent)
      XSetWindowBackgroundPixmap(getXDisplay(), labelwin, pixmap.label);
  }

  if (resource->show.loadbar) {
    int height;
    if (!resource->show.vertical)
      height=frame.height;
    else
      height=frame.height/2;
    pixmap.loadbar_active =	
          getImageControl()->renderImage(resource->gauge.width+4,height+4,
                                     &resource->gauge.active_texture);
    pixmap.loadbar_inactive =
           getImageControl()->renderImage(resource->gauge.width+4,height+4,
                                      &resource->gauge.inactive_texture);

    pixmap.loadbar_background = 
            getImageControl()->renderImage(resource->gauge.width+4,height+4,
                                       &resource->gauge.background_texture);
    if (resource->gauge.number_of_bars!=0) {
      if (!resource->show.vertical)
        gauge_height=frame.height-2*resource->frame.bevelWidth;
      else
        gauge_height=frame.height/2-2*resource->frame.bevelWidth;
    }
    int space=0;
    if (resource->gauge.tickmarks) {
      loadbar_height=(gauge_height-(resource->gauge.number_of_bars-1))/
                     (resource->gauge.number_of_bars);
      space=1;
    } 
    else
      loadbar_height=gauge_height/(resource->gauge.number_of_bars);
       
    for (i=0;i<resource->gauge.number_of_bars;i++) {
      load[i].x= (int) (resource->gauge.left_step_width*i
                        +resource->frame.bevelWidth);
      if (!resource->show.vertical)
        load[i].y=frame.height-//resource->frame.bevelWidth-
                     (loadbar_height+space)*(i+1);
      else
        load[i].y=frame.height/2-//+resource->frame.bevelWidth-
                     (loadbar_height+space)*(i+1);

      load[i].width=(int)(resource->gauge.width-
                         (resource->gauge.left_step_width+
                          resource->gauge.right_step_width)*i);
      if (!resource->gauge.levels_defined)
        resource->gauge.level[i]=
              (i+1)*(resource->gauge.max_load/resource->gauge.number_of_bars);
    }
  }

  if (!reconfigure) {
    if (resource->show.label) {
      gcv.font = resource->label.font->fid;
      frameGC = XCreateGC(getXDisplay(), framewin,GCFont, &gcv);
    }
    else
      frameGC = XCreateGC(getXDisplay(), framewin,0, &gcv);
  } else {
    if (resource->show.label) {
      gcv.font = resource->label.font->fid;
      XChangeGC(getXDisplay(), frameGC,GCFont, &gcv);
    }
    else
      XChangeGC(getXDisplay(), frameGC,0, &gcv);
  }


  if (resource->gauge.use_simple_color) {
    Pixmap dummy;
    for (i=0;i<resource->gauge.number_of_bars;i++) {
      dummy=getImageControl()->renderImage(resource->gauge.width+4,
                      frame.height+4,&resource->gauge.segment_texture[i]);
      XCopyArea(getXDisplay(),dummy,pixmap.loadbar_active,frameGC,
                2,2,resource->gauge.width,loadbar_height,2,load[i].y-1);
      getImageControl()->removeImage(dummy);
    }
  }

  if (!reconfigure) {
    XClearWindow(getXDisplay(), framewin);
    XMapSubwindows(getXDisplay(), framewin);
    XMapWindow(getXDisplay(), framewin);
    mapped=true;
    if (!decorated&&!withdrawn)
      XMoveResizeWindow(getXDisplay(),framewin,frame.x,
                         	frame.y, frame.width,
                         	frame.height);

  }

}


void ToolWindow::RedrawGauge(double level, int offset)
{
  int i;
  
  for (i=0;i<resource->gauge.number_of_bars;i++)
  {
    if (level>=resource->gauge.level[i])
      XCopyArea(getXDisplay(),pixmap.loadbar_active,framewin,frameGC,
                2,load[i].y-1,load[i].width,loadbar_height,
                load[i].x+offset*(load[0].width+
                resource->frame.bevelWidth),load[i].y-1);
    else
      XCopyArea(getXDisplay(),pixmap.loadbar_inactive,framewin,frameGC,
                2,load[i].y-1,load[i].width,loadbar_height,
                load[i].x+offset*(load[0].width+
                resource->frame.bevelWidth),load[i].y-1);
  }
  if (resource->gauge.background)
  {
    for (i=1;i<resource->gauge.number_of_bars;i++)
    {
       XCopyArea(getXDisplay(),pixmap.loadbar_background,framewin,frameGC,
                2,load[i].y-1,load[i].width,1,
                load[i].x+offset*(load[0].width+
                resource->frame.bevelWidth),load[i].y+loadbar_height-1);
    }
  }
}

void ToolWindow::RedrawLabel(double level,int labelnr)
{
  char t[6];

  if (labelnr!=0)
  {
    XSetForeground(getXDisplay(),frameGC,
                   resource->label.seperator_textColor.getPixel());
    sprintf(t, "/");
    XDrawString(getXDisplay(), labelwin, frameGC, xposition,
                (label.height+resource->label.font->ascent
                 -resource->label.font->descent) / 2,
                 t, strlen(t));
    xposition+=label.seperator_width;
  }
 
  XSetForeground(getXDisplay(),frameGC,
                 resource->label.load1m_textColor.getPixel());
  labelnr++;
  sprintf(t,"%1.2f",level);
  XDrawString(getXDisplay(), labelwin, frameGC, xposition,
             (label.height+resource->label.font->ascent-
             resource->label.font->descent) / 2,
             t, strlen(t));
  xposition+=label.width[2];
}

void ToolWindow::Redraw()
{
    int offset=0;
    if (resource->show.load1m)
    {
        RedrawGauge(check_load->getLoad().average1m,offset);  
        offset++;
    }
    if (resource->show.load5m)
    {
      RedrawGauge(check_load->getLoad().average5m,offset);  
      offset++;
    }
    if (resource->show.load15m)
    {
      RedrawGauge(check_load->getLoad().average15m,offset);
      offset++;
    }
    if (resource->show.memused)
    {
      RedrawGauge(check_load->getNorm().mem_used,offset);
      offset++;
    }
    if (resource->show.swapused)
    {
      RedrawGauge(check_load->getNorm().swap_used,offset);
      offset++;
    }
    if (resource->show.totalused)
    {
      RedrawGauge(check_load->getNorm().total_used,offset);
      offset++;
    }
    if (resource->show.usercpu)
    {
       RedrawGauge(check_load->getCpu().user,offset);
       offset++;    
    }
    if (resource->show.nicecpu)
    {
       RedrawGauge(check_load->getCpu().nice,offset);
       offset++;    
    }
    if (resource->show.systemcpu)
    {
       RedrawGauge(check_load->getCpu().system,offset);
       offset++;    
    }
    if (resource->show.idlecpu)
    {
       RedrawGauge(check_load->getCpu().idle,offset);
       offset++;    
    }
    if (resource->show.networkin)
    {
       RedrawGauge(check_load->getNet().inchange,offset);
       offset++;    
    }
    if (resource->show.networkout)
    {
       RedrawGauge(check_load->getNet().outchange,offset);
       offset++;    
    }
    if (resource->show.label)
    {
        char t[6];
        int labelnr=0;
        xposition=label.space_width;
        XClearWindow(getXDisplay(),labelwin);

        if (resource->show.running_counter)
        {
            labelnr++;
            XSetForeground(getXDisplay(),frameGC,
                           resource->label.running_textColor.getPixel());

            if (check_load->getTask().running<100)
            {
                sprintf(t, "%02d", check_load->getTask().running);
                XDrawString(getXDisplay(), labelwin, frameGC, label.space_width,
                            (label.height+resource->label.font->ascent-
                             resource->label.font->descent) / 2,
                            t, strlen(t));
            }
            xposition+=label.width[0];
        }
        if (resource->show.task_counter)
        {
            if (labelnr!=0)
            {
                XSetForeground(getXDisplay(),frameGC,
                            resource->label.seperator_textColor.getPixel());
                sprintf(t, "/");
                XDrawString(getXDisplay(), labelwin, frameGC, xposition,
                            (label.height+resource->label.font->ascent
                             -resource->label.font->descent) / 2,
                            t, strlen(t));
                xposition+=label.seperator_width;
            }
            
            labelnr++;
            
                XSetForeground(getXDisplay(),frameGC,
                                   resource->label.task_textColor.getPixel());

                sprintf(t, "%02d", check_load->getTask().total);
                XDrawString(getXDisplay(), labelwin, frameGC, xposition,
                            (label.height+resource->label.font->ascent-
                             resource->label.font->descent) / 2,
                            t, strlen(t));
                xposition+=label.width[1];
        }

        if (resource->show.label_load1m)
        {
          RedrawLabel(check_load->getLoad().average1m,labelnr);
          labelnr++;
        }
        if (resource->show.label_load5m)
        {
          RedrawLabel(check_load->getLoad().average5m,labelnr);
          labelnr++;
        }
        if (resource->show.label_load15m)
        {
          RedrawLabel(check_load->getLoad().average15m,labelnr);
          labelnr++;
        }
        if (resource->show.label_memused)
        {
          RedrawLabel(check_load->getNorm().mem_used,labelnr);
          labelnr++;
        }
        if (resource->show.label_swapused)
        {
          RedrawLabel(check_load->getNorm().swap_used,labelnr);
          labelnr++;
        }
        if (resource->show.label_totalused)
        {
          RedrawLabel(check_load->getNorm().total_used,labelnr);
          labelnr++;
        }
        if (resource->show.label_usercpu)
        {
          RedrawLabel(check_load->getCpu().user,labelnr);
          labelnr++;
        }
        if (resource->show.label_nicecpu)
        {
          RedrawLabel(check_load->getCpu().nice,labelnr);
          labelnr++;
        }
        if (resource->show.label_systemcpu)
        {
          RedrawLabel(check_load->getCpu().system,labelnr);
          labelnr++;
        }
        if (resource->show.label_idlecpu)
        {
          RedrawLabel(check_load->getCpu().idle,labelnr);
          labelnr++;
        }
	if (resource->show.label_networkin)
        {
          RedrawLabel(check_load->getNet().inchange,labelnr);
          labelnr++;
        }
	if (resource->show.label_networkout)
        {
          RedrawLabel(check_load->getNet().outchange,labelnr);
          labelnr++;
        }
    }    
        
    if (check_load->getBroken())
    {
        /* show broken */
    }
}

void ToolWindow::CheckConfig()
{
    struct stat file_status;

    if (stat(config_filename,&file_status)!=0)
    {
        fprintf(stderr,"Could not config file %s\n",
                config_filename);
    }
        else if(file_status.st_mtime != resource->style.mtime)
    {
        resource->style.mtime=file_status.st_mtime;
        reconfigure();
    }
}

void ToolWindow::process_event(XEvent *Event) {
  switch (Event->type) {
    case ClientMessage: {
      if ((unsigned)Event->xclient.data.l[0]==wm_delete_window) shutdown();
    }
    break;
    case Expose: {
      Redraw();
    }
    break;
    case ButtonPress: {
      if (Event->xbutton.button == LEFT_BUTTON) {
        if (!(raised)) {
          XRaiseWindow(getXDisplay(),framewin);
          raised=True;
        }
      } else if (Event->xbutton.button == MIDDLE_BUTTON) {
        if (raised) {
          XLowerWindow(getXDisplay(),framewin);
          raised=False;
        }
      }
    }
    break;
    case ButtonRelease: {
      if (Event->xbutton.button == RIGHT_BUTTON) {
        if (Event->xbutton.window==labelwin) 
          reconfigure();
      }
    }
    break;
 	 	case ConfigureNotify: {
     	if (Event->xconfigure.window==framewin && Event->xconfigure.send_event) {
 
        if (withdrawn) {
          reconfigure();
        }
        int parent_x,parent_y;
        Window parent_root;
        unsigned int parent_width;
        unsigned int parent_height;
        unsigned int parent_border_width;
        unsigned int parent_depth;
        if (withdrawn) {
          XGetGeometry(getXDisplay(),Event->xconfigure.above,&parent_root,
                     &parent_x,&parent_y,&parent_width,&parent_height,
                     &parent_border_width,&parent_depth);
          frame.x=Event->xconfigure.x+parent_x;
          frame.y=Event->xconfigure.y+parent_y;
        } else {
          frame.x=Event->xconfigure.x;
          frame.y=Event->xconfigure.y;
          if (decorated) {
            if (position==NULL) position= new char [13];
              sprintf(position,"+%i+%i",frame.x,frame.y);
          }
        }
      }
	 	}
  }
}
