/* $Id: ArkRay.cpp,v 1.7 2002/10/07 02:51:35 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
 #include <config.h>
#endif

#include <Ark/ArkRay.h>
#include <stdio.h>

#define EPSILON 0.000001
#define CROSS(dest,v1,v2) \
          dest[0]=v1[1]*v2[2]-v1[2]*v2[1]; \
          dest[1]=v1[2]*v2[0]-v1[0]*v2[2]; \
          dest[2]=v1[0]*v2[1]-v1[1]*v2[0];
#define DOT(v1,v2) (v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2])

#define SUB(dest,v1,v2) \
          dest[0]=v1[0]-v2[0]; \
          dest[1]=v1[1]-v2[1]; \
          dest[2]=v1[2]-v2[2];

/* Ray-Triangle Intersection Test Routines
 * Different optimizations of my and Ben Trumbore's 
 * code from journals of graphics tools (JGT)
 * http://www.acm.org/jgt/                    
 * by Tomas Moller, May 2000      
 */

extern "C" int
intersect_triangle1 (const scalar *orig,
		     const scalar *dir,
		     const scalar *vert0,
		     const scalar *vert1,
		     const scalar *vert2,
		     scalar *t, scalar *u, scalar *v)
{
   scalar edge1[3], edge2[3], tvec[3], pvec[3], qvec[3];

   /* find vectors for two edges sharing vert0 */
   SUB(edge1, vert1, vert0);
   SUB(edge2, vert2, vert0);

   /* begin calculating determinant - also used to calculate U parameter */
   CROSS(pvec, dir, edge2);

   /* if determinant is near zero, ray lies in plane of triangle */
   const scalar det = DOT(edge1, pvec);

   if (det > EPSILON)
   {
      /* calculate distance from vert0 to ray origin */
      SUB(tvec, orig, vert0);
      
      /* calculate U parameter and test bounds */
      *u = DOT(tvec, pvec);
      if (*u < 0.0f || *u > det)
	 return 0;
      
      /* prepare to test V parameter */
      CROSS(qvec, tvec, edge1);
      
      /* calculate V parameter and test bounds */
      *v = DOT(dir, qvec);
      if (*v < 0.0f || *u + *v > det)
	 return 0;
      
   }
   else if(det < -EPSILON)
   {
      /* calculate distance from vert0 to ray origin */
      SUB(tvec, orig, vert0);
      
      /* calculate U parameter and test bounds */
      *u = DOT(tvec, pvec);
      if (*u > 0.0f || *u < det)
	 return 0;
      
      /* prepare to test V parameter */
      CROSS(qvec, tvec, edge1);
      
      /* calculate V parameter and test bounds */
      *v = DOT(dir, qvec) ;
      if (*v > 0.0f || *u + *v < det)
	 return 0;
   }
   else return 0;  /* ray is parallell to the plane of the triangle */


   const scalar inv_det = 1.0f / det;

   /* calculate t, ray intersects triangle */
   *t = DOT(edge2, qvec) * inv_det;
   (*u) *= inv_det;
   (*v) *= inv_det;

   return 1;
}


namespace Ark
{

   enum Quadrant
   {
      Right, 
      Left, 
      Middle 
   };


   /*
    * Fast Ray-Box Intersection
    * by Andrew Woo
    * from "Graphics Gems", Academic Press, 1990
    */

   bool
   Ray::HitBBox (const BBox &box, Vector3 *vcoord) const
   {
      bool inside = true;
      Quadrant quadrant[3];
      register int i;

      /// Hum. dunno wether this will be optimized by the compiler...
      const scalar *minB = &box.m_Min.X;
      const scalar *maxB = &box.m_Max.X;
      const scalar *origin = &m_From.X;
      const scalar *dir = &m_Dir.X;
      scalar *coord = &vcoord->X;

      int whichPlane;
      scalar maxT[3];
      scalar candidatePlane[3];

      /* Find candidate planes; this loop can be avoided if
       * rays cast all from the eye(assume perpsective view)
       */
      for (i = 0; i < 3; i++)
      {
	 if(origin[i] < minB[i])
	 {
	    quadrant[i] = Left;
	    candidatePlane[i] = minB[i];
	    inside = false;
	 } 
	 else if (origin[i] > maxB[i])
	 {
	    quadrant[i] = Right;
	    candidatePlane[i] = maxB[i];
	    inside = false;
	 }
	 else
	 {
	    quadrant[i] = Middle;
	 }
      }
	 
      /* Ray origin inside bounding box */
      if (inside)
      {
	 *vcoord = m_From;
	 return true;
      }


      /* Calculate T distances to candidate planes */
      for (i = 0; i < 3; i++)
      {
	 if (quadrant[i] != Middle && dir[i] !=0.)
	    maxT[i] = (candidatePlane[i]-origin[i]) / dir[i];
	 else
	    maxT[i] = -1.;
      }

      /* Get largest of the maxT's for final choice of intersection */
      whichPlane = 0;
      for (i = 1; i < 3; i++)
	 if (maxT[whichPlane] < maxT[i])
	    whichPlane = i;
      
      /* Check final candidate actually inside box */
      if (maxT[whichPlane] < 0.)
	 return false;

      for (i = 0; i < 3; i++)
      {
	 if (whichPlane != i)
	 {
	    coord[i] = origin[i] + maxT[whichPlane] * dir[i];

	    if (coord[i] < minB[i] || coord[i] > maxB[i])
	       return false;
	 }
	 else
	 {
	    coord[i] = candidatePlane[i];
	 }
      }
      
      /* ray hits box */
      return true;
   }

   bool
   Ray::HitTriangle (const Vector3 &vert0,
		     const Vector3 &vert1,
		     const Vector3 &vert2,
		     Vector3 *coord) const
   {     
      scalar t, u, v;
      int ret = ::intersect_triangle1 (&m_From.X, &m_Dir.X,
				       &vert0.X, &vert1.X, &vert2.X,
				       &t, &u, &v);
      
      if (ret == 0)
	 return false;

      *coord = Vector3 (t * m_Dir.X, t * m_Dir.Y, t * m_Dir.Z);
      (*coord) += m_From;

      return true;
   }

}

