/* $Id: ArkLexer.cpp,v 1.20 2002/10/11 01:10:03 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <iostream>
#include <sstream>
#include <string>

#include <Ark/ArkLexer.h>
#include <Ark/ArkSystem.h>

namespace Ark
{

Lexer::Lexer (const String& name, Stream& file) :
    m_Name (name),
    m_File (file),
    m_LastToken (),
    m_LastChar (-1),
    m_CurrentLine (),
    m_LinePosition (0),
    m_LineSize (0),
    m_LineNumber (0),
    m_TokenAvail (false),
    m_CharAvail (false)
    
{
}

Lexer::~Lexer ()
{}

int Lexer::GetChar ()
{

    if (m_CharAvail)
    {
	m_CharAvail = false;
	return m_LastChar;
    }


    if ( m_LineSize <= m_LinePosition )
    {
	do 
	{
	    ++m_LineNumber;
	    std::getline(m_File, m_CurrentLine);
	} 
	while ( (! m_File.fail() ) && m_CurrentLine.empty() );

	if ( m_File.fail() )
	{
	    // could not send the character
	    m_LastChar = -1;
	    return -1;
	}

	m_LineSize = m_CurrentLine.size();
	m_LinePosition = 0;

	// FIXME hack to be able to return a CR
	m_LastChar = '\n';
	return '\n';
    }

    uchar c = m_CurrentLine[m_LinePosition];
    ++m_LinePosition;
    m_LastChar = c;
    return c;

}

void Lexer::UngetChar ()
{
  assert (m_CharAvail == false);

  m_CharAvail = true;
}

void Lexer::UngetToken ()
{
  assert (m_TokenAvail == false);

  m_TokenAvail = true;
}

int Lexer::SkipBlanks ()
{
  int c = GetChar ();

  while (c != -1)
  {
    while (isspace (char(c)) && c >= 0)
      c = GetChar ();

    if (c == '/')
    {
      c = GetChar ();

      if (c == '*')
      {
        /* Skip C-style comment */
        int lc = 0;

        c = GetChar ();

        while (c != -1 && ! (c == '/' && lc == '*'))
        {
          lc = c;
          c = GetChar();
        }

        c = GetChar();
        continue;
      }
      else if (c == '/')
      {
        /* Skip C++-style comment */

        do
          c = GetChar ();
        while (c != -1 && c != '\n');

        c = GetChar();
        continue;
      }
      else
      {
        UngetChar ();
        return '/';
      }
    }
    else return c;
  }

  return c;
}

Ark::String Lexer::GetToken (Type type)
{
   Ark::String token;
   Type tp = UNKNOW;

   if (m_TokenAvail)
   {
      m_TokenAvail = false;
      return m_LastToken;
   }
   
  int c = SkipBlanks ();

  if (c == -1)
    return String();

  if (isalpha (char(c)) || char(c) == '_')
  {
    do
    {
      token.push_back(char(c));
      c = GetChar ();
    } while (isalnum (char(c)) || char(c) == '_');
    
    UngetChar ();
    tp = IDENTIFIER;
  }
  else if (isdigit (char(c)) || char(c) == '-')
  {
    int i = 0;
    int c2 = GetChar ();

    if (char(c2) == 'x' && c == '0')
    {
       token.push_back(c);
       c = c2;
       do
       {
	  token.push_back(char(c));
	  c = GetChar ();
       } while (isxdigit (char(c)));
    }
    else
    {
       UngetChar();
       do
       {
	  token.push_back(char(c));
	  c = GetChar ();
       } while (isdigit (char(c)) || char(c) == '.');

       if (char(c) == 'e' || char(c) == 'E')
       {
	  i = 0; c = 'e';
	  do
	  {
	     token.push_back(char(c));
	     c = GetChar ();
	     i++;
	  } while (isdigit (char(c)) ||
		   ((char(c) == '-' || char(c) == '+') && i == 1));
       }  
    }

    UngetChar();
    tp = CONSTANT;
  }
  else if (char(c) == '"')
  {
    char lc = 0;
    bool encounter_bs = false;

    do
    {
      token.push_back(char(c));

      lc = char(c);
      c = GetChar ();

      if (lc == '\\' && !encounter_bs) encounter_bs = true;
      else encounter_bs = false;
    } while (char(c) != '"'
	     || (char(c) == '"' && encounter_bs));

    token.push_back('"');
    tp = STRING;
  }
  else if (strchr ("{}()=,;|:", c))
  {
    token.push_back(char(c));
    tp = SYMBOL;
  }

  m_LastToken = token;

  if (type != UNKNOW && tp != type && !m_LastToken.empty())
  {
    m_TokenAvail = true;
    Error (String("parse error : unexpected token '") + 
	   QuoteString(m_LastToken) + "'");
    return "";
  }

  return m_LastToken;
}


bool
Lexer::CheckToken(const String &waitfor)
{
  String token = GetToken();

  if (token != waitfor)
  {
    UngetToken();

    std::ostringstream os;
    os << "Got '" << token << "' while waiting for '" << waitfor << "'";
    Error( os.str() );
    return false;
  }

  return true;
}

bool
Lexer::ReadInteger(int *to)
{
    assert (to != NULL);

    String val = GetToken (CONSTANT);

    return Ark::StringToInt(val, *to);
}

bool
Lexer::ReadScalar(scalar *to)
{
    assert (to != NULL);

    String val = GetToken (CONSTANT);

    return Ark::StringToScalar(val, *to);
}

bool
Lexer::ReadScalarVector (scalar *to, size_t n)
{
   assert (to != NULL);

   if (!CheckToken ("{"))
      return false;

   for (size_t i = 0; i < n; i++)
   {
       if (! ReadScalar(&to[i]) )
	   return false;
      
      if ((i != n-1) && !CheckToken (","))
	 return false;
   }
      
   return CheckToken ("}");
}

void
Lexer::Error (const String& desc) const
{
  Sys()->Warning ("%s, line %d: %s", m_Name.c_str(), m_LineNumber, desc.c_str());
}

}
