# -*- coding: utf-8 -*-
#####################################################################
#  Rafael Proença <cypherbios@ubuntu.com>
#
#  Copyright 2006 APTonCD DevTeam.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published
#  by the Free Software Foundation; version 2 only.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#####################################################################

import os
import string
import apt_inst
import apt_pkg
import shutil
import re

from APTonCD.core import constants
from APTonCD.core import utils
from APTonCD.core.utils import humanize_file_size

class DebPackage(object):
        """
          This class holds packages info stored in deb package file. 
        """
        def __init__(self, package_file = '', package_path = ''):
            self._deb_file = ''
            self._location = ''
            self._full_path = ''
            self._remote_path = ''
            self._package = ''
            self._package_size = 0
            self._package_size_text = ''
            self._version = ''
            self._section = ''
            self._priority = ''
            self._architecture = ''
            self._maintainer = ''
            self._short_description = ''
            self._long_descripton = ''
            self._description = ''
            self._depends = []
            self._destination = ''
            self._custom_package = False
            self._bad_package = False
            self._selected = False
            self._installed = False
            self._old = False
            apt_pkg.init()
            
            if package_file != '':
                self.load_package(package_file, package_path)
               
        def _get_package_section(self, sections, section_name):
            """
                Get section information from package.
            """
            if sections.has_key(section_name):
                result = sections[section_name]
                if isinstance(result,str):
                    try:
                        return result.decode('utf-8')
                    except (UnicodeEncodeError, UnicodeDecodeError), e:
                        #print "Invalid UTF-8 string: %s, em %d-%d" % (e.reason, e.start, e.end) , '\n', result
                        #change char ® for it's hex value so we could show it without errors
                        result = result.replace(result[e.start],'\xc2\xae') 
                        return result
                
                else:
                    return result
            else:
                return ''
            
        def load_package(self, filename, path = ''):
            """
                Extract package information from deb file.
            """
            try:
                if path == '':
                    self._location = constants.LOCAL_APT_FOLDER
                    self._deb_file = os.path.basename(filename)
                    self._full_path = os.path.join(self._location, filename)
                    
                else:
                    self._location = path
                    self._deb_file = os.path.basename(filename)
                    self._full_path = os.path.join(self._location, filename)
                    
                self._package_size = os.path.getsize(self._full_path)
                self._package_size_text = humanize_file_size(self._package_size)
                
                # now try to get additional packages information from itself
                local = apt_inst.debExtractControl(open(self._full_path))
                
                #get pacakges sections
                sections = apt_pkg.ParseSection(local)
                
                self._package = self._get_package_section(sections, 'Package')
                self._section = self._get_package_section(sections, 'Section')
                self._architecture = self._get_package_section(sections, 'Architecture')
                self._version = self._get_package_section(sections,'Version')
                self._maintainer = self._get_package_section(sections, 'Mantainer')
                self._priority = self._get_package_section(sections, 'Priority')
                self._description = self._get_package_section(sections,'Description')
                
                idx = self._description.find("\n")
                
                self._long_description = self._description[idx + 1:]
                self._short_description = self._description[:idx]
                
                #get packages whom the current packages depends of
                for key in ["Depends","PreDepends"]:
                    if sections.has_key(key):
                        self.depends = self._parse_package_depends(apt_pkg.ParseDepends(sections[key]))
                
                self._custom_package = False
                self._bad_package = False
                self._selected = False
                self._installed = False
                self._old = False
                return True
                
            except Exception, err:
                print str(err)
                self._package = filename
                self._section = ''
                self._architecture = ''
                self._version = 0
                self._package_size = 0
                self._package_size_text = humanize_file_size(self._package_size)
                self._maintainer = ''
                self._priority = ''
                self._description = constants.MESSAGE_0050
                self._long_description = constants.MESSAGE_0050
                self._short_description = constants.MESSAGE_0051
                self.depends = []
                self._custom_package = False
                self._bad_package = True
                self._selected = False
                self._installed = False
                self._old = False
                return False
                
        def _parse_package_depends(self,pkg_list):
            """
                Returns a list of packages names whom this package is dependent.
            """
            #split package name and version
            regex = re.compile(r"(?si)\'([a-z].*?)\'", re.DOTALL )
            dep_list = []
            rep_str = "%s" % pkg_list
            try:
                g = regex.findall(rep_str)
            except Exception, error:
                print self._package, str(error)
            for n in g:
                dep_list.append(n)
                
            return dep_list
        
        def compare_version(self,version):
            """
                Compare packages versions.
            """
            if not self._bad_package:
                return apt_pkg.VersionCompare(self._version,version)
            else:
                return -1
            
        def copy_to(self, destination = '', asRoot = False, root_shell = None):
            """
                Copy a package to an specified location or to package's destination property value.
            """
            try:
                if destination == '':
                    if not asRoot:
                        shutil.copyfile(self._full_path, self._destination)
                    else:
                        command = 'cp %s %s' % (self._full_path, destination)
                        print command
                        root_shell.execute(command)
                else:
                    if not asRoot:
                        shutil.copyfile(self._full_path, destination)
                    else:
                        #command = 'cp %s %s' % (self._full_path, destination)
                        command = 'cp %s %s' % (self._full_path, destination)
                        print command
                        root_shell.execute(command)
                return True
            except Exception, e:
                print str(e)
                return False

        #package name
        def _get_package(self): return self._package
        def _set_pakcage(self, value): self._package = value
        package = property(fget = _get_package, fset = _set_pakcage, doc = 'Get/Set Package name.')
        
        #package priority
        def _get_priority(self): return self._priority
        def _set_priority(self, value): self._priority = value
        priority = property(fget = _get_priority, fset = _set_priority, doc = 'Get/Set Package Priority.')
        
        # package section
        def _get_section(self): return self._section
        def _set_section(self, value): self._section = value
        section = property(fget = _get_section, fset = _set_section, doc = 'Get/Set Package Section.')
        
        #package maintainer
        def _get_maintainer(self): return self._maintainer
        def _set_maintainer(self, value): self._maintainer = value
        maintainer = property(fget = _get_maintainer, fset = _set_maintainer, doc = "Get/Set Package Maintainer's name.")
        
        #package architecture
        def _get_architecture(self): return self._architecture
        def _set_architecture(self, value): self._architecture = value
        architecture = property(fget = _get_architecture, fset = _set_architecture, doc = 'Get/Set Package Architecture.')
        
        #package version
        def _get_version(self): return self._version
        def _set_version(self, value): self._version = value
        version = property(fget = _get_version, fset = _set_version, doc = 'Get/Set Package Version.')
        
        #packages whom the current packages depends of
        def _get_depends(self): return self._depends
        def _set_depends(self, value): self._depends = value
        depends = property(fget = _get_depends, fset = _set_depends, doc = 'Get/Set Package Depends.')
        
        #pacage deb file name
        def _get_deb_file_name(self): return self._deb_file
        def _set_deb_file_name(self, value): self._deb_file = value
        deb_filename = property(fget = _get_deb_file_name, fset = _set_deb_file_name, doc = 'Get/Set Package Deb Filename.')
        
        def _get_deb_full_file_name(self): return self._full_path
        def _set_deb_full_file_name(self, value): self._full_path = value
        deb_full_filename = property(fget = _get_deb_full_file_name, fset = _set_deb_full_file_name, doc = 'Get/Set Package Deb Filename.')
        
        #package size
        def _get_size(self): return self._package_size
        def _set_size(self, value): self._package_size = value
        size = property(fget = _get_size, fset = _set_size, doc = 'Get/Set Package Size.')
        
        #package size
        def _get_size_text(self): return self._package_size_text
        def _set_size_text(self, value): self._package_size_text = value
        size_text = property(fget = _get_size_text , fset = _set_size_text, doc = 'Get/Set Package Size Text.')

        #package full description
        def _get_description(self): return self._description
        def _set_description(self, value): self._description = value
        description = property(fget = _get_description, fset = _set_description, doc = 'Get/Set Package Full Description.')
        
        #package full description
        def _get_short_description(self): return self._short_description
        def _set_short_description(self, value): self._short_description = value
        short_description = property(fget = _get_short_description, fset = _set_short_description, doc = 'Get/Set Package Short Description.')
        
        #package full description
        def _get_long_description(self):
        	try:
        		long_desc = ""
        		raw_desc = string.split(self._long_description, "\n")
        		for line in raw_desc:
        			tmp = string.strip(line)
        			if tmp == ".":
        				long_desc += "\n"
        			else:
        				long_desc += tmp + "\n"
        		return long_desc
        	except KeyError:
        		long_desc = "No description is available"
            
        	return long_desc
        	
        def _set_long_description(self, value): self._long_description = value
        long_description = property(fget = _get_long_description, fset = _set_long_description, doc = 'Get/Set Package Long Description.')
        
        #Local package location
        def _get_location(self):    return self._location
        def _set_location(self,value):    self._location = value
        source_path = property(fget = _get_location, fset = _set_location, doc = 'Get/Set Package Location.')
        
        #package remote path
        def _get_remote_path(self): return self._remote_path
        def _set_remote_path(self,value): self._remote_path = value
        remote_source = property(fget = _get_remote_path, fset = _set_remote_path, doc = 'Get/Set Package Remote Source.')
        
        #package destination on copy operation
        def _get_destination(self): return self._destination
        def _set_destination(self, value): self._destination = value
        destination = property(fget = _get_destination, fset = _set_destination, doc='Get/Set Package Destination file and path.')
       
        #Package marked as custom
        def _get_is_custom(self): return self._custom_package
        def _set_is_custom(self, value): self._custom_package = value
        custom = property(fget = _get_is_custom, fset = _set_is_custom, doc = 'Get/Set package as a custom package.')
        
        #package is marked as bad if it coudn't be read or for another reason
        def _get_is_bad(self): return self._bad_package
        def _set_is_bad(self, value): self._bad_package = value
        bad = property(fget = _get_is_bad, fset = _set_is_bad, doc = 'Get/Set a package as a bad package.')
        
        #package selected or marked 
        def _get_selected(self): return self._selected
        def _set_selected(self, value): self._selected = value
        selected = property(fget = _get_selected, fset = _set_selected, doc = 'Get/Set a package as selected.')
        
        #is package installed?
        def _get_installed(self): return self._installed
        def _set_installed(self, value): self._installed = value
        installed = property( fget = _get_installed, fset = _set_installed , doc = 'Get/Set package as installed or not.')
        
        def _get_old_version(self): return self._old
        def _set_old_version(self, value): self._old = value
        old = property(fget = _get_old_version, fset = _set_old_version, doc = 'Get/Set package version as an old version.')
        
        def get_packages_list(self):
            """
                Returns package properties as a python list.
            """
            #used this way to better add new itens on structure changes
            p_list = []
            p_list.append(self._deb_file)
            p_list.append(self._location)
            p_list.append(self._full_path)
            p_list.append(self._remote_path)
            p_list.append(self._package)
            p_list.append(self._package_size)
            p_list.append(self._package_size_text)
            p_list.append(self._version)
            p_list.append(self._section)
            p_list.append(self._priority)
            p_list.append(self._architecture)
            p_list.append(self._maintainer)
            p_list.append(self._short_description)
            p_list.append(self._long_descripton)
            p_list.append(self._description)
            p_list.append(self._depends)
            p_list.append(self._destination)
            p_list.append(self._custom_package)
            p_list.append(self._bad_package)
            p_list.append(self._selected)
            p_list.append(self._installed)
            return p_list
    
