//============================================================================
// Name        : acngfs.cpp
// Author      : Eduard Bloch
// Description : Simple FUSE-based filesystem for HTTP access (apt-cacher NG)
//============================================================================


#define FUSE_USE_VERSION 25

#include <fuse.h>
#include "meta.h"
#include "dlcon.h"
#include "header.h"
#include "filestorage.h"
#include <pthread.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>    
#include <unistd.h>
#include <inttypes.h>
#include <stdint.h>
#include <algorithm>
#include <iostream>

#include <set>

#define LOCAL_DEBUG
#include "debug.h"


#define MAXMEM 512000

using namespace std;

namespace acfg
{
tHttpUrl proxy_info; // empty for now. TODO: parse environment therein?
int dnscachetime(3600);
// a FAT buffer, helps avoiding multiple calls with the hot buffer
int dlbufsize(512000);
string agentname("mount.acng"), remoteport("3142");
}

namespace aclog
{
void err(const char*msg, const char*)
{
	fprintf(stderr, "%s\n", msg);
}
}

static struct stat statTempl;
static struct statfs stfsTemp;
static tHttpUrl baseUrl;

// Caching for one download agent
lockable cacheLock;
vector<dlcon*> vpCachedDler;
dlcon * GetDler() {
	lockguard g(cacheLock);
	if(vpCachedDler.empty())
		return new dlcon;
	dlcon *ret=vpCachedDler.back();
	vpCachedDler.pop_back();
	return ret;
}

void ReturnDler(dlcon *d)
{
	lockguard g(cacheLock);
	d->Reset();
	if(vpCachedDler.size()>=4)
	{
		// some refresh shouldn't hurt
		delete vpCachedDler[0];
		vpCachedDler[0]=d;
	}
	else
	{
		vpCachedDler.push_back(d);
	}
}



lockable mxDirNameCacheLock;
set<string> dirNameCache;
inline bool IsDir(const string &s)
{
	lockguard g(mxDirNameCacheLock);
	return dirNameCache.end() != dirNameCache.find(s);  
}
inline void learnParentDirs(const string & s)
{
	lockguard g(mxDirNameCacheLock);
	tStrPos pos(0);
	while(true)
	{
		pos=s.find('/', pos+1);
		if(pos==string::npos)
			return;
		dirNameCache.insert(s.substr(0, pos));
		//ldbg("### Learned dir: " << s.substr(0, pos));
	}
}

lockable hotBufLock;
string sHotBufId;
vector<char> vHotBuf;

/*
lockable hotHeadLock;
time_t stampHead(0);
string sHotHeadId;
header hotHead;
*/

// simple wrapper class for dlcon callbacks
class dlpassthrough : public dlstorage, public condition
{
public:
	header m_head;
    //char *rbuf;
    // this sucks, is not LFS ready
    //size_t nFileSize, nVirtStart, nVirtEnd;
    uint64_t m_nFileSize, m_nStreamPos, m_nWantedPos;
    const char *m_pCurBuf;
    size_t m_nCurSize;
    bool m_bError;
    bool m_bTerminating;
    string m_sPath;
    
	void Reset() 
	{
		m_bError=false;
		m_bTerminating=false;
		m_pCurBuf=NULL;
		m_nCurSize=0;
		
		//nFileSize=nVirtStart=nVirtEnd=0;
		m_nFileSize=m_nStreamPos=m_nWantedPos=0;
		m_head.clear();
	}
	
	dlpassthrough(const string & sPath) : m_sPath(sPath)
	{
		Reset();
		//rbuf=new char[MAXMEM];
	}
	~dlpassthrough()
	{
		/*if(rbuf)
		{
			delete [] rbuf;
			rbuf=0;
		}
		*/
	}
	
	virtual bool StoreNewHead(const header & head) {
		
		setLockGuard;
		
		m_nFileSize=atol(head.get("Content-Length").c_str());
		m_head=head;
		
		ldbg("Got head. Status? " << m_head.getStatus() << " und size? " << 
				m_nFileSize << " contents: " << m_head.as_string(false));
		
		m_bError=(m_nFileSize<=0);
		notifyAll();
		return true;
	}
	virtual long StoreData(const char *data, const unsigned int size, off_t nOffset)
	{
		if(size==0)
			return 0;
		
		if(m_bTerminating)
			return -1; // MUST abort the downloader here
				
		setLockGuard;
		dbgline;
				
		m_pCurBuf=data;
		m_nCurSize=size;
		m_nStreamPos=nOffset;
		
		if(0==nOffset)
		{
			lockguard g(hotBufLock);
			sHotBufId=m_sPath;
			vHotBuf.resize(size);
			memcpy(&vHotBuf[0], data, size);
			ldbg("### cached file start chunk, size: " << size << " for id: " << m_sPath );
		}
		
		notifyAll();
		dbgline;
		while(m_pCurBuf && !m_bTerminating)
			wait();
		dbgline;
		if(m_bTerminating)
			return -1; // MUST abort the downloader here
		dbgline;
		return size;
	}

	virtual bool AssignDownloaderUnlessConflicting()
	{
		return true;
	}
	
	virtual bool ReleaseDownloaderUnlessBusy()
	{
		// downloader wants to stop? Let him and remember the failure
		m_bError=true;
		return true;
	}
	
	virtual void Finalize(long nNewContLen) {
		// nothing, blind/without-cont.len. downloads not supported
	}
	
	virtual void SetFailureMode(MYSTD::string const & FailureMessage, bool bNonFatal=false)
	{
		setLockGuard;
		m_bError=true;
		notifyAll();
	}
	virtual void AddTransferCount(UINT) 
	{
		
	}
	virtual void GetStatusEx(long & nCachedLen, long &nContLen, bool & bIsDynType) 
	{
		// resume at the position specified by parent
		nCachedLen=m_nWantedPos;
		nContLen=-1; // unknown
		bIsDynType=false;
	}
	virtual MYSTD::string ToString() {
		return m_head.as_string(false);
	}
	virtual void GetHeader(header &hOut)
	{
		hOut=m_head;
	}
	void SignalStop()
	{
		ldbg("Setting item termination flag...");
		setLockGuard;
		m_bTerminating=true;
		notifyAll();
	}
};



void * _StartDownloader(void *pVoidDler)
{
	static_cast<dlcon*>(pVoidDler) -> WorkLoop();
	return NULL;
}

class dlobj : public lockable {

public:
	dlcon *m_pDlClient;
	SHARED_PTR<dlpassthrough> pItem;
	tHttpUrl url;
	bool bDlDetached;
	pthread_t m_dlerthr;
	
	/*
	inline void Reset(const string &sPath)
	{
		url=baseUrl;
		url.sPath+=sPath;
	}
	*/

	inline void KillDl()
	{
		setLockGuard;
		pItem->SignalStop();
		dbgline;
		m_pDlClient->SignalStop();
		dbgline;
		pthread_join(m_dlerthr, NULL);
		pItem->Reset();
		
		// TODO: better not removing and only reseting?
		//delete m_pDlClient;
		//m_pDlClient=NULL;
		m_pDlClient->Reset();
		
		ldbg("DL stopped");
		
		bDlDetached=false;
		
	}
	
	inline bool StartDl(size_t offset)
	{
		setLockGuard;
		
		if(!m_pDlClient)
			m_pDlClient=GetDler();
				
		if (!bDlDetached)
		{
			ldbg("StartDL");
			pItem->m_nWantedPos=offset;
			m_pDlClient->AddJob(false, pItem, url);

			if (0==pthread_create(&m_dlerthr, NULL, _StartDownloader,
					(void *)m_pDlClient))
				bDlDetached=true;
			else
				return false;
		}
		return true;
	}
	
	dlobj(const string & sPath) : m_pDlClient(NULL), bDlDetached(false)
	{
		m_pDlClient=GetDler();
		url=baseUrl;
		url.sPath+=sPath;
		pItem.reset(new dlpassthrough(url.sPath));
		//Reset(sPath);
	}
	~dlobj()
	{
		if(bDlDetached)
			KillDl();
		
		if(m_pDlClient)
			ReturnDler(m_pDlClient);
	}
	
	/*void GetHead()
	{
		setLockGuard;
		if (bDlDetached)
		{
			lockguard g(*pItem);
			while (!pItem->m_bError && pItem->m_nFileSize==0)
				pItem->wait();
			return;
		}
		
		//lockguard g(*pItem);
		m_pDlClient->AddJob(false, pItem, url, true);
		m_pDlClient->WorkLoop(true);
	}*/

	bool IsFile() {
		
		if(IsDir(url.sPath))
		{
			//ldbg("### Is Directory, from cache");
			return false;
		}
		/*
		bool bGotHead(false);
		{
			lockguard g(hotHeadLock);
			if(sHotHeadId==url.sPath && time(NULL)< stampHead+10)
			{
				bGotHead=true;
				pItem->StoreNewHead(hotHead);
			}
		}
		
		bool bRet;
		
		if (!bGotHead)
		{*/
//////////////// this is the only effective code, everything around is caching
			// don't ever interfere with the running one, only reuse the file item wrapper
			dlcon *checker = GetDler();
			checker->AddJob(false, pItem, url, true);
			checker->WorkLoop(true);
			bool bRet=( !pItem->m_bError && pItem->m_head.getStatus()==200
					&& pItem->m_nFileSize>0);
			ReturnDler(checker);
/////////////////////////////////////////////////////////////////////////////
			/*
			lockguard g(hotHeadLock);
			stampHead=time(NULL);
			sHotHeadId=url.sPath;
			hotHead=pItem->m_head;
					
		}
		*/
			
		if(bRet)
			learnParentDirs(url.sPath);
		
		return bRet;
		
	}
	void GetFstat(struct stat *stbuf)
	{
		memcpy(stbuf, &statTempl, sizeof(statTempl));
		stbuf->st_mode &= ~S_IFMT; // delete mode flags and set them as needed
		// assume that everything downloadable is a file and a directory is 404
		if(IsFile())
		{
			ldbg("Is a file!");
			stbuf->st_mode |= S_IFREG;
			stbuf->st_size=pItem->m_nFileSize;
		}
		else
		{
			ldbg("Is a directory!");
			stbuf->st_mode |= S_IFDIR;
			stbuf->st_size=4;
		}
	}
	
	const char * GetDataAt(size_t offset, size_t &howMuch)
	{
		
		{
			lockguard g(hotBufLock);
			ldbg("### any cached data for " << url.sPath << " at " << offset << "?");
			if(url.sPath==sHotBufId && offset<vHotBuf.size())
			{
				howMuch=vHotBuf.size()-offset;
				return (&vHotBuf[0]) + offset; 
			}
		}
		
		reconnected:
		if (!StartDl(offset))
			return NULL;

		dbgline;
		
		lockguard g(*pItem);
		if(offset>=pItem->m_nFileSize)
		{
			howMuch=0;
			return "";
		}
		while (true)
		{
			dbgline;
			if (pItem->m_bError)
				return NULL;
			dbgline;
			
			const char *p=pItem->m_pCurBuf;
			if (p)
			{
				UINT cursz=pItem->m_nCurSize;
				size_t stpos=pItem->m_nStreamPos;
				ldbg("bufsize:" << cursz << " at streampos: " << stpos<< " vs. req. offset: " << offset);

				if (offset>=stpos+cursz)
				{
					dbgline;
					// not there yet, continue getting stuff
					pItem->m_pCurBuf=NULL;
					pItem->notifyAll();
					pItem->wait();
					
					dbgline;
					continue;
				}

				if (offset < stpos)
				{
					ldbg("OOB!");
					// out of bounds...
					// TODO: temporarer
					//return NULL;
					pItem->unlock();
					KillDl();
					pItem->lock(); // for the guard
					goto reconnected;
				}

				//if (pItem->m_nStreamPos > offset)
				dbgline;
				ASSERT(offset>=stpos);
				howMuch=stpos+cursz-offset;
				size_t nBufOff=offset-stpos;
				ldbg("Return: " << howMuch << " bytes, buf.offset: " << nBufOff);
				return p+nBufOff;
			}
			pItem->wait();

		}
		ASSERT(!"not reached, must be returned before");
	}
};

/// If found as downloadable, present as a file, or become a directory otherwise.
static int acngfs_getattr(const char *path, struct stat *stbuf)
{
   dlobj p(path);
   p.GetFstat(stbuf);
   return 0;
}

static int acngfs_fgetattr(const char *, struct stat *stbuf,
      struct fuse_file_info *fi)
{
	// quick and dirty... this is a file, it has been opened before
	memcpy(stbuf, &statTempl, sizeof(statTempl));
	stbuf->st_mode &= ~S_IFMT; // delete mode flags and set them as needed

	stbuf->st_mode |= S_IFREG;
	stbuf->st_size=((dlobj*)fi)->pItem->m_nFileSize;
	return 0;
}

static int acngfs_access(const char *path, int mask)
{
	// non-zero (failure) when trying to write
   return mask&W_OK;
}

static int acngfs_readlink(const char *path, char *buf, size_t size)
{
   return -EINVAL;
}

static int acngfs_opendir(const char *path, struct fuse_file_info *fi)
{
	// let FUSE manage directories
	return 0;
}

static int acngfs_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
      off_t offset, struct fuse_file_info *fi)
{
   /*
   map<string, struct dirent> smap;
   DIR *dp = opendir((fromdir+path).c_str());
   if(!dp)
      return 0;
   struct dirent *de;
   while(true) {
      de=readdir(dp);
      if(!de)
         break;
      smap[de->d_name]=*de;
   }
   closedir(dp);
   map<string, struct dirent>::iterator tmpit, it=smap.begin();
   string toskip;

proc_next_no_inc:
   for(;it!=smap.end();it++) {
      if(!toskip.empty()) {
         if(toskip!=it->first)
            toskip.clear();
         else {
            tmpit=it;
            it++;
            smap.erase(tmpit);
            // check next name
            incName(toskip);
            goto proc_next_no_inc;
         }
      }
      string key=it->first; // the reported name
      string::size_type s=it->first.size();
      if(s>3 && it->first.substr(s-3, 3)==".aa") { // hit, look for company
         toskip=it->first;
         key.erase(s-3, s);
         incName(toskip);
      }
      struct stat st;
      memset(&st, 0, sizeof(st));
      st.st_ino = it->second.d_ino;
      st.st_mode = it->second.d_type << 12;
      if (filler(buf, key.c_str(), &st, 0))
         break;
   }

   return 0;
   */
   
	return -EPERM;
}

static int acngfs_releasedir(const char *path, struct fuse_file_info *fi)
{

   /*
   DIR *dp = get_dirp(fi);
   (void) path;
   closedir(dp);
   */
   return 0;
}

static int acngfs_mknod(const char *path, mode_t mode, dev_t rdev)
{
	return -EROFS;
}

static int acngfs_mkdir(const char *path, mode_t mode)
{
   return -EROFS;
}

static int acngfs_unlink(const char *path)
{
   return -EROFS;
}

static int acngfs_rmdir(const char *path)
{
   return -EROFS;
}

static int acngfs_symlink(const char *from, const char *to)
{
  return -EROFS;
}

static int acngfs_rename(const char *from, const char *to)
{
  return -EROFS;
}

static int acngfs_link(const char *from, const char *to)
{
	return -EROFS;
}

static int acngfs_chmod(const char *path, mode_t mode)
{
   return -EROFS;
}

static int acngfs_chown(const char *path, uid_t uid, gid_t gid)
{
return -EROFS;
}

static int acngfs_truncate(const char *path, off_t size)
{
   return -EROFS;
}

static int acngfs_ftruncate(const char *path, off_t size,
      struct fuse_file_info *fi)
{
return -EROFS;
}

static int acngfs_utime(const char *path, struct utimbuf *buf)
{
 return -EROFS;
}

static int acngfs_open(const char *path, struct fuse_file_info *fi)
{
	if (fi->flags & (O_WRONLY|O_RDWR|O_TRUNC|O_CREAT))
		return -EROFS;

	dlobj *p = new dlobj(path);
	if (!p->IsFile())
	{
		delete p;
		return -EISDIR;
	}

	fi->fh = (uintptr_t) p;
	return 0;
}


static int acngfs_read(const char *path, char *buf, size_t size, off_t offset,
      struct fuse_file_info *fi)
{
   dlobj *p = (dlobj*) fi->fh;
	uint64_t sofar(0);
	ldbg("Get data... " << size);
			
	while (size>0)
	{
		size_t nHowMuch;
		const char *dbuf = p->GetDataAt(offset+sofar, nHowMuch);
		if (!dbuf)
			return -EIO;
		if (nHowMuch>size)
			nHowMuch=size;
		ldbg("Got " << nHowMuch << " for offset: " << offset << " got sofar: " << sofar);
		//ldbg("data: " << dbuf);

		if (nHowMuch==0)
			break;
		
		memcpy(buf+sofar, dbuf, nHowMuch);
		size-=nHowMuch;
		sofar+=nHowMuch;
		
		if(size==0)
			break;
		
	}
   return sofar;
}

static int acngfs_write(const char *path, const char *buf, size_t size,
      off_t offset, struct fuse_file_info *fi)
{
	return -EBADF;
}

static int acngfs_statfs(const char *path, struct statvfs *stbuf)
{
   memcpy(stbuf, &stfsTemp, sizeof(stbuf));
	return 0;
}

static int acngfs_release(const char *path, struct fuse_file_info *fi)
{
	dlobj *p=(dlobj*)fi->fh;
	delete p;
	return 0;
}

static int acngfs_fsync(const char *path, int isdatasync,
      struct fuse_file_info *fi)
{
   return 0;
}


struct fuse_operations acngfs_oper;

void _ExitUsage() {
   cerr << "USAGE: httpmfs URL mountPoint [FUSE Mount Options]\n"
        << "valid FUSE Mount Options follow:\n";
    const char *argv[] = {"...", "-h"};
    fuse_main( 2, const_cast<char**>(argv), &acngfs_oper);
    exit(EXIT_FAILURE);
}

#define barf(x) { cerr <<endl << "ERROR: " << x <<endl<<endl; _ExitUsage(); }

int main(int argc, char *argv[])
{

   memset(&acngfs_oper, 0, sizeof(acngfs_oper));

   acngfs_oper.getattr	= acngfs_getattr;
   acngfs_oper.fgetattr	= acngfs_fgetattr;
   acngfs_oper.access	= acngfs_access;
   acngfs_oper.readlink	= acngfs_readlink;
   acngfs_oper.opendir	= acngfs_opendir;
   acngfs_oper.readdir	= acngfs_readdir;
   acngfs_oper.releasedir	= acngfs_releasedir;
   acngfs_oper.mknod	= acngfs_mknod;
   acngfs_oper.mkdir	= acngfs_mkdir;
   acngfs_oper.symlink	= acngfs_symlink;
   acngfs_oper.unlink	= acngfs_unlink;
   acngfs_oper.rmdir	= acngfs_rmdir;
   acngfs_oper.rename	= acngfs_rename;
   acngfs_oper.link	= acngfs_link;
   acngfs_oper.chmod	= acngfs_chmod;
   acngfs_oper.chown	= acngfs_chown;
   acngfs_oper.truncate	= acngfs_truncate;
   acngfs_oper.ftruncate	= acngfs_ftruncate;
   acngfs_oper.utime	= acngfs_utime;
//   acngfs_oper.create	= acngfs_create;
   acngfs_oper.open	= acngfs_open;
   acngfs_oper.read	= acngfs_read;
   acngfs_oper.write	= acngfs_write;
   acngfs_oper.statfs	= acngfs_statfs;
   acngfs_oper.release	= acngfs_release;
   acngfs_oper.fsync	= acngfs_fsync;

   umask(0);

   if(argc<3)
      barf("Needs a source and a target directory, see --help.");

   if(argv[1] && baseUrl.SetHttpUrl(argv[1]))
   {
#ifdef VERBOSE
	   cout << "Base URL: " << baseUrl.ToString()<<endl;
#endif
   }
   else {
      cerr << "Invalid source directory, " << argv[1] <<endl;
      exit(EXIT_FAILURE);
   }
   // FUSE adds starting / already, drop ours if present
   trimBack(baseUrl.sPath, "/");
   if(atoi(baseUrl.sPort.c_str())>0)
	   acfg::remoteport==baseUrl.sPort;

   struct stat stbuf;
   if(stat(argv[2], &stbuf) || !S_ISDIR(stbuf.st_mode))
      barf(endl<< argv[2] << " is not a directory.");

   const char *todir=argv[2];
   int fuseArgPos=3;
   /* hide the from argument */
   argv[fuseArgPos-1]=argv[2]; // mountpoint
   argv[fuseArgPos-2]=argv[0]; // application path
   argv=&argv[fuseArgPos-2];
   argc=argc-fuseArgPos+2;

   int p=stat(todir, &statTempl);
   int r=statfs(todir, &stfsTemp);
   if(p||r)
   {
	   cerr << "Error: unable to stat target FS\n";
	   return 0;
   }
   
   return fuse_main(argc, argv, &acngfs_oper);
}

