/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include "mysqlDatabase.hpp"
#include "mysqlResultRow.hpp"
#include "mysqlStatement.hpp"

using namespace std;

MySQLStatement::MySQLStatement( MySQLDatabase *_database,
                                const std::string &query )
  throw (Error):
  database( _database ), result( NULL )
{
  if ( !query.empty() ) {
#ifndef NDEBUG
    // cerr << query << endl;
#endif
    // Apparently a trailing semicolon doesn't cause any problems.
    ERRORMACRO( mysql_real_query( database->getConnection(), query.c_str(),
                                  query.size() ) == 0,
                Error, , "Error executing query '" << query << "': "
                << mysql_error( database->getConnection() ) );

    // Retrieve column-information, if applicable.
    if ( getNumCols() > 0 ) {
      result = mysql_use_result( database->getConnection() );
      ERRORMACRO( result != NULL, Error, ,
                  "Error retrieving column-information: "
                  << mysql_error( database->getConnection() ) );
    };
  } else
    database = NULL;
}

MySQLStatement::~MySQLStatement(void)
{

  if ( resultRow )
    resultRow->invalidate();

  if ( result != NULL ) {

    // Purge remaining data (see mysql-documentation!).
    while ( mysql_fetch_row( result ) );
    mysql_free_result( result );

  };

}

int MySQLStatement::getNumCols(void) throw (Error)
{
  int retVal;
  if ( database != NULL )
    retVal = mysql_field_count( database->getConnection() );
  else
    retVal = 0;
  return retVal;
}

std::string MySQLStatement::getColAttr( int col ) throw (Error)
{
  ERRORMACRO( col >= 0 && col < getNumCols(), Error, ,
              "Column number of meta-column must be between 0 and "
              << ( getNumCols() - 1 ) << " but was " << col << "." );

  assert( result != NULL );
  mysql_field_seek( result, col );
  return mysql_fetch_field( result )->name;
}

ResultRowPtr MySQLStatement::fetchRow(void) throw (Error)
{
  assert( result != NULL );

  if ( resultRow ) {
    ERRORMACRO( resultRow.use_count() <= 1, Error, ,
                "Can not fetch new row, "
                "because previous row was not released yet." );
    assert( resultRow.use_count() == 1 );
  };

  MYSQL_ROW row = mysql_fetch_row( result );
  if ( row != NULL )
    resultRow = ResultRowPtr( new MySQLResultRow( row ) );
  else
    resultRow.reset();

  return resultRow;
}
