/*
   Project: EnergyConverter

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   Created: 2005-11-04 16:42:49 +0100 by michael johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#include "EnergyConverter.h"

@implementation EnergyConverter

- (AdDataSet*) processEnergies: (AdDataSet*) energies withOptions: (NSMutableDictionary*) options
{
	int i, start, end, stepsize;
	double value, conversionFactor;
	AdDataSet* processedEnergies;
	AdDataMatrix* energyMatrix;
	AdMutableDataMatrix *processedMatrix;
	NSEnumerator* systemNamesEnum, *termEnum;
	NSString* unit;
	NSArray* selectedTerms;
	NSMutableArray *row;
	id systemName, term;
	
	//Read the options	
	systemNamesEnum = [[[options valueForMenuItem: @"Systems"]
				selectedItems] objectEnumerator];
	
	start = [[options valueForKeyPath: @"Frames.Start"] 
			intValue];
	end = start + [[options valueForKeyPath: @"Frames.Length"] 
			intValue];
	stepsize = [[options valueForKeyPath: @"Frames.StepSize"] 
			intValue];
	if(stepsize <= 0)
		stepsize = 1;
		
	unit = [[[options valueForMenuItem: @"Units"]
			selectedItems] objectAtIndex: 0];
	conversionFactor = [[conversionFactors objectForKey: unit]
				doubleValue];
	
	//Process the energies
	processedEnergies = [[AdDataSet alloc] 
				initWithName: @"Energies"
				inputReferences: nil
				dataGeneratorName: @"EnergyConverter"
				dataGeneratorVersion:
				[infoDict objectForKey: @"PluginVersion"]];
	[processedEnergies autorelease];

	while((systemName = [systemNamesEnum nextObject]))
	{
		selectedTerms = [[[options valueForMenuItem: @"Systems"]
					valueForMenuItem: systemName]
					selectedItems];
		energyMatrix = [energies dataMatrixWithName: systemName];
		processedMatrix = [[AdMutableDataMatrix alloc] 
					initWithNumberOfColumns: [selectedTerms count]
					columnHeaders: selectedTerms
					columnDataTypes: nil];
		[processedMatrix setName: systemName];
		[processedMatrix autorelease];			
		for(i=start; i<end; i+= stepsize)
		{
			termEnum = [selectedTerms objectEnumerator];
			row = [NSMutableArray array];
			while((term = [termEnum nextObject]))
			{
				value = [[energyMatrix elementAtRow: i
						ofColumnWithHeader: term] 
						doubleValue];
				if(!([term isEqual: @"Time"] || [term isEqual: @"Temperature"] 
					||[term isEqual: @"Iteration"]))
					value *= conversionFactor;

				[row addObject: 
					[NSNumber numberWithDouble: value]];
			}
			[processedMatrix extendMatrixWithRow: row];
		}
		[processedEnergies addDataMatrix: processedMatrix];
	}

	return processedEnergies;
}

/**********

Protocols

************/

- (id) init
{
	if(self == [super init])
	{
		infoDict = [[NSBundle bundleForClass: [self class]]
				infoDictionary];
		[infoDict retain];
		conversionFactors = [NSDictionary dictionaryWithObjectsAndKeys:
					[NSNumber numberWithDouble: 1.0], @"Simulation",
					[NSNumber numberWithDouble: STCAL], @"KCal per mol",
					[NSNumber numberWithDouble: STJMOL], @"J per mol",
					nil];
		[conversionFactors retain];		
	}

	return self;
}

- (void) dealloc
{
	[conversionFactors release];
	[super dealloc];
}

//Default implementation
- (BOOL) checkInputs: (NSArray*) inputs error: (NSError**) error
{
	return YES;
}

- (NSDictionary*) pluginOptions: (NSArray*) inputs
{
	NSMutableDictionary* options, *systemMenu, *termMenu, *unitsMenu ,*framesMenu;
	NSEnumerator* matrixEnum;
	AdSimulationData* data;
	AdDataSet* energies;
	id matrix;

	data = [inputs objectAtIndex: 0];
	energies = [data energies];

	options = [NSMutableDictionary newNodeMenu: NO];
	
	//SystemsMenu
	systemMenu = [NSMutableDictionary newNodeMenu: YES];
	[systemMenu setSelectionMenuType: @"Multiple"];
	
	matrixEnum = [[energies dataMatrices] objectEnumerator];
	while((matrix = [matrixEnum nextObject]))
	{
		termMenu = [NSMutableDictionary newLeafMenu];
		[termMenu addMenuItems: [matrix columnHeaders]];
		[termMenu setSelectionMenuType: @"Multiple"];
		[systemMenu addMenuItem: [matrix name]
			withValue: termMenu];
	}

	if([[systemMenu menuItems] count] != 0)
		[options addMenuItem: @"Systems"
			withValue: systemMenu];

	//Units Menu
	unitsMenu = [NSMutableDictionary newLeafMenu];
	[unitsMenu addMenuItems: 
		[NSArray arrayWithObjects: 
			@"Simulation",
			@"KCal per mol",
			@"J per mol",
			nil]];
	[unitsMenu setDefaultSelection: @"KCal per mol"];
	[options addMenuItem: @"Units"
		withValue: unitsMenu];

	//Frames Menu
	framesMenu = [NSMutableDictionary newNodeMenu: NO];
	[framesMenu addMenuItem: @"Start"
		withValue: [NSNumber numberWithInt: 0]];
	//FIXME: May have different number of frames per system	
	[framesMenu addMenuItem: @"Length"
		withValue: [NSNumber numberWithInt: 0]];
	[framesMenu addMenuItem: @"StepSize"
		withValue: [NSNumber numberWithInt: 1]];
	[options addMenuItem: @"Frames"
		withValue: framesMenu];
	
	return  options;
}	

- (NSDictionary*) processInputs: (NSArray*) inputs userOptions: (NSDictionary*) options; 
{ 
	NSMutableDictionary* resultsDict = [NSMutableDictionary dictionary];
	NSMutableDictionary* mutableOptions;
	AdSimulationData* simulationData;
	AdDataSet* processedEnergies;
	id energies;

	simulationData = [inputs objectAtIndex: 0];

	if(![simulationData isKindOfClass: [AdSimulationData class]])
		[NSException raise: NSInvalidArgumentException
			format: @"EnergyConverter cannot process %@ objects", 
			NSStringFromClass([simulationData class])];
	
	energies = [simulationData energies];
	if([[energies dataMatrices] count] == 0)
		return nil;
	
	mutableOptions = [[options mutableCopy] autorelease];
	processedEnergies = [self processEnergies: energies 
				withOptions: mutableOptions];
	if([[processedEnergies dataMatrices] count] == 0)
		return nil;
	[resultsDict setObject: [NSArray arrayWithObject: processedEnergies] 
		forKey: @"ULAnalysisPluginDataSets"];

	return resultsDict;
}

@end
