# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


# class for an item in the VTS list
package VTSListCommonItem;
use Qt;
use Qt::isa qw (Qt::CheckListItem);
use Qt::attributes qw (ordn vlit hilited);
use constant
{
  VTS_ITEM => 0,
  PGC_ITEM => 1,
  PGM_ITEM => 2
};


# VTSListCommonItem (ordinal_nr, parent, text, vli_type)
sub NEW
{
  my $this = shift;
  my $ordn = shift;
  my $vlit = pop;

  push @_, ($vlit == &PGM_ITEM ? &Qt::CheckListItem::CheckBox :
	    &Qt::CheckListItem::CheckBoxController);

  $this -> SUPER::NEW (@_);

  ordn    = $ordn;
  vlit    = $vlit;
  hilited = 0;
}


sub DESTROY
{
#  print STDERR "VTSListCommonItem::DESTROY\n";

  return unless (SUPER);

  ordn    = 0;
  vlit    = 0;
  hilited = 0;

  SUPER -> DESTROY ();
}


sub remove
{
  ordn    = 0;
  vlit    = 0;
  hilited = 0;

  while (my $item = firstChild ())
  {
    $item -> remove ();
  }

  my $parent = depth () ? parent () : listView ();

  $parent -> takeItem (this);
}


# item's ordinal number - used for sorting
sub ordinal_nr
{
  return ordn;
}


# VTS list item type
sub vli_type
{
  return vlit;
}


sub is_hilited
{
  return hilited;
}


sub set_hilited
{
  my $hilited = shift;

  return if ($hilited == hilited);

  hilited = $hilited;

  repaint ();
  setOpen ($hilited);
}


sub protected_set_hilited
{
  hilited = shift;
}


# overload function of Qt::CheckListItem
sub paintCell
{
  if (hilited)
  {
    my $painter  = shift;
    my $org_cgrp = shift;

    my $new_cgrp = Qt::ColorGroup ($org_cgrp);
    $new_cgrp -> setColor (&Qt::ColorGroup::Text, &Qt::darkGreen);
    unshift @_, ($painter, $new_cgrp);
  }

  SUPER -> paintCell (@_);
}


1;


package VTSListPGMItem;
use Qt;
use Qt::isa qw (VTSListCommonItem);
use Qt::attributes qw (list_items selected copied nr_of_sectors angle);


# VTSListPGMItem (vts_list_items, ordinal_nr, parent, text)
sub NEW
{
  my $this       = shift;
  my $list_items = shift;

  push @_, &PGM_ITEM;

  $this -> SUPER::NEW (@_);

  angle         = 1;
  copied        = 0;
  nr_of_sectors = [];
  list_items    = $list_items;
}


sub DESTROY
{
#  print STDERR "VTSListPGMItem::DESTROY\n";

  nr_of_sectors = undef;
  list_items    = undef;

  SUPER -> DESTROY ();
}


sub remove
{
  nr_of_sectors = undef;
  list_items    = undef;

  SUPER -> remove ();
}


# overload function of VTSListCommonItem
sub set_hilited
{
  my $hilited = shift;

  return if ($hilited == &hilited);

  protected_set_hilited ($hilited);

  repaint ();
  list_items -> trigger_size_update ();
}


sub is_selected
{
  return isOn ();
}


sub is_copied
{
  return copied;
}


sub set_copied
{
  copied = shift;
}


sub size
{
  return (nr_of_sectors -> [0] || 0) + (nr_of_sectors -> [angle] || 0);
}


sub add_size
{
  my $angle         = shift;
  my $nr_of_sectors = shift;

  nr_of_sectors -> [$angle] ||= 0;
  nr_of_sectors -> [$angle]  += $nr_of_sectors;
}


sub has_angle_block
{
  return @{&nr_of_sectors} > 1;
}


sub selected_angle
{
  return angle;
}


sub set_selected_angle
{
  my $angle = shift;

  list_items -> trigger_size_update ()
    if ($angle != angle && @{&nr_of_sectors} > 1);

  angle = $angle;
}


# overload function of Qt::CheckListItem
sub stateChange
{
  list_items -> trigger_size_update ();
}


1;


package PgmToolTip;
use Qt;
use Qt::isa qw (Qt::ToolTip);
use Qt::attributes qw (list_items vts_list);
use DvdIfoRead;


sub NEW
{
  my $this       = shift;
  my $list_items = shift;
  my $vts_list   = shift;

  $this -> SUPER::NEW ($vts_list -> viewport ());

  list_items = $list_items;
  vts_list   = $vts_list;
}


sub DESTROY
{
#  print STDERR "PgmToolTip::DESTROY\n";

  return unless (SUPER);

  list_items = undef;
  vts_list   = undef;

  SUPER -> DESTROY ();
}


sub maybeTip
{
  my $item = vts_list -> itemAt (@_);

  return unless ($item);

  my $tip = list_items -> tip_text ($item);

  return unless ($tip);

  my $rect = vts_list -> itemRect ($item);

  return unless ($rect -> isValid ());

  tip ($rect, $tip);
}


sub intact
{
  return defined (vts_list);
}


1;


package VTSListItems;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (title_list vts_list dvd title_nr angle timer
		       total_sectors selected_sectors tool_tip item_refs);
use Qt::signals total_selected_changed => [ "int" ],
                title_selected_changed => [ "int" ];
use Qt::slots update              => [ "int", "int", "int" ],
              item_double_clicked => [ "QListViewItem *",
				       "const QPoint &", "int" ],
              size_update         => [ ];
use Carp;
use DvdIfoRead;
use QVamps qw (read_setting replace tr);
use PgmToolTip;
use VTSListCommonItem;
use VTSListPGMItem;


# VTSListItems (title_list, dvd, vts_list, name)
sub NEW
{
  my $this       = shift;
  my $title_list = shift;
  my $dvd        = shift;
  my $vts_list   = shift;

  unshift @_, $vts_list;

  $this -> SUPER::NEW (@_);

  title_list       = $title_list;
  dvd              = $dvd;
  vts_list         = $vts_list;
  title_nr         = 0;
  angle            = 1;
  total_sectors    = 0;
  selected_sectors = 0;
  item_refs        = [];

  $vts_list -> setEnabled (1);

  my $nr_of_title_sets = $dvd -> nr_of_title_sets ();

  for (my $tsn = $nr_of_title_sets; $tsn; $tsn--)
  {
    my $vts = VTSListCommonItem ($tsn, $vts_list,
				 (sprintf "%s %d", &tr ("VTS"), $tsn),
				 &VTSListCommonItem::VTS_ITEM);
    push @{&item_refs}, $vts;

    my $nr_of_program_chains = $dvd -> nr_of_program_chains ($tsn);

    for (my $pgcn = $nr_of_program_chains; $pgcn; $pgcn--)
    {
      my @pgc_playback_time = dvd -> pgc_playback_time ($tsn, $pgcn);

      # FIXME!
      next unless (1 ||
		   $pgc_playback_time [3] > 12 || $pgc_playback_time [2] ||
		   $pgc_playback_time [1]      || $pgc_playback_time [0]);

      my $pgc = VTSListCommonItem ($pgcn, $vts,
				   (sprintf "%s %d", &tr ("PGC"), $pgcn),
				   &VTSListCommonItem::PGC_ITEM);
      push @{&item_refs}, $pgc;

      my $pgc_sectors    = 0;
      my $nr_of_programs = $dvd -> nr_of_programs ($tsn, $pgcn);

      for (my $pgn = $nr_of_programs; $pgn; $pgn--)
      {
	my $pgm = VTSListPGMItem (this, $pgn, $pgc,
				  (sprintf "%s %d", &tr ("PG"), $pgn));
	push @{&item_refs}, $pgm;
	my @cells = dvd -> program_cells ($tsn, $pgcn, $pgn);

	my $angle         = 0;
	my $pgm_sectors   = 0;
	my @playback_time = ( 0, 0, 0, 0 );

	foreach my $cell (@cells)
	{
	  my @pbt = dvd -> cell_playback_time ($tsn, $pgcn, $cell);
	  my $fps = dvd -> cell_frame_rate ($tsn, $pgcn, $cell);

	  if (grep ($_, @pbt))
	  {
	    my $nr_of_sectors = dvd -> nr_of_sectors ($tsn, $pgcn, $cell);

	    $pgc_sectors += $nr_of_sectors;
	    $pgm_sectors += $nr_of_sectors;

	    if ($angle)
	    {
	      $angle++;
	    }
	    else
	    {
	      $angle = 1 if (dvd -> block_type ($tsn, $pgcn, $cell) == 1);
	    }

	    $pgm -> add_size ($angle, $nr_of_sectors);

	    $angle = 0 if ($angle &&
			   dvd -> block_mode ($tsn, $pgcn, $cell) == 3);
	  }

	  @playback_time = DvdIfoRead::add_time (@playback_time, @pbt, $fps);
	}

	#$pgm -> set_size ($pgm_sectors);
	$pgm -> setText (2, (sprintf "%02d:%02d:%02d.%02d", @playback_time));
	$pgm -> setText (3, (sprintf "%.1f MB", $pgm_sectors/512.));
	total_sectors += $pgm_sectors;
      }

      $pgc -> setText (2, (sprintf "%02d:%02d:%02d.%02d", @pgc_playback_time));
      $pgc -> setText (3, (sprintf "%.1f MB", $pgc_sectors/512.));

      # FIXME!
      if (0 && $pgc_sectors < 250)
      {
	for (my $it = Qt::ListViewItemIterator ($pgc);
	     my $item = $it -> current (); $it++)
	{
	  $pgc -> takeItem ($item);
	}

	$vts -> takeItem ($pgc);

        while (pop @{&item_refs} != $pgc)
        {
        }
      }
    }
  }

  $vts_list -> setSorting (-1);

  this -> connect ($vts_list,
	   SIGNAL "doubleClicked(QListViewItem *, const QPoint &, int)",
	   SLOT   "item_double_clicked(QListViewItem *, const QPoint &, int)");

  timer = Qt::Timer ();
  this -> connect (timer, SIGNAL "timeout()", SLOT "size_update()");

  tool_tip = PgmToolTip (this, $vts_list);
}


sub DESTROY
{
#  print STDERR "VTSListItems::DESTROY\n";

  parent () -> removeChild (this);

  vts_list         = undef;
  dvd              = undef;
  title_nr         = 0;
  angle            = 0;
  timer            = undef;
  total_sectors    = 0;
  selected_sectors = 0;
  tool_tip         = undef;
  item_refs        = undef;

  SUPER -> DESTROY ();
}


sub remove_all
{
  my $vts_list = vts_list;
  my $tool_tip = tool_tip;

  vts_list         = undef;
  dvd              = undef;
  title_nr         = 0;
  angle            = 0;
  timer            = undef;
  total_sectors    = 0;
  selected_sectors = 0;
  tool_tip         = undef;
  item_refs        = undef;

  $vts_list -> setEnabled (0);
  $vts_list -> disconnect ();
  $tool_tip -> remove ($vts_list -> viewport ()) if ($tool_tip -> intact ());

  while (my $item = $vts_list -> firstChild ())
  {
    $item -> remove ();
  }
}


sub nr_of_total_sectors
{
  return total_sectors;
}


sub nr_of_selected_sectors
{
  return selected_sectors;
}


sub update : SLOT(int, int, int)
{
  my $title_nr = shift;
  my $angle    = shift;
  my $state    = shift;

  return if ($title_nr == title_nr &&
	     $angle    == angle    && $state == &VTSListCommonItem::NoChange);

  title_nr = $title_nr;
  angle    = $angle;

  my ($vts_nr, @ptt_map, $pgc_map);
  my $tsn = dvd -> title_set_nr ($title_nr);
  my $ttn = dvd -> title_nr_in_title_set ($title_nr);

  # create empty ptt_map
  my $nr_of_program_chains = dvd -> nr_of_program_chains ($tsn);

  foreach my $pgcn (1 .. $nr_of_program_chains)
  {
    $ptt_map [$pgcn] = [];
  }

  # initialize ptt_map for given title
  my $nr_of_ptts = dvd -> nr_of_ptts ($tsn, $ttn);

  foreach my $i (1 .. $nr_of_ptts)
  {
    my $pgcn = dvd -> program_chain_nr ($tsn, $ttn, $i);
    my $pgn  = dvd -> program_nr ($tsn, $ttn, $i);

    $ptt_map [$pgcn] [$pgn] = [] unless ($ptt_map [$pgcn] [$pgn]);
    push @{$ptt_map [$pgcn] [$pgn]}, $i;
  }

  my @last_chapters;

  # iterate over all items in VTS list
  for (my $it = Qt::ListViewItemIterator (vts_list);
       my $item = $it -> current (); $it++)
  {
    my $vlit = $item -> vli_type ();

    if ($vlit == &VTSListCommonItem::PGM_ITEM)
    {
      if (@{$pgc_map})
      {
	# check for undef'ed array ref introduced
	# due to trouble with a certain DVD - 2005/09/12
	my $chapters   = $pgc_map -> [$item -> ordinal_nr ()];
	my @chapters   = @{$chapters} if ($chapters);
	@chapters      = @last_chapters unless (@chapters);
	@last_chapters = @chapters;

	if (@chapters)
	{
	  # chapter(s) of given title map(s) to this program
	  $item -> set_hilited (1);
	  $item -> setText (1, join (",", @chapters));
	  $item -> setState ($state)
	    unless ($state == &VTSListCommonItem::NoChange);
	  $item -> set_selected_angle ($angle);
	  next;
	}
      }

      $item -> set_hilited (0);
      $item -> setText (1, "");
      next;
    }

    if ($vlit == &VTSListCommonItem::PGC_ITEM)
    {
      @last_chapters = ();

      if ($vts_nr == $tsn)
      {
	# matching title set
	$pgc_map = $ptt_map [$item -> ordinal_nr ()];

	if (@{$pgc_map})
	{
	  # this PGC holds chapter(s) of given title
	  $item -> set_hilited (1);
	  next;
	}

	$item -> set_hilited (0);
	next;
      }

      $pgc_map = [];
      $item -> set_hilited (0);
      next;
    }

    if ($vlit == &VTSListCommonItem::VTS_ITEM)
    {
      @last_chapters = ();
      $vts_nr        = $item -> ordinal_nr ();

      if ($vts_nr == $tsn)
      {
	# matching title set
	$item -> set_hilited (1);
	next;
      }

      $item -> set_hilited (0);
      next;
    }
  }
}


sub item_double_clicked : SLOT(QListViewItem *, const QPoint &, int)
{
  my $item = shift;

  play_program ($item);
}


sub item_preview
{
  return unless (vts_list -> hasFocus ());

  play_program (vts_list -> currentItem ());
}


sub play_program
{
  my $item = shift;

  return unless ($item);

  if ($item -> vli_type () == &VTSListCommonItem::PGM_ITEM)
  {
    my $parent = $item -> parent ();
    my $tsn    = $parent -> parent () -> ordinal_nr ();
    my $pgcn   = $parent -> ordinal_nr ();
    my $pgn    = $item -> ordinal_nr ();
    my @cells  = dvd -> program_cells ($tsn, $pgcn, $pgn);

    @cells = dvd -> refine_angle_cells ($item -> selected_angle (),
					$tsn, $pgcn, \@cells)
      if ($item -> has_angle_block ());

    my $cells   = join (" ", @cells);
    my $cells_1 = join (" ", map $_ - 1, @cells);
    my $opt     = "";

    if ($item -> is_hilited ())
    {
      # if the program item is highlighted, we
      # can easily determine the corresponding title
      my $title_list = title_list;
      my $tli        = $title_list -> current_item ();
      my $ttn        = $tli -> title_nr ();
      my $astrm      = $title_list -> first_selected_audio ($tli, $ttn);
      my $sstrm      = $title_list -> first_selected_subtitle ($tli, $ttn);

      my $aid = $title_list -> audio_base_id ($ttn, $astrm) + $astrm - 1;
      my $sid = 0x20 + $sstrm - 1;

      my $aopt = read_setting ("/MediaPlayer/select_audio");
      my $sopt = read_setting ("/MediaPlayer/select_subtitle");

      my @opt;
      push @opt, replace ($aopt, { "n" => $astrm-1,
				   "N" => $astrm, "i" => $aid }) if ($aopt);
      push @opt, replace ($sopt, { "n" => $sstrm-1,
				   "N" => $sstrm, "i" => $sid })
	if ($sopt && $sstrm);
      $opt = " " . join (" ", @opt);
    }

    my $cmd = read_setting ("/MediaPlayer/play_dvd_cells");
    $cmd    = replace ($cmd, { "d" => dvd -> {dev},
			       "o" => $opt,
			       "s" => $tsn-1,   "S" => $tsn,
			       "g" => $pgcn-1,  "G" => $pgcn,
			       "c" => $cells_1, "C" => $cells });
    system ("$cmd &");
  }
}


sub item_details
{
  my $vts_list = vts_list;

  return unless ($vts_list -> hasFocus ());

  my $item = $vts_list -> currentItem ();

  return unless ($item);

  my $tip = tip_text ($item);

  return unless ($tip);

  my $vpos = $vts_list -> itemRect ($item) -> center ();
  my $gpos = $vts_list -> viewport () -> mapToGlobal ($vpos);

  Qt::WhatsThis::display ($tip, $gpos);
}


sub tip_text
{
  my $item = shift;

  return undef unless ($item);

  if ($item -> vli_type () == &VTSListCommonItem::PGM_ITEM)
  {
    my $parent = $item -> parent ();
    my $tsn    = $parent -> parent () -> ordinal_nr ();
    my $pgcn   = $parent -> ordinal_nr ();
    my $pgn    = $item -> ordinal_nr ();
    my @cells  = dvd -> program_cells ($tsn, $pgcn, $pgn);

    my @attr = ( &tr ("Cell Numbers") );
    my @val  = ( join (",", @cells) );

    if ($item -> has_angle_block ())
    {
      push @attr, &tr ("Selected Angle's Cells");
      push @val, join (",",
		       dvd -> refine_angle_cells ($item -> selected_angle (),
						  $tsn, $pgcn, \@cells));
    }

    my $text = "<table cellspacing=0 cellpadding=0>";

    for (my $i = 0; $i < @attr; $i++)
    {
      $text .= sprintf "<tr><td><nobr>%s:</nobr></td>" .
	               "<td><nobr>%s</nobr></td></tr>", $attr [$i], $val [$i];
    }

    $text .= "</table>";

    return $text;
  }

  return undef;
}


sub trigger_size_update
{
  timer -> start (0, 1);
}


sub size_update : SLOT()
{
  my ($total_selected, $title_selected) = ( 0, 0 );

  # iterate over all items in VTS list
  for (my $it = Qt::ListViewItemIterator (vts_list);
       my $item = $it -> current (); $it++)
  {
    if ($item -> vli_type () == &VTSListCommonItem::PGM_ITEM &&
	$item -> state ()    != &VTSListCommonItem::Off)
    {
      my $size         = $item -> size ();
      $total_selected += $size;
      $title_selected += $size if ($item -> hilited ());
    }
  }

  selected_sectors = $total_selected;

  emit total_selected_changed ($total_selected);
  emit title_selected_changed ($title_selected);
}


sub find_item
{
  my $tsn  = shift;
  my $pgcn = shift;
  my $pgn  = shift;

  my ($found_vts, $found_pgc);

  for (my $it = Qt::ListViewItemIterator (vts_list);
       my $item = $it -> current (); $it++)
  {
    return $item if ($found_vts && $found_pgc &&
		     $item -> vli_type ()   == &VTSListCommonItem::PGM_ITEM &&
		     $item -> ordinal_nr () == $pgn);

    if ($item -> vli_type () == &VTSListCommonItem::PGC_ITEM)
    {
      $found_pgc = $item -> ordinal_nr () == $pgcn;

      next;
    }

    if ($item -> vli_type () == &VTSListCommonItem::VTS_ITEM)
    {
      $found_vts = $item -> ordinal_nr () == $tsn;

      next;
    }
  }

  croak "Item not found";
}


sub clear_all_copied_flags
{
  for (my $it = Qt::ListViewItemIterator (vts_list);
       my $item = $it -> current (); $it++)
  {
    $item -> set_copied (0)
      if ($item -> vli_type () == &VTSListCommonItem::PGM_ITEM);
  }
}


sub any_program_selected
{
  my $tsn  = shift;
  my $pgcn = shift;

  my $found_vts = !$tsn;
  my $found_pgc = !$pgcn;

  for (my $it = Qt::ListViewItemIterator (vts_list);
       my $item = $it -> current (); $it++)
  {
    # Qt::ListViewItemIterator::Checked iterator flag does
    # not work on CheckBoxController (tristate) type items
    next if ($item -> state () == &TitleListCommonItem::Off);

    return 1 if ($found_vts && $found_pgc &&
		 $item -> vli_type () == &VTSListCommonItem::PGM_ITEM &&
		 !$item -> is_copied ());

    if ($found_vts && $pgcn &&
	$item -> vli_type () == &VTSListCommonItem::PGC_ITEM)
    {
      if ($item -> ordinal_nr () == $pgcn)
      {
	$found_pgc = 1;
      }
      elsif ($found_pgc)
      {
	# optimization
	return 0;
      }

      next;
    }

    if ($tsn && $item -> vli_type () == &VTSListCommonItem::VTS_ITEM)
    {
      if ($item -> ordinal_nr () == $tsn)
      {
	$found_vts = 1;
      }
      elsif ($found_vts)
      {
	# optimization
	return 0;
      }

      next;
    }
  }

  return 0;
}


1;
