/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmrip-pref.h"
#include "ogmrip-progress.h"
#include "ogmrip-gconf.h"
#include "ogmrip-options.h"

#include "ogmrip-audio-options.h"
#include "ogmrip-subp-options.h"

#include "ogmdvd.h"
#include "ogmdvd-gtk.h"
#include "ogmrip-gtk.h"

#include "ogmjob.h"
#include "ogmrip.h"

#ifdef HAVE_ENCHANT_SUPPORT
#include "ogmrip-spell.h"
#endif /* HAVE_ENCHANT_SUPPORT */

#include <sys/stat.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>

#include <libxml/tree.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>

#ifdef HAVE_LIBNOTIFY_SUPPORT
#include <libnotify/notify.h>
#endif /* HAVE_LIBNOTIFY_SUPPORT */

#define OGMRIP_UI_FILE    "ogmrip/ogmrip-ui.xml"
#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-main.glade"
#define OGMRIP_ICON_FILE  "pixmaps/ogmrip.png"

#define OGMRIP_DEFAULT_FILE_NAME "movie"

typedef struct
{
  OGMDvdDisc *disc;

  GtkWidget *window;

  GtkWidget *pref_dialog;
  GtkWidget *options_dialog;
  GtkWidget *progress_dialog;

  GtkWidget *title_entry;
  GtkWidget *title_chooser;
  GtkWidget *length_label;
  GtkWidget *relative_check;

  GtkWidget *audio_list;
  GtkWidget *subp_list;

  GtkWidget *chapter_list;

  GtkAction *extract_action;
  GtkWidget *extract_button;
  GtkAction *import_chap_action;
  GtkAction *export_chap_action;

  gboolean encoding;

} OGMRipData;

typedef struct
{
  OGMDvdTitle *title;

  OGMRipEdl *edl;

  GSList *audio_files;
  GSList *audio_streams;

  GSList *subp_files;
  GSList *subp_streams;

  gboolean ccheck;
  gboolean copy_dvd;
  gboolean ensure_sync;
  gboolean log_output;
  gboolean keep_temp;
  gboolean progressive;
  gboolean pullup;

  guint passes;
  GType container;
  GType video_codec;

  guint start_chap;
  gint  end_chap;

  guint bitrate;
  guint target_number;
  guint target_size;

  gint64 rip_size;
  gdouble rip_length;

  gchar *outfile;
  gchar *logfile;

  const gchar *label;
} OGMRipEncode;

typedef struct
{
  OGMDvdAudioStream *stream;
  GType codec;
  gint quality;
  gint srate;
  gint channels;
  gboolean normalize;
} OGMRipAudioSource;

typedef struct
{
  OGMDvdSubpStream *stream;
  GType codec;
  gint charset;
  gint eol;
  gboolean spell;
  gboolean forced_subs;
} OGMRipSubpSource;

enum
{
  OGMRIP_AFTER_ENC_REMOVE,
  OGMRIP_AFTER_ENC_KEEP,
  OGMRIP_AFTER_ENC_UPDATE,
  OGMRIP_AFTER_ENC_ASK
};

#ifdef HAVE_ENCHANT_SUPPORT
/*
 * Performs spell checking
 */
static gboolean
ogmrip_main_spell_check (OGMRipData *data, OGMRipSubp *subp)
{
  gboolean retval = FALSE;
  gchar *text, *corrected;
  gchar *new_file = NULL;
  const gchar *old_file;
  gint lang;

  GtkWidget *checker = NULL;
  GIOChannel *input = NULL, *output = NULL;
  GIOStatus status = G_IO_STATUS_NORMAL;

  old_file = ogmrip_codec_get_output (OGMRIP_CODEC (subp));

  lang = ogmdvd_subp_stream_get_language (ogmrip_subp_get_dvd_subp_stream (subp));
  checker = ogmrip_spell_dialog_new (ogmdvd_get_language_iso639_1 (lang));
  if (!checker)
  {
    ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s",
        _("Could not create dictionary"), _("Spell will not be checked."));
    goto spell_check_cleanup;
  }

  input = g_io_channel_new_file (old_file, "r", NULL);
  if (!input)
    goto spell_check_cleanup;

  new_file = ogmrip_fs_mktemp ("sub.XXXXXX", NULL);
  if (!new_file)
    goto spell_check_cleanup;

  output = g_io_channel_new_file (new_file, "w", NULL);
  if (!output)
    goto spell_check_cleanup;

  gtk_window_set_parent (GTK_WINDOW (checker), GTK_WINDOW (data->window));
  gtk_widget_show (checker);

  do
  {
    status = g_io_channel_read_line (input, &text, NULL, NULL, NULL);
    if (status == G_IO_STATUS_NORMAL)
    {
      retval = ogmrip_spell_dialog_check_text (OGMRIP_SPELL_DIALOG (checker), text, &corrected);
      if (retval)
      {
        do
        {
          status = g_io_channel_write_chars (output, corrected ? corrected : text, -1, NULL, NULL);
        }
        while (status == G_IO_STATUS_AGAIN);

        g_free (corrected);
      }
      g_free (text);
    }
  } 
  while (retval == TRUE && (status == G_IO_STATUS_NORMAL || status == G_IO_STATUS_AGAIN));

  retval &= status == G_IO_STATUS_EOF;

spell_check_cleanup:
  if (checker)
    gtk_widget_destroy (checker);

  if (output)
  {
    g_io_channel_shutdown (output, TRUE, NULL);
    g_io_channel_unref (output);
  }

  if (input)
  {
    g_io_channel_shutdown (input, TRUE, NULL);
    g_io_channel_unref (input);
  }

  if (retval)
    retval = ogmrip_fs_rename (new_file, old_file, NULL);
  else
    g_unlink (new_file);

  g_free (new_file);

  return retval;
}
#endif /* HAVE_ENCHANT_SUPPORT */

/*
 * Makes the log file for 2-pass encoding
 */
static gchar *
ogmrip_main_mklog (void)
{
#if !MPLAYER_CHECK_VERSION(1,0,0,8)
  gint codec;
  const gchar *name;
#endif /* !MPLAYER_CHECK_VERSION(1,0,0,8) */
  gchar *filename;

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  filename = ogmrip_fs_mktemp ("log.XXXXXX", NULL);
#else /* MPLAYER_CHECK_VERSION(1,0,0,8) < 8 */
  /*
   * Workaround against xvid pass log file
   * Should disappear someday
   */
  codec = ogmrip_gconf_get_video_codec_type (NULL);
  name = ogmrip_plugin_get_video_codec_name (codec);
  
  if (strcmp (name, "xvid") == 0)
    filename = g_build_filename (g_get_tmp_dir (), "xvid-twopass.stats", NULL);
   else
    filename = ogmrip_fs_mktemp ("log.XXXXXX", NULL);
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  return filename;
}

/*
 * Builds the output filename
 */
static void
ogmrip_main_build_filenames (OGMRipData *data, OGMRipEncode *encode)
{
  OGMRipAudioSource *audio_source = NULL;

  gchar basename[FILENAME_MAX], utf8name[FILENAME_MAX];
  gchar *pathname, *filename, *ext;
  const gchar *lang = "Undetermined";

  if (encode->audio_streams)
  {
    audio_source = encode->audio_streams->data;
    lang = ogmdvd_get_language_label (ogmdvd_audio_stream_get_language (audio_source->stream));
  }

  switch (ogmrip_preferences_get_int (OGMRIP_GCONF_FILENAME, OGMRIP_DEFAULT_FILENAME))
  {
    case 0:
      strncpy (basename, encode->label, FILENAME_MAX);
      break;
    case 1:
      snprintf (basename, FILENAME_MAX, "%s - %s", encode->label, lang);
      break;
    case 2:
      snprintf (basename, FILENAME_MAX, "%s - %s - %s", encode->label, lang,
          ogmrip_plugin_get_video_codec_name (encode->video_codec));
      break;
    case 3:
      if (audio_source)
        snprintf (basename, FILENAME_MAX, "%s - %s - %s %s", encode->label, lang,
            ogmrip_plugin_get_video_codec_name (encode->video_codec),
            ogmrip_plugin_get_audio_codec_name (audio_source->codec));
      else
        snprintf (basename, FILENAME_MAX, "%s - %s - %s", encode->label, lang,
            ogmrip_plugin_get_video_codec_name (encode->video_codec));
      break;
    default:
      strncpy (basename, OGMRIP_DEFAULT_FILE_NAME, FILENAME_MAX);
      break;
  }

  pathname = ogmrip_preferences_get_filename (OGMRIP_GCONF_OUTPUT_DIR, OGMRIP_DEFAULT_OUTPUT_DIR);
  filename = g_build_filename (pathname, basename, NULL);
  g_free (pathname);

  ext = g_utf8_strdown (ogmrip_plugin_get_container_name (encode->container), -1);
  snprintf (utf8name, FILENAME_MAX, "%s.%s", filename, ext);
  g_free (ext);

  encode->outfile = g_filename_from_utf8 (utf8name, -1, NULL, NULL, NULL);

  encode->logfile = NULL;
  if (encode->log_output)
  {
    snprintf (utf8name, FILENAME_MAX, "%s.log", filename);
    encode->logfile = g_filename_from_utf8 (utf8name, -1, NULL, NULL, NULL);
  }

  g_free (filename);
}

/*
 * Returns the rip length
 */
static gdouble
ogmrip_main_get_rip_length (OGMRipData *data, OGMRipEncode *encode)
{
  if (encode->start_chap == 0 && encode->end_chap == -1)
    return ogmdvd_title_get_length (encode->title, NULL);
  else
    return ogmdvd_title_get_chapters_length (encode->title, encode->start_chap, encode->end_chap, NULL);
}

/*
 * Returns the estimated rip size
 */
static gint64
ogmrip_main_get_rip_size (OGMRipData *data, OGMRipEncode *encode)
{
  gdouble factor = 1.0;

  if (GTK_WIDGET_SENSITIVE (data->relative_check) &
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->relative_check)))
  {
    gdouble full_len;

    full_len = ogmdvd_title_get_length (encode->title, NULL);
    if (full_len < 0)
      return -1;

    factor = encode->rip_length / full_len;
  }

  return (gint64) ceil (factor * encode->target_size * encode->target_number * 1024 * 1024);
}

/*
 * Returns the estimed audio size used to enforce sync
 */
static gint64
ogmrip_main_get_sync_size (OGMRipData *data, OGMRipEncode *encode)
{
  gdouble chap_len;

  if (!encode->ensure_sync)
    return 0;

  chap_len = ogmdvd_title_get_chapters_length (encode->title, encode->start_chap, encode->end_chap, NULL);
  if (chap_len < 0.0)
    return -1;

  return (gint64) (chap_len * 16000);
}

/*
 * Returns the VMG + VTS size
 */
static gint64
ogmrip_main_get_dvd_size (OGMRipData *data, OGMRipEncode *encode)
{
  OGMDvdDisc *disc;
  gint64 vts_size;

  if (!encode->copy_dvd)
    return 0;

  vts_size = ogmdvd_title_get_vts_size (encode->title);
  if (vts_size < 0)
    return -1;

  disc = ogmdvd_title_get_disc (encode->title);

  return vts_size + ogmdvd_disc_get_vmg_size (disc);
}

/*
 * Checks if the DVD can be copied
 */
static gboolean
ogmrip_main_get_copy_dvd (OGMRipData *data, GError **error)
{
  gboolean copy_dvd;

  copy_dvd = ogmrip_preferences_get_bool (OGMRIP_GCONF_COPY_DVD, OGMRIP_DEFAULT_COPY_DVD);

  if (copy_dvd)
  {
    const gchar *device;
    struct stat buf;

    device = ogmdvd_disc_get_device (data->disc);
    if (g_stat (device, &buf) < 0)
    {
      g_set_error (error, OGMRIP_ERROR, 0, _("Cannot stat '%s'"), device);
      return FALSE;
    }

    if (!S_ISBLK (buf.st_mode))
      return FALSE;
  }

  return copy_dvd;
}

/*
 * Checks if container and codecs are compatible
 */
static gboolean
ogmrip_main_check_container (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  OGMRipAudioSource *audio_source;
  OGMRipSubpSource *subp_source;
  GSList *link;

  if (encode->container == G_TYPE_NONE)
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s\n%s",
        _("No container has been selected."),
        _("This is an unexpected error and should not happen."),
        _("Please, fill a bug report at http://ogmrip.sf.net."));
    return FALSE;
  }

  if (encode->video_codec != G_TYPE_NONE &&
      !ogmrip_plugin_can_contain_video (encode->container, encode->video_codec))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The container and the video codec are incompatible."), 
        _("Please, choose some others."));
    return FALSE;
  }

  for (link = encode->audio_streams; link; link = link->next)
  {
    audio_source = link->data;

    if (audio_source->codec != G_TYPE_NONE &&
        !ogmrip_plugin_can_contain_audio (encode->container, audio_source->codec))
    {
      g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
          _("The container and the audio codec are incompatible."), 
          _("Please, choose some others."));
      return FALSE;
    }
  }

  for (link = encode->subp_streams; link; link = link->next)
  {
    subp_source = link->data;

    if (subp_source->codec != G_TYPE_NONE &&
        !ogmrip_plugin_can_contain_subp (encode->container, subp_source->codec))
    {
      g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
          _("The container and the subtitles codec are incompatible."), 
          _("Please, choose some others."));
      return FALSE;
    }
  }

  for (link = encode->audio_files; link; link = link->next)
  {
    if (!ogmrip_plugin_can_contain_format (encode->container, 
          ogmrip_file_get_format (OGMRIP_FILE (link->data))))
    {
      g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
          _("The container and the audio file are incompatible."), 
          _("Please, choose some others."));
      return FALSE;
    }
  }

  for (link = encode->subp_files; link; link = link->next)
  {
    if (!ogmrip_plugin_can_contain_format (encode->container, 
          ogmrip_file_get_format (OGMRIP_FILE (link->data))))
    {
      g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
          _("The container and the subtitles file are incompatible."), 
          _("Please, choose some others."));
      return FALSE;
    }
  }

  return TRUE;
}

/*
 * Check if container can contain multiple streams
 */
static gboolean
ogmrip_main_check_streams (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  OGMRipAudioSource *audio_source;
  GSList *link;
  gint nstreams;

  if (encode->container == G_TYPE_NONE)
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s\n%s",
        _("No container has been selected."),
        _("This is an unexpected error and should not happen."),
        _("Please, fill a bug report at http://ogmrip.sf.net."));
    return FALSE;
  }

  nstreams = g_slist_length (encode->audio_streams) + g_slist_length (encode->audio_files);
  if (!ogmrip_plugin_can_contain_n_audio (encode->container, nstreams))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The selected container does not support multiple audio streams"), 
        _("Please, choose one audio stream only."));
    return FALSE;
  }

  nstreams = g_slist_length (encode->subp_streams) + g_slist_length (encode->subp_files);
  if (!ogmrip_plugin_can_contain_n_subp (encode->container, nstreams))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The selected container does not support multiple subtitles streams"), 
        _("Please, choose one subtitles stream only."));
    return FALSE;
  }

  for (link = encode->audio_streams; link; link = link->next)
  {
    audio_source = link->data;

    if (ogmdvd_audio_stream_get_format (audio_source->stream) == OGMDVD_AUDIO_FORMAT_DTS)
    {
#ifndef HAVE_DTS_SUPPORT
      g_set_error (error, OGMRIP_ERROR, 0,
          _("Your version of MPlayer does not support DTS streams")); 
      return FALSE;
#endif

      if (!ogmrip_plugin_can_contain_format (encode->container, OGMRIP_FORMAT_DTS))
      {
        g_set_error (error, OGMRIP_ERROR, 0,
            _("The selected container does not support DTS streams")); 
        return FALSE;
      }
    }
  }

  for (link = encode->audio_files; link; link = link->next)
  {
    if (ogmrip_file_get_format (OGMRIP_FILE (link->data)) == OGMRIP_FORMAT_DTS)
    {
#ifndef HAVE_DTS_SUPPORT
      g_set_error (error, OGMRIP_ERROR, 0,
          _("Your version of MPlayer does not support DTS streams"));
      return FALSE;
#endif

      if (ogmrip_plugin_can_contain_format (encode->container, OGMRIP_FORMAT_DTS))
      {
        g_set_error (error, OGMRIP_ERROR, 0,
            _("The selected container does not support DTS streams")); 
        return FALSE;
      }
    }
  }

  return TRUE;
}

/*
 * Checks if the output file already exists
 */
static gboolean
ogmrip_main_check_filename (OGMRipData *data, OGMRipEncode *encode)
{
  if (encode->ccheck)
    return TRUE;

  /*
   * TODO What if multiple targets ?
   */
  if (g_file_test (encode->outfile, G_FILE_TEST_EXISTS))
  {
    gint response;

    response = ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_QUESTION,
        _("A file named '%s' already exists.\nDo you want to replace it?"), encode->outfile);

    if (response == GTK_RESPONSE_NO)
      return FALSE;

    g_unlink (encode->outfile);
  }

  return TRUE;
}

/*
 * Checks if there is enough space on the drive
 */
static gboolean
ogmrip_main_check_space (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  gchar *output_mount_point, *tmp_mount_point;
  gint64 sync_size, dvd_size;
  gboolean retval = FALSE;

  /*
   * TODO voir taille requise quand ccheck
   */

  output_mount_point = ogmrip_fs_get_mount_point (encode->outfile);
  tmp_mount_point = ogmrip_fs_get_mount_point (ogmrip_fs_get_tmp_dir ());

  sync_size = ogmrip_main_get_sync_size (data, encode);
  dvd_size = ogmrip_main_get_dvd_size (data, encode);

  if (output_mount_point && tmp_mount_point)
  {
    if (strcmp (tmp_mount_point, output_mount_point) == 0)
    {
      retval = encode->rip_size * 2 + dvd_size + sync_size < ogmrip_fs_get_left_space (encode->outfile);
      if (!retval)
      {
        gint64 needed = (2 * encode->rip_size + dvd_size) / 1024 / 1024;
        gchar *needed_str = g_strdup_printf ("%" G_GUINT64_FORMAT,needed);

        g_set_error (error, OGMRIP_ERROR, 0, 
            _("Not enough space to store output and temporary files (%sMB needed)."), needed_str);
        g_free (needed_str);
      }
    }
    else
    {
      retval = encode->rip_size + dvd_size + sync_size < ogmrip_fs_get_left_space (ogmrip_fs_get_tmp_dir ());
      if (!retval)
      {
        gint64 needed = (encode->rip_size + dvd_size + sync_size) / 1024 / 1024;
        gchar *needed_str = g_strdup_printf ("%" G_GUINT64_FORMAT, needed);

        g_set_error (error, OGMRIP_ERROR, 0, 
            _("Not enough space to store the temporary files (%sMB needed)."), needed_str);
        g_free (needed_str);
      }
      else
      {
        retval = encode->rip_size < ogmrip_fs_get_left_space (encode->outfile);
        if (!retval)
        {
          gint64 needed = encode->rip_size / 1024 / 1024;
          gchar *needed_str = g_strdup_printf ("%" G_GUINT64_FORMAT, needed);

          g_set_error (error, OGMRIP_ERROR, 0, 
              _("Not enough space to store the output file (%sMB needed)."), needed_str);
          g_free (needed_str);
        }
      }
    }
  }

  g_free (output_mount_point);
  g_free (tmp_mount_point);

  return retval;
}

/*
 * Makes a list of the selected files
 */
static void
ogmrip_main_source_chooser_foreach_files (OGMRipSourceChooser *chooser, GSList **list)
{
  OGMRipSource *source;

  source = ogmrip_source_chooser_get_active (OGMRIP_SOURCE_CHOOSER (chooser), NULL);

  *list = g_slist_append (*list, source);
}

/*
 * Retrieves the default audio settings
 */
static void
ogmrip_main_audio_source_get_defaults (OGMRipAudioSource *source)
{
  source->codec = ogmrip_gconf_get_audio_codec_type (NULL);

  source->srate = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_SRATE, OGMRIP_DEFAULT_AUDIO_SRATE);
  source->quality = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_QUALITY, OGMRIP_DEFAULT_AUDIO_QUALITY);
  source->channels = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_CHANNELS, OGMRIP_DEFAULT_AUDIO_CHANNELS);
  source->normalize = ogmrip_preferences_get_bool (OGMRIP_GCONF_AUDIO_NORMALIZE, OGMRIP_DEFAULT_AUDIO_NORMALIZE);
}

/*
 * Makes a list of the selected audio streams
 */
static void
ogmrip_main_audio_chooser_foreach_streams (OGMRipSourceChooser *chooser, GSList **list)
{
  OGMRipAudioSource *audio_source;
  OGMRipSource *source;

  audio_source = g_object_get_data (G_OBJECT (chooser), "__audio_source__");
  if (audio_source)
    audio_source = g_memdup (audio_source, sizeof (OGMRipAudioSource));
  else
  {
    audio_source = g_new0 (OGMRipAudioSource, 1);
    ogmrip_main_audio_source_get_defaults (audio_source);
  }

  source = ogmrip_source_chooser_get_active (OGMRIP_SOURCE_CHOOSER (chooser), NULL);
  audio_source->stream = OGMDVD_AUDIO_STREAM (source);

  *list = g_slist_append (*list, audio_source);
}

/*
 * Retrieves the default subp settings
 */
static void
ogmrip_main_subp_source_get_defaults (OGMRipSubpSource *source)
{
  source->codec = ogmrip_gconf_get_subp_codec_type (NULL);

  source->eol = ogmrip_preferences_get_int (OGMRIP_GCONF_SUBP_EOL, OGMRIP_DEFAULT_SUBP_EOL);
  source->charset = ogmrip_preferences_get_int (OGMRIP_GCONF_SUBP_CHARSET, OGMRIP_DEFAULT_SUBP_CHARSET);
  source->forced_subs = ogmrip_preferences_get_bool (OGMRIP_GCONF_FORCED_SUBS, OGMRIP_DEFAULT_FORCED_SUBS);
  source->spell = ogmrip_preferences_get_bool (OGMRIP_GCONF_SPELL_CHECK, OGMRIP_DEFAULT_SPELL_CHECK);
}

/*
 * Makes a list of the selected subp streams
 */
static void
ogmrip_main_subp_chooser_foreach_streams (OGMRipSourceChooser *chooser, GSList **list)
{
  OGMRipSubpSource *subp_source;
  OGMRipSource *source;

  subp_source = g_object_get_data (G_OBJECT (chooser), "__subp_source__");
  if (subp_source)
    subp_source = g_memdup (subp_source, sizeof (OGMRipSubpSource));
  else
  {
    subp_source = g_new0 (OGMRipSubpSource, 1);
    ogmrip_main_subp_source_get_defaults (subp_source);
  }

  source = ogmrip_source_chooser_get_active (OGMRIP_SOURCE_CHOOSER (chooser), NULL);
  subp_source->stream = OGMDVD_SUBP_STREAM (source);

  *list = g_slist_append (*list, subp_source);
}

/*
 * Encodes the video stream pass
 */
static gint
ogmrip_main_extract_video_stream_pass (OGMRipData *data, OGMRipEncode *encode, OGMJobSpawn *spawn, guint pass, const gchar *logfile, GError **error)
{
  gchar *message;

  ogmrip_video_set_pass (OGMRIP_VIDEO (spawn), pass);

  if (encode->ccheck)
  {
    if (pass == 0)
      message = g_strdup_printf (_("Testing compressibility of video title %d"), ogmdvd_title_get_nr (encode->title) + 1);
    else if (pass == 1 || pass == 2)
      message = g_strdup_printf (_("Testing compressibility of video title %d, Pass %d"), ogmdvd_title_get_nr (encode->title) + 1, pass);
    else
      message = g_strdup_printf (_("Testing compressibility of video title %d, Pass %d"), ogmdvd_title_get_nr (encode->title) + 1, pass - 1);
  }
  else
  {
    if (pass == 0)
      message = g_strdup_printf (_("Encoding video title %d"), ogmdvd_title_get_nr (encode->title) + 1);
    else if (pass == 1 || pass == 2)
      message = g_strdup_printf (_("Encoding video title %d, Pass %d"), ogmdvd_title_get_nr (encode->title) + 1, pass);
    else
      message = g_strdup_printf (_("Encoding video title %d, Pass %d"), ogmdvd_title_get_nr (encode->title) + 1, pass - 1);
  }

  ogmrip_progress_dialog_set_spawn (OGMRIP_PROGRESS_DIALOG (data->progress_dialog), spawn, message);
  g_free (message);

  return ogmjob_spawn_run (spawn, error);
}

/*
 * Encodes the video stream
 */
static gint
ogmrip_main_extract_video_stream (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMJobSpawn *spawn;

  gboolean turbo;
  gint64 nonvideo_size, overhead_size;
  guint x, y, width, height, start_delay;
  gchar *output;
  gdouble bpp;
  gint result;

  output = ogmrip_fs_mktemp ("video.XXXXXX", error);
  if (!output)
  {
    g_set_error (error, OGMRIP_ERROR, 0, _("Cannot create temporary file '%s'"), output);
    return OGMJOB_RESULT_ERROR;
  }

  spawn = g_object_new (encode->video_codec, "input", encode->title, "output", output, NULL);
  g_free (output);

  ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
  ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);
  ogmrip_codec_set_edl (OGMRIP_CODEC (spawn), encode->edl);

  ogmrip_video_set_pullup (OGMRIP_VIDEO (spawn), encode->pullup);

  ogmrip_video_set_quality (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_int (OGMRIP_GCONF_VIDEO_QUALITY, OGMRIP_DEFAULT_VIDEO_QUALITY));
  ogmrip_video_set_denoise (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_bool (OGMRIP_GCONF_VIDEO_DENOISE, OGMRIP_DEFAULT_VIDEO_DENOISE));
  ogmrip_video_set_deblock (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_bool (OGMRIP_GCONF_VIDEO_DEBLOCK, OGMRIP_DEFAULT_VIDEO_DEBLOCK));
  ogmrip_video_set_dering (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_bool (OGMRIP_GCONF_VIDEO_DERING, OGMRIP_DEFAULT_VIDEO_DERING));
  ogmrip_video_set_trellis (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_bool (OGMRIP_GCONF_VIDEO_TRELLIS, OGMRIP_DEFAULT_VIDEO_TRELLIS));
  ogmrip_video_set_qpel (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_bool (OGMRIP_GCONF_VIDEO_QPEL, OGMRIP_DEFAULT_VIDEO_QPEL));
  ogmrip_video_set_scaler (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_int (OGMRIP_GCONF_VIDEO_SCALER, OGMRIP_DEFAULT_VIDEO_SCALER));
  ogmrip_video_set_threads (OGMRIP_VIDEO (spawn),
      ogmrip_preferences_get_int (OGMRIP_GCONF_VIDEO_THREADS, OGMRIP_DEFAULT_VIDEO_THREADS));

  ogmrip_video_set_deinterlacer (OGMRIP_VIDEO (spawn),
      ogmrip_options_dialog_get_deinterlacer (OGMRIP_OPTIONS_DIALOG (data->options_dialog)));
  ogmrip_codec_set_framestep (OGMRIP_CODEC (spawn),
      ogmrip_options_dialog_get_framestep (OGMRIP_OPTIONS_DIALOG (data->options_dialog)));
  ogmrip_video_set_cartoon (OGMRIP_VIDEO (spawn),
      ogmrip_options_dialog_get_cartoon (OGMRIP_OPTIONS_DIALOG (data->options_dialog)));

  turbo = ogmrip_preferences_get_bool (OGMRIP_GCONF_VIDEO_TURBO, OGMRIP_DEFAULT_VIDEO_TURBO);

  if (encode->progressive || encode->pullup)
    ogmrip_codec_set_framerate (OGMRIP_CODEC (spawn), 24000, 1001);

  if (!encode->ccheck && encode->ensure_sync && encode->audio_streams)
  {
    OGMRipAudioSource *audio_source = encode->audio_streams->data;
    ogmrip_video_set_ensure_sync (OGMRIP_VIDEO (spawn), audio_source->stream);
  }

  ogmrip_container_set_video (container, OGMRIP_VIDEO (spawn));
  g_object_unref (spawn);

  nonvideo_size = ogmrip_container_get_nonvideo_size (container);
  overhead_size = ogmrip_container_get_overhead_size (container);

  start_delay = ogmrip_video_get_start_delay (OGMRIP_VIDEO (spawn));
  if (start_delay > 0)
    ogmrip_container_set_start_delay (container, start_delay);

  ogmrip_progress_dialog_set_spawn (OGMRIP_PROGRESS_DIALOG (data->progress_dialog), spawn, _("Calculating bitrate, cropping, scaling"));

  if (ogmrip_options_dialog_get_bitrate (OGMRIP_OPTIONS_DIALOG (data->options_dialog), &encode->bitrate) == OGMRIP_OPTIONS_AUTOMATIC)
  {
    ogmrip_video_autobitrate (OGMRIP_VIDEO (spawn), nonvideo_size, overhead_size, encode->rip_size);
    encode->bitrate = ogmrip_video_get_bitrate (OGMRIP_VIDEO (spawn));
  }
  else
    ogmrip_video_set_bitrate (OGMRIP_VIDEO (spawn), encode->bitrate);

  switch (ogmrip_options_dialog_get_crop (OGMRIP_OPTIONS_DIALOG (data->options_dialog), &x, &y, &width, &height))
  {
    case OGMRIP_OPTIONS_AUTOMATIC:
      if (!ogmrip_video_autocrop (OGMRIP_VIDEO (spawn), 0))
        return OGMJOB_RESULT_CANCEL;
      break;
    case OGMRIP_OPTIONS_MANUAL:
      ogmrip_video_set_crop_size (OGMRIP_VIDEO (spawn), x, y, width, height);
      break;
    default:
      break;
  }

  switch (ogmrip_options_dialog_get_scale (OGMRIP_OPTIONS_DIALOG (data->options_dialog), &width, &height))
  {
    case OGMRIP_OPTIONS_AUTOMATIC:
        bpp = ogmrip_options_dialog_get_bits_per_pixel (OGMRIP_OPTIONS_DIALOG (data->options_dialog));
        ogmrip_video_set_bits_per_pixel (OGMRIP_VIDEO (spawn), bpp);
        ogmrip_video_autoscale (OGMRIP_VIDEO (spawn));
      break;
    case OGMRIP_OPTIONS_MANUAL:
      ogmrip_video_set_scale_size (OGMRIP_VIDEO (spawn), width, height);
      break;
    default:
      break;
  }

  if (encode->ccheck)
    ogmrip_video_set_quantizer (OGMRIP_VIDEO (spawn), 2.3);

  if (encode->passes == 1)
    result = ogmrip_main_extract_video_stream_pass (data, encode, spawn, 0, NULL, error);
  else
  {
    gchar *logfile;

    logfile = ogmrip_main_mklog ();
    ogmrip_video_set_log (OGMRIP_VIDEO (spawn), logfile);

    ogmrip_video_set_turbo (OGMRIP_VIDEO (spawn), turbo);
    result = ogmrip_main_extract_video_stream_pass (data, encode, spawn, 1, logfile, error);

    if (result == OGMJOB_RESULT_COMPLETED)
    {
      if (encode->passes == 2)
      {
        ogmrip_video_set_turbo (OGMRIP_VIDEO (spawn), FALSE);
        result = ogmrip_main_extract_video_stream_pass (data, encode, spawn, 2, logfile, error);
      }
      else
      {
        int pass;

        for (pass = 1; pass < encode->passes; pass ++)
        {
          if (pass == encode->passes - 1)
            ogmrip_video_set_turbo (OGMRIP_VIDEO (spawn), FALSE);
          result = ogmrip_main_extract_video_stream_pass (data, encode, spawn, pass + 2, logfile, error);
          if (result != OGMJOB_RESULT_COMPLETED)
            break;
        }
      }
    }

    g_unlink (logfile);
    g_free (logfile);
  }

  if (result == OGMJOB_RESULT_ERROR && error && !(*error))
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("Unknown error while extracting video stream"),
        _("Please, check http://ogmrip.sf.net to see if this is a known issue."));

  return result;
}

/*
 * Encodes the audio streams
 */
static gint
ogmrip_main_extract_audio_streams (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMRipAudioDemuxer demuxer;
  OGMRipAudioSource *source;
  OGMJobSpawn *spawn;

  GSList *link;
  gint framestep, language, result;
  gchar *output, *message;

  static const gint sample_rate[] =
  { 48000, 44100, 32000, 24000, 22050, 16000, 12000, 11025, 8000 };
  static const gint channels[] = 
  { OGMDVD_AUDIO_CHANNELS_MONO, OGMDVD_AUDIO_CHANNELS_STEREO, OGMDVD_AUDIO_CHANNELS_SURROUND, OGMDVD_AUDIO_CHANNELS_5_1 };

  framestep = ogmrip_options_dialog_get_framestep (OGMRIP_OPTIONS_DIALOG (data->options_dialog));

  for (link = encode->audio_streams; link; link = link->next)
  {
    source = link->data;

    if (source->codec != G_TYPE_NONE)
    {
      output = ogmrip_fs_mktemp ("audio.XXXXXX", error);
      if (!output)
      {
        g_set_error (error, OGMRIP_ERROR, 0, _("Cannot create temporary file '%s'"), output);
        return OGMJOB_RESULT_ERROR;
      }

      spawn = g_object_new (source->codec, "stream", source->stream, "output", output, NULL);
      g_free (output);

      ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
      ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);
      ogmrip_codec_set_framestep (OGMRIP_CODEC (spawn), framestep);
      ogmrip_codec_set_edl (OGMRIP_CODEC (spawn), encode->edl);

      ogmrip_audio_set_quality (OGMRIP_AUDIO (spawn), source->quality);
      ogmrip_audio_set_normalize (OGMRIP_AUDIO (spawn), source->normalize);
      ogmrip_audio_set_sample_rate (OGMRIP_AUDIO (spawn), sample_rate[source->srate]);
      ogmrip_audio_set_channels (OGMRIP_AUDIO (spawn), channels[source->channels]);
      ogmrip_audio_set_fast (OGMRIP_AUDIO (spawn), !encode->ensure_sync);

      if (encode->progressive || encode->pullup)
        ogmrip_codec_set_framerate (OGMRIP_CODEC (spawn), 24000, 1001);

      demuxer = OGMRIP_AUDIO_DEMUXER_AUTO;
      if (ogmrip_plugin_get_audio_codec_format (source->codec) == OGMRIP_FORMAT_COPY)
      {
        switch (ogmdvd_audio_stream_get_format (source->stream))
        {
          case OGMDVD_AUDIO_FORMAT_AC3:
            demuxer = OGMRIP_AUDIO_DEMUXER_AC3;
            break;
          case OGMDVD_AUDIO_FORMAT_DTS:
            demuxer = OGMRIP_AUDIO_DEMUXER_DTS;
            break;
          default:
            break;
        }
      }

      language = ogmdvd_audio_stream_get_language (source->stream);
      ogmrip_container_add_audio (container, OGMRIP_AUDIO (spawn), demuxer, language);
      g_object_unref (spawn);

      message = g_strdup_printf (_("Extracting audio stream %d"), 
          ogmdvd_stream_get_nr (OGMDVD_STREAM (source->stream)) + 1);
      ogmrip_progress_dialog_set_spawn (OGMRIP_PROGRESS_DIALOG (data->progress_dialog), spawn, message);
      g_free (message);

      if ((result = ogmjob_spawn_run (spawn, error)) != OGMJOB_RESULT_COMPLETED)
      {
        if (result == OGMJOB_RESULT_ERROR && error && !(*error))
          g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
              _("Unknown error while extracting audio stream"),
              _("Please, check http://ogmrip.sf.net to see if this is a known issue."));
        return result;
      }
    }
  }

  return OGMJOB_RESULT_COMPLETED;
}

/*
 * Encodes the subp streams
 */
static gint
ogmrip_main_extract_subp_streams (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMRipSubpDemuxer demuxer;
  OGMRipSubpSource *source;
  OGMJobSpawn *spawn;
  GSList *link;

  gint language, result;
  gchar *output, *message;

  for (link = encode->subp_streams; link; link = link->next)
  {
    source = link->data;

    if (source->codec != G_TYPE_NONE)
    {
      output = ogmrip_fs_mktemp ("subp.XXXXXX", error);
      if (!output)
      {
        g_set_error (error, OGMRIP_ERROR, 0, _("Cannot create temporary file '%s'"), output);
        return OGMJOB_RESULT_ERROR;
      }

      spawn = g_object_new (source->codec, "stream", source->stream, "output", output, NULL);
      g_free (output);

      ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
      ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);
      ogmrip_codec_set_edl (OGMRIP_CODEC (spawn), encode->edl);

      ogmrip_subp_set_charset (OGMRIP_SUBP (spawn), source->charset);
      ogmrip_subp_set_eol_style (OGMRIP_SUBP (spawn), source->eol);
      ogmrip_subp_set_forced_only (OGMRIP_SUBP (spawn), source->forced_subs);

      if (encode->progressive || encode->pullup)
        ogmrip_codec_set_framerate (OGMRIP_CODEC (spawn), 24000, 1001);

      demuxer = OGMRIP_SUBP_DEMUXER_AUTO;
      if (ogmrip_plugin_get_subp_codec_format (source->codec) == OGMRIP_FORMAT_VOBSUB)
        demuxer = OGMRIP_SUBP_DEMUXER_VOBSUB;

      language = ogmdvd_subp_stream_get_language (source->stream);
      ogmrip_container_add_subp (container, OGMRIP_SUBP (spawn), demuxer, language);
      g_object_unref (spawn);

      message = g_strdup_printf (_("Extracting subtitle stream %d"),
          ogmdvd_stream_get_nr (OGMDVD_STREAM (source->stream)) + 1);
      ogmrip_progress_dialog_set_spawn (OGMRIP_PROGRESS_DIALOG (data->progress_dialog), spawn, message);
      g_free (message);

      if ((result = ogmjob_spawn_run (spawn, error)) != OGMJOB_RESULT_COMPLETED)
      {
        if (result == OGMJOB_RESULT_ERROR && error && !(*error))
          g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
              _("Unknown error while extracting subtitle stream"),
              _("Please, check http://ogmrip.sf.net to see if this is a known issue."));

        return result;
      }

#ifdef HAVE_ENCHANT_SUPPORT
      if (ogmrip_plugin_get_subp_codec_text (source->codec) && source->spell)
        ogmrip_main_spell_check (data, OGMRIP_SUBP (spawn));
#endif /* HAVE_ENCHANT_SUPPORT */
    }
  }

  return OGMJOB_RESULT_COMPLETED;
}

/*
 * Extracts chapter information
 */
static gint
ogmrip_main_extract_chapters (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMJobSpawn *spawn;
  gchar *output, *message, *label;
  gint result, chapter, lang;

  output = ogmrip_fs_mktemp ("chapters.XXXXXX", error);
  if (!output)
  {
    g_set_error (error, OGMRIP_ERROR, 0, _("Cannot create temporary file '%s'"), output);
    return OGMJOB_RESULT_ERROR;
  }

  spawn = ogmrip_chapters_new (encode->title, output);
  g_free (output);

  for (chapter = encode->start_chap; chapter <= encode->end_chap; chapter ++)
  {
    if ((label = ogmdvd_chapter_list_get_label (OGMDVD_CHAPTER_LIST (data->chapter_list), chapter)))
    {
      ogmrip_chapters_set_label (OGMRIP_CHAPTERS (spawn), chapter, label);
      g_free (label);
    }
  }

  ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
  ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);
  ogmrip_codec_set_edl (OGMRIP_CODEC (spawn), encode->edl);

  lang = ogmrip_preferences_get_int (OGMRIP_GCONF_CHAPTER_LANG, OGMRIP_DEFAULT_CHAPTER_LANG);
  ogmrip_container_add_chapters (container, OGMRIP_CHAPTERS (spawn), lang);
  g_object_unref (spawn);

  message = g_strdup_printf (_("Extracting chapters information"));
  ogmrip_progress_dialog_set_spawn (OGMRIP_PROGRESS_DIALOG (data->progress_dialog), spawn, message);
  g_free (message);

  if ((result = ogmjob_spawn_run (spawn, error)) != OGMJOB_RESULT_COMPLETED)
  {
    if (result == OGMJOB_RESULT_ERROR && error && !(*error))
      g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
          _("Unknown error while extracting chapters"),
          _("Please, check http://ogmrip.sf.net to see if this is a known issue."));

    return result;
  }

  return OGMJOB_RESULT_COMPLETED;
}

/*
 * Creates the container
 */
static OGMRipContainer *
ogmrip_main_container_new (OGMRipData *data, OGMRipEncode *encode)
{
  OGMJobSpawn *spawn;
  gint fcc;

  static const gchar *fourcc[] =
  {
    NULL,
    "XVID",
    "DIVX",
    "DX50",
    "FMP4"
  };

  spawn = g_object_new (encode->container, "output", encode->outfile, NULL);

  ogmrip_container_set_split (OGMRIP_CONTAINER (spawn), encode->target_number, encode->target_size);
  ogmrip_container_set_label (OGMRIP_CONTAINER (spawn), encode->label);

  fcc = ogmrip_preferences_get_int (OGMRIP_GCONF_FOURCC, OGMRIP_DEFAULT_FOURCC);
  if (fcc >= 0 && fcc <= 4)
    ogmrip_container_set_fourcc (OGMRIP_CONTAINER (spawn), fourcc[fcc]);
  else
    ogmrip_container_set_fourcc (OGMRIP_CONTAINER (spawn), NULL);

  return OGMRIP_CONTAINER (spawn);
}

/*
 * Adds external files to the container
 */
static gboolean
ogmrip_main_container_add_files (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  GSList *link;

  for (link = encode->audio_files; link; link = link->next)
    ogmrip_container_add_file (container, OGMRIP_FILE (link->data));

  for (link = encode->subp_files; link; link = link->next)
    ogmrip_container_add_file (container, OGMRIP_FILE (link->data));

  return TRUE;
}

/*
 * Merges the streams together
 */
static gint
ogmrip_main_merge (OGMRipData *data, OGMRipContainer *container, GError **error)
{
  gint result;
#if !MPLAYER_CHECK_VERSION(1,0,0,8)
  gchar *cwd;
#endif /* !MPLAYER_CHECK_VERSION(1,0,0,8) */

  ogmrip_progress_dialog_set_spawn (OGMRIP_PROGRESS_DIALOG (data->progress_dialog), 
      OGMJOB_SPAWN (container), _("Merging audio and video streams"));

#if !MPLAYER_CHECK_VERSION(1,0,0,8)
  cwd = g_get_current_dir ();
  g_chdir (ogmrip_fs_get_tmp_dir ());
#endif /* !MPLAYER_CHECK_VERSION(1,0,0,8) */

  result = ogmjob_spawn_run (OGMJOB_SPAWN (container), error);

  if (result == OGMJOB_RESULT_ERROR && error && !(*error))
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("Unknown error while merging"),
        _("Please, check http://ogmrip.sf.net to see if this is a known issue."));

#if !MPLAYER_CHECK_VERSION(1,0,0,8)
  g_chdir (cwd);
  g_free (cwd);
#endif /* !MPLAYER_CHECK_VERSION(1,0,0,8) */

  return result;
}

/*
 * Extracts a DVD title
 */
static gboolean
ogmrip_main_extract (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  OGMRipContainer *container;
  gboolean result = FALSE;

  encode->keep_temp = ogmrip_preferences_get_bool (OGMRIP_GCONF_KEEP_TMP, OGMRIP_DEFAULT_KEEP_TMP);

  container = ogmrip_main_container_new (data, encode);
  if (!container)
    return FALSE;

  if (!ogmrip_main_container_add_files (data, encode, container, error))
    goto extract_cleanup;

  result = ogmrip_main_extract_chapters (data, encode, container, error);
  if (result != OGMJOB_RESULT_COMPLETED)
    goto extract_cleanup;

  result = ogmrip_main_extract_subp_streams (data, encode, container, error);
  if (result != OGMJOB_RESULT_COMPLETED)
    goto extract_cleanup;

  result = ogmrip_main_extract_audio_streams (data, encode, container, error);
  if (result != OGMJOB_RESULT_COMPLETED)
    goto extract_cleanup;

  if (encode->video_codec != G_TYPE_NONE)
  {
    result = ogmrip_main_extract_video_stream (data, encode, container, error);
    if (result != OGMJOB_RESULT_COMPLETED)
      goto extract_cleanup;
  }

  result = ogmrip_main_merge (data, container, error);

extract_cleanup:
  g_object_unref (container);

  return result != OGMJOB_RESULT_ERROR;
}

/*
 * Performs a compressibility test on a DVD title
 */
static gboolean
ogmrip_main_ccheck (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  gint result = OGMJOB_RESULT_ERROR;
  OGMRipContainer *container;
  OGMRipVideo *video;
  OGMRipFile *file;

  GtkWidget *dialog;

  gchar *message;
  const gchar *filename;
  gdouble user_quality, optimal_quality, cfactor;
  guint width, height;
  gint bitrate;
  gdouble fps;

  encode->keep_temp = ogmrip_preferences_get_bool (OGMRIP_GCONF_KEEP_TMP, OGMRIP_DEFAULT_KEEP_TMP);

  container = ogmrip_main_container_new (data, encode);
  if (!container)
    return FALSE;

  if (ogmrip_options_dialog_get_bitrate (OGMRIP_OPTIONS_DIALOG (data->options_dialog), NULL) == OGMRIP_OPTIONS_AUTOMATIC)
  {
    result = ogmrip_main_extract_chapters (data, encode, container, error);
    if (result != OGMJOB_RESULT_COMPLETED)
      goto ccheck_cleanup;

    if (!ogmrip_main_container_add_files (data, encode, container, error))
      goto ccheck_cleanup;

    result = ogmrip_main_extract_subp_streams (data, encode, container, error);
    if (result != OGMJOB_RESULT_COMPLETED)
      goto ccheck_cleanup;

    result = ogmrip_main_extract_audio_streams (data, encode, container, error);
    if (result != OGMJOB_RESULT_COMPLETED)
      goto ccheck_cleanup;
  }

  if (encode->video_codec != G_TYPE_NONE)
  {
    result = ogmrip_main_extract_video_stream (data, encode, container, error);
    if (result != OGMJOB_RESULT_COMPLETED)
      goto ccheck_cleanup;
  }

  video = ogmrip_container_get_video (container);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));

  file = ogmrip_video_file_new (filename, error);
  if (!file)
    goto ccheck_cleanup;

  fps = ogmrip_video_file_get_framerate (OGMRIP_VIDEO_FILE (file));
  bitrate = ogmrip_video_file_get_bitrate (OGMRIP_VIDEO_FILE (file));
  ogmrip_video_file_get_size (OGMRIP_VIDEO_FILE (file), &width, &height);

  ogmrip_file_unref (file);

  dialog = ogmrip_message_dialog_new (GTK_WINDOW (data->window),
      GTK_MESSAGE_INFO, _("Compressibility test results"));

  user_quality = encode->bitrate / (width * height * fps);
  optimal_quality = bitrate / (width * height * fps);
  cfactor = user_quality / optimal_quality * 100;

  message = g_strdup_printf (_("This DVD title will be encoded at %u kbps with a resolution of %ux%u pixels for a quality factor of %.2lf. "
        "With this resolution, the optimal bitrate is %u kbps for a quality factor of %.2lf. "
        "This gives a compressibility factor of %.0lf%%"),
      encode->bitrate / 1000, width, height, user_quality, bitrate / 1000, optimal_quality, cfactor);

  if (cfactor < 55.0)
    gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dialog), "%s %s\n\n<b>%s</b>",
        message, _("which is a bit to low."), _("You should decrease the resolution."));
  else if (cfactor > 65.0)
    gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dialog), "%s %s\n\n<b>%s</b>",
        message, _("which is a bit to high."), _("You can safely increase the resolution."));
  else
    gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dialog), "%s.\n\n<b>%s</b>",
        message, _("You can safely encode using those settings.")); 

  g_free (message);

  g_signal_connect (dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
  gtk_widget_show (dialog);
  /*
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  */

  /*
   * TODO
   * Reporter les paramètres dans la fenêtre d'options ?
   * Afficher la fenêtre d'options
   */

ccheck_cleanup:
  g_object_unref (container);

  return result != OGMJOB_RESULT_ERROR;
}

/*
 * Backups a DVD title
 */
static gint
ogmrip_main_backup (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  OGMDvdDisc *disc;
  OGMRipAudioSource *audio_source;
  OGMRipSubpSource *subp_source;
  GSList *link;

  OGMJobSpawn *spawn;
  gint result, nr;
  gchar *path;

  path = ogmrip_fs_mkdtemp ("dvdXXXXXX", error);
  if (!path)
  {
    g_set_error (error, OGMRIP_ERROR, 0, _("Cannot create temporary directory '%s'"), path);
    return OGMJOB_RESULT_ERROR;
  }

  spawn = ogmrip_dvdcpy_new (encode->title, path);
  ogmrip_progress_dialog_set_spawn (OGMRIP_PROGRESS_DIALOG (data->progress_dialog), spawn, _("DVD backup"));
  g_object_unref (spawn);

  result = ogmjob_spawn_run (spawn, error);
  if (result != OGMJOB_RESULT_COMPLETED)
  {
    if (result == OGMJOB_RESULT_ERROR && error && !(*error))
      g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
          _("Unknown error while copying the DVD on the hard drive"),
          _("Please, check http://ogmrip.sf.net to see if this is a known issue."));

    ogmrip_fs_rmdir (path, TRUE, NULL);
    g_free (path);

    return result;
  }

  disc = ogmdvd_disc_open (path, error);
  g_free (path);

  if (!disc)
    return OGMJOB_RESULT_ERROR;

  nr = ogmdvd_title_get_nr (encode->title);
  ogmdvd_title_unref (encode->title);

  encode->title = ogmdvd_disc_get_nth_title (disc, nr);
  ogmdvd_disc_unref (disc);

  if (!encode->title)
  {
    g_set_error (error, OGMRIP_ERROR, 0, "Cannot open video title set for title %d", nr);
    return OGMJOB_RESULT_ERROR;
  }

  for (link = encode->audio_streams; link; link = link->next)
  {
    audio_source = link->data;
    nr = ogmdvd_stream_get_nr (OGMDVD_STREAM (audio_source->stream));
    ogmdvd_stream_unref (OGMDVD_STREAM (audio_source->stream));

    audio_source->stream = ogmdvd_title_get_nth_audio_stream (encode->title, nr);
  }

  for (link = encode->subp_streams; link; link = link->next)
  {
    subp_source = link->data;
    nr = ogmdvd_stream_get_nr (OGMDVD_STREAM (subp_source->stream));
    ogmdvd_stream_unref (OGMDVD_STREAM (subp_source->stream));

    subp_source->stream = ogmdvd_title_get_nth_subp_stream (encode->title, nr);
  }

  return OGMJOB_RESULT_COMPLETED;
}

/*
 * Loads a DVD disk or a DVD structure
 */
static void
ogmrip_main_load (OGMRipData *data, const gchar *path)
{
  OGMDvdDisc *disc;
  GError *error = NULL;

  disc = ogmdvd_disc_open (path, &error);
  if (!disc)
  {
    if (error)
    {
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s", 
          _("Could not open the DVD"), _(error->message));
      g_error_free (error);
    }
    else
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s",
          _("Could not read the DVD"), _("Unknown error"));
  }
  else
  {
    gchar *label = NULL;
    gint nvid;

    if (data->disc)
      ogmdvd_disc_unref (data->disc);
    data->disc = disc;

    ogmdvd_title_chooser_set_disc (OGMDVD_TITLE_CHOOSER (data->title_chooser), disc);

    nvid = ogmdvd_disc_get_n_titles (disc);
    if (nvid > 0)
      label = ogmdvd_disc_get_label (disc);

    gtk_widget_set_sensitive (data->title_chooser, nvid > 0);
    gtk_widget_set_sensitive (data->title_entry, nvid > 0);
    gtk_widget_set_sensitive (data->extract_button, nvid > 0);
    gtk_action_set_sensitive (data->extract_action, nvid > 0);
    gtk_action_set_sensitive (data->import_chap_action, nvid > 0);
    gtk_action_set_sensitive (data->export_chap_action, nvid > 0);

    gtk_entry_set_text (GTK_ENTRY (data->title_entry), label ? label : "");
    g_free (label);
  }
}

/*
 * Opens a simple chapter file
 */
gboolean
ogmrip_main_import_simple_chapters (OGMRipData *data, const gchar *filename)
{
  FILE *file;
  gchar buf[201], *str;
  gint chap;

  file = fopen (filename, "r");
  if (!file)
    return FALSE;

  while (!feof (file))
  {
    if (fgets (buf, 200, file) != NULL)
    {
      if (sscanf (buf, "CHAPTER%02dNAME=", &chap) == 1 && chap > 0)
      {
        str = g_strstrip (strchr (buf, '='));
        ogmdvd_chapter_list_set_label (OGMDVD_CHAPTER_LIST (data->chapter_list), chap - 1, str + 1);
      }
    }
  }

  fclose (file);

  return TRUE;
}

/*
 * Opens a matroska chapter file
 */
static gboolean
ogmrip_main_import_matroska_chapters (OGMRipData *data, const gchar *filename)
{
  xmlDoc *doc;
  xmlNode *node, *child;
  xmlChar *str;

  gint chap = 0;

  doc = xmlParseFile (filename);
  if (!doc)
    return FALSE;

  node = xmlDocGetRootElement (doc);
  if (!node)
  {
    xmlFreeDoc (doc);
    return FALSE;
  }

  if (!xmlStrEqual (node->name, (xmlChar *) "Chapters"))
  {
    xmlFreeDoc (doc);
    return FALSE;
  }

  for (node = node->children; node; node = node->next)
    if (xmlStrEqual (node->name, (xmlChar *) "EditionEntry"))
      break;

  if (!node)
  {
    xmlFreeDoc (doc);
    return FALSE;
  }

  for (node = node->children; node; node = node->next)
  {
    if (xmlStrEqual (node->name, (xmlChar *) "ChapterAtom"))
    {
      for (child = node->children; child; child = child->next)
        if (xmlStrEqual (child->name, (xmlChar *) "ChapterDisplay"))
          break;

      if (child)
      {
        for (child = child->children; child; child = child->next)
          if (xmlStrEqual (child->name, (xmlChar *) "ChapterString"))
            break;

        if (child)
        {
          str = xmlNodeGetContent (child);
          ogmdvd_chapter_list_set_label (OGMDVD_CHAPTER_LIST (data->chapter_list), chap, (gchar *) str);
          chap ++;
        }
      }
    }
  }

  xmlFreeDoc (doc);

  return TRUE;
}

/*
 * Saves chapter info in simple format
 */
void
ogmrip_main_export_simple_chapters (OGMRipData *data, const gchar *filename)
{
  guint start_chap;
  gint end_chap;

  if (ogmrip_chapter_list_get_selected (OGMRIP_CHAPTER_LIST (data->chapter_list), &start_chap, &end_chap))
  {
    OGMDvdTitle *title;
    OGMJobSpawn *spawn;
    GError *error = NULL;

    gchar *label;
    gint chap;

    title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
    spawn = ogmrip_chapters_new (title, filename);
    ogmdvd_title_unref (title);

    for (chap = start_chap; chap <= end_chap; chap ++)
    {
      if ((label = ogmdvd_chapter_list_get_label (OGMDVD_CHAPTER_LIST (data->chapter_list), chap)))
      {
        ogmrip_chapters_set_label (OGMRIP_CHAPTERS (spawn), chap, label);
        g_free (label);
      }
    }

    ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), FALSE);
    ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), start_chap, end_chap);

    if (ogmjob_spawn_run (spawn, &error) != OGMJOB_RESULT_COMPLETED)
    {
      if (error)
      {
        ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s",
            _("Could not export the chapters"), error->message);
        g_error_free (error);
      }
      else
        ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR,
            "<b>%s</b>\n\n%s", _("Unknown error while exporting the chapters"),
            _("Please, check http://ogmrip.sf.net to see if this is a known issue."));
    }
  }
}

/*
 * Clear the GUI
 */
static void
ogmrip_main_clear (OGMRipData *data)
{
  ogmdvd_title_chooser_set_disc (OGMDVD_TITLE_CHOOSER (data->title_chooser), NULL);

  gtk_widget_set_sensitive (data->title_chooser, FALSE);
  gtk_widget_set_sensitive (data->title_entry, FALSE);
  gtk_widget_set_sensitive (data->extract_button, FALSE);
  gtk_action_set_sensitive (data->extract_action, FALSE);
  gtk_action_set_sensitive (data->import_chap_action, FALSE);
  gtk_action_set_sensitive (data->export_chap_action, FALSE);

  gtk_entry_set_text (GTK_ENTRY (data->title_entry), "");
}

/*
 * Events
 */

/*
 * When the main window receives the delete event
 */
static gboolean
ogmrip_main_delete_event (OGMRipData *data)
{
  if (data->encoding)
    return TRUE;

  return FALSE;
}

/*
 * When the main window is destroyed
 */
static void
ogmrip_main_destroyed (OGMRipData *data)
{
  if (data->disc)
    ogmdvd_disc_unref (data->disc);
  data->disc = NULL;

  if (data->pref_dialog)
    gtk_widget_destroy (data->pref_dialog);
  data->pref_dialog = NULL;

  if (data->options_dialog)
    gtk_widget_destroy (data->options_dialog);
  data->options_dialog = NULL;

  g_free (data);
}

static void
ogmrip_main_pref_dialog_construct (OGMRipData *data)
{
  data->pref_dialog = ogmrip_pref_dialog_new ();
  gtk_window_set_parent (GTK_WINDOW (data->pref_dialog), GTK_WINDOW (data->window));

  g_signal_connect (data->pref_dialog, "destroy", G_CALLBACK (gtk_widget_destroyed), &data->pref_dialog);
  g_signal_connect (data->pref_dialog, "response", G_CALLBACK (gtk_widget_hide), NULL);
  g_signal_connect (data->pref_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
}

static void
ogmrip_main_options_dialog_construct (OGMRipData *data)
{
  data->options_dialog = ogmrip_options_dialog_new ();
  gtk_window_set_parent (GTK_WINDOW (data->options_dialog), GTK_WINDOW (data->window));

  g_signal_connect (data->options_dialog, "destroy", G_CALLBACK (gtk_widget_destroyed), &data->options_dialog);
  g_signal_connect (data->options_dialog, "response", G_CALLBACK (gtk_widget_hide), NULL);
  g_signal_connect (data->options_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
}

static void
ogmrip_main_progress_dialog_construct (OGMRipData *data, OGMRipEncode *encode)
{
  guint steps;

  if (encode->ccheck)
  {
    steps = encode->passes + encode->copy_dvd + 1;
    if (ogmrip_options_dialog_get_bitrate (OGMRIP_OPTIONS_DIALOG (data->options_dialog), NULL) == OGMRIP_OPTIONS_AUTOMATIC)
      steps += 1 +                            /* chapters */
        g_slist_length (encode->audio_streams) + /* audio */
        g_slist_length (encode->subp_streams);   /* subp  */
  }
  else
    steps = encode->passes +                   /* video */
      (encode->target_number > 1 ? 1 : 0) +    /* split */
      g_slist_length (encode->audio_streams) + /* audio */
      g_slist_length (encode->subp_streams) +  /* subp  */
      encode->copy_dvd +                       /* copy  */
      3;    /* chapters + bitrate, crop, scale  + merge */

  data->progress_dialog = ogmrip_progress_dialog_new (encode->label, encode->ccheck, steps);
  gtk_window_set_parent (GTK_WINDOW (data->progress_dialog), GTK_WINDOW (data->window));

  g_signal_connect (data->progress_dialog, "destroy", G_CALLBACK (gtk_widget_destroyed), &data->progress_dialog);
  g_signal_connect (data->progress_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
}

static GtkWidget *
ogmrip_main_update_remove_dialog_new (OGMRipData *data)
{
  GtkWidget *dialog, *button, *widget, *hbox;

  dialog = gtk_message_dialog_new (GTK_WINDOW (data->window), 
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE,
      _("Do you want to remove the copy of the DVD,\n"
        "keep it on the hard drive, or\n"
        "keep it and update the GUI ?"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_DIALOG_QUESTION);

  button = gtk_button_new ();
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, OGMRIP_AFTER_ENC_REMOVE);
  gtk_widget_show (button);

  widget = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (button), widget);
  gtk_widget_show (widget);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (widget), hbox);
  gtk_widget_show (hbox);

  widget = gtk_image_new_from_stock (GTK_STOCK_DELETE, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
  gtk_widget_show (widget);

  widget = gtk_label_new (_("_Remove"));
  gtk_label_set_use_underline (GTK_LABEL (widget), TRUE);
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
  gtk_widget_show (widget);

  button = gtk_button_new ();
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, OGMRIP_AFTER_ENC_KEEP);
  gtk_widget_show (button);

  widget = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (button), widget);
  gtk_widget_show (widget);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (widget), hbox);
  gtk_widget_show (hbox);

  widget = gtk_image_new_from_stock (GTK_STOCK_SAVE, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
  gtk_widget_show (widget);

  widget = gtk_label_new (_("_Keep"));
  gtk_label_set_use_underline (GTK_LABEL (widget), TRUE);
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
  gtk_widget_show (widget);

  button = gtk_button_new ();
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, OGMRIP_AFTER_ENC_UPDATE);
  gtk_widget_show (button);

  widget = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (button), widget);
  gtk_widget_show (widget);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (widget), hbox);
  gtk_widget_show (hbox);

  widget = gtk_image_new_from_stock (GTK_STOCK_REFRESH, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
  gtk_widget_show (widget);

  widget = gtk_label_new (_("_Update"));
  gtk_label_set_use_underline (GTK_LABEL (widget), TRUE);
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
  gtk_widget_show (widget);

  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

  return dialog;
}

/*
 * When the extract button is activated
 */
static void
ogmrip_main_extract_activated (OGMRipData *data)
{
  OGMRipEncode encode;
  GError *error = NULL;
  gboolean result = TRUE;
  gint value;

  memset (&encode, 0, sizeof (OGMRipEncode));

  encode.title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
  ogmdvd_title_ref (encode.title);

  ogmrip_options_dialog_set_title (OGMRIP_OPTIONS_DIALOG (data->options_dialog), encode.title);

  value = gtk_dialog_run (GTK_DIALOG (data->options_dialog));
/*
  if (value != OGMRIP_OPTIONS_CCHECK)
    gtk_widget_hide (data->options_dialog);
*/
  if (value == OGMRIP_OPTIONS_EXTRACT || value == OGMRIP_OPTIONS_CCHECK)
  {
    encode.ccheck = value == OGMRIP_OPTIONS_CCHECK;

    encode.label = gtk_entry_get_text (GTK_ENTRY (data->title_entry));

    encode.container = ogmrip_gconf_get_container_type (NULL);
    encode.video_codec = ogmrip_gconf_get_video_codec_type (NULL);

    if (!error)
    {
      ogmrip_chooser_list_foreach (OGMRIP_CHOOSER_LIST (data->audio_list), OGMRIP_SOURCE_FILE,
          (GFunc) ogmrip_main_source_chooser_foreach_files, &encode.audio_files);
      ogmrip_chooser_list_foreach (OGMRIP_CHOOSER_LIST (data->audio_list), OGMRIP_SOURCE_STREAM,
          (GFunc) ogmrip_main_audio_chooser_foreach_streams, &encode.audio_streams);

      ogmrip_chooser_list_foreach (OGMRIP_CHOOSER_LIST (data->subp_list), OGMRIP_SOURCE_FILE,
          (GFunc) ogmrip_main_source_chooser_foreach_files, &encode.subp_files);
      ogmrip_chooser_list_foreach (OGMRIP_CHOOSER_LIST (data->subp_list), OGMRIP_SOURCE_STREAM,
          (GFunc) ogmrip_main_subp_chooser_foreach_streams, &encode.subp_streams);
    }

    if (!error &&
        (result = ogmrip_main_check_container (data, &encode, &error)) &&
        (result = ogmrip_main_check_streams (data, &encode, &error)))
    {
      encode.log_output = ogmrip_preferences_get_bool (OGMRIP_GCONF_LOG_OUTPUT, OGMRIP_DEFAULT_LOG_OUTPUT);

      ogmrip_main_build_filenames (data, &encode);
      if (ogmrip_main_check_filename (data, &encode))
      {
        encode.ensure_sync = ogmrip_preferences_get_bool (OGMRIP_GCONF_ENSURE_SYNC, OGMRIP_DEFAULT_ENSURE_SYNC);
        encode.target_number = ogmrip_preferences_get_int (OGMRIP_GCONF_TNUMBER, OGMRIP_DEFAULT_TNUMBER);
        encode.target_size = ogmrip_preferences_get_int (OGMRIP_GCONF_TSIZE, OGMRIP_DEFAULT_TSIZE);

        encode.pullup = ogmrip_options_dialog_get_pullup (OGMRIP_OPTIONS_DIALOG (data->options_dialog));
        encode.progressive = ogmrip_options_dialog_get_progressive (OGMRIP_OPTIONS_DIALOG (data->options_dialog));

        ogmrip_chapter_list_get_selected (OGMRIP_CHAPTER_LIST (data->chapter_list), &encode.start_chap, &encode.end_chap);

        encode.rip_length = ogmrip_main_get_rip_length (data, &encode);
        encode.rip_size = ogmrip_main_get_rip_size (data, &encode);

        if (encode.ccheck)
        {
          gdouble percent = 0.05;
          gchar *filename;
          gint fd, i;

          fd = ogmrip_fs_open_tmp ("edl.XXXXXX", &filename, &error);
          if (fd < 0)
          {
          }
          close (fd);

          encode.edl = ogmrip_edl_new (filename);
          for (i = 0; i < encode.rip_length; i += 1 / percent)
            ogmrip_edl_add (encode.edl, OGMRIP_EDL_ACTION_SKIP, i + 1, i + 1 / percent);

          g_free (filename);
        }

        encode.copy_dvd = ogmrip_main_get_copy_dvd (data, &error);
        if (!error)
        {
          if ((result = ogmrip_main_check_space (data, &encode, &error)))
          {
            encode.passes = MAX (1, ogmrip_preferences_get_int (OGMRIP_GCONF_VIDEO_PASSES, OGMRIP_DEFAULT_VIDEO_PASSES));

            ogmrip_main_progress_dialog_construct (data, &encode);
            gtk_widget_show (data->progress_dialog);
            
            if (encode.log_output)
              ogmjob_log_open (encode.logfile, NULL);

            if (encode.copy_dvd)
            {
              gint backup;

              backup = ogmrip_main_backup (data, &encode, &error);
              if (backup == OGMJOB_RESULT_COMPLETED)
                result = encode.ccheck ? ogmrip_main_ccheck (data, &encode, &error) :
                  ogmrip_main_extract (data, &encode, &error);
              else
                result = backup != OGMJOB_RESULT_ERROR;
            }
            else
              result = encode.ccheck ? ogmrip_main_ccheck (data, &encode, &error) :
                ogmrip_main_extract (data, &encode, &error);

            if (encode.log_output)
              ogmjob_log_close (NULL);
          }
        }
      }

      g_free (encode.outfile);
      g_free (encode.logfile);
    }

    g_slist_foreach (encode.audio_streams, (GFunc) g_free, NULL);
    g_slist_free (encode.audio_streams);

    g_slist_foreach (encode.subp_streams, (GFunc) g_free, NULL);
    g_slist_free (encode.subp_streams);

    g_slist_free (encode.audio_files);
    g_slist_free (encode.subp_files);
  }

  if (data->progress_dialog)
    gtk_widget_destroy (data->progress_dialog);
  data->progress_dialog = NULL;

  if (encode.copy_dvd)
  {
    const gchar *path;
    gint after_enc = OGMRIP_AFTER_ENC_REMOVE;

    if (result)
    {
      after_enc = ogmrip_preferences_get_int (OGMRIP_GCONF_AFTER_ENC, OGMRIP_DEFAULT_AFTER_ENC);
      if (after_enc == OGMRIP_AFTER_ENC_ASK)
      {
        GtkWidget *dialog;

        dialog = ogmrip_main_update_remove_dialog_new (data);
        after_enc = gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
      }
    }

    path = ogmdvd_disc_get_device (ogmdvd_title_get_disc (encode.title));

    if (after_enc == OGMRIP_AFTER_ENC_UPDATE)
      ogmrip_main_load (data, path);
    else if (after_enc == OGMRIP_AFTER_ENC_REMOVE)
      ogmrip_fs_rmdir (path, TRUE, NULL);
  }

  if (!result || error)
  {
    if (!error)
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, _("Unknown error"));
    else
    {
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, error->message);
      g_error_free (error);
    }
  }

  if (encode.edl)
  {
    /*
    const gchar *filename;

    filename = ogmrip_edl_get_filename (encode.edl);
    if (filename && g_file_test (filename, G_FILE_TEST_EXISTS))
      g_unlink (filename);
    */
    ogmrip_edl_unref (encode.edl);
  }

  if (encode.title)
    ogmdvd_title_unref (encode.title);

  gtk_window_set_title (GTK_WINDOW (data->window), "OGMRip");
}

/*
 * When the eject button is activated
 */
static void
ogmrip_main_eject_activated (OGMRipData *data, GtkWidget *dialog)
{
  if (data->disc)
  {
    NautilusBurnDrive *drive;
    const gchar *device;

    drive = ogmdvd_drive_chooser_get_drive (OGMDVD_DRIVE_CHOOSER (dialog));
    if (drive)
    {
      device = ogmdvd_disc_get_device (data->disc);
      if (strcmp (device, drive->device) == 0)
      {
        ogmdvd_disc_unref (data->disc);
        data->disc = NULL;

        ogmrip_main_clear (data);
      }
      g_object_unref (drive);
    }
  }
}

/*
 * When the load button is activated
 */
static void
ogmrip_main_load_activated (OGMRipData *data)
{
  if (!data->encoding)
  {
    GtkWidget *dialog;

    dialog = ogmdvd_drive_chooser_dialog_new ();
    gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_REFRESH);
    gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

    g_signal_connect_swapped (dialog, "eject", G_CALLBACK (ogmrip_main_eject_activated), data);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      NautilusBurnDrive *drive;

      drive = ogmdvd_drive_chooser_get_drive (OGMDVD_DRIVE_CHOOSER (dialog));
      if (drive)
      {
        ogmrip_main_load (data, drive->device);
        g_object_unref (drive);
      }
    }
    gtk_widget_destroy (dialog);
  }
}

/*
 * When the open button is activated
 */
static void
ogmrip_main_open_activated (OGMRipData *data)
{
  if (!data->encoding)
  {
    GtkWidget *dialog;

    dialog = gtk_file_chooser_dialog_new (_("Select a DVD Structure"), 
        GTK_WINDOW (data->window), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER, 
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);
    gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_OPEN);
    gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      gchar *dir;

      dir = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      if (dir)
        ogmrip_main_load (data, dir);
      g_free (dir);
    }
    gtk_widget_destroy (dialog);
  }
}

/*
 * When the import chapters menu item is activated
 */
static void
ogmrip_main_import_chapters_activated (OGMRipData *data)
{
  if (!data->encoding)
  {
    GtkWidget *dialog;
    GtkFileFilter *filter;

    dialog = gtk_file_chooser_dialog_new (_("Select a chapters file"), 
        GTK_WINDOW (data->window), GTK_FILE_CHOOSER_ACTION_OPEN, 
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);
    gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_OPEN);
    gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

    filter = gtk_file_filter_new ();
    gtk_file_filter_add_mime_type (filter, "text/*");
    gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog), filter);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      gchar *filename;

      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      if (filename)
        if (!ogmrip_main_import_matroska_chapters (data, filename))
          if (!ogmrip_main_import_simple_chapters (data, filename))
            ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR,
                _("Could not open the chapters file '%s'"), filename);
      g_free (filename);
    }
    gtk_widget_destroy (dialog);
  }
}

/*
 * When the export chapters menu item is activated
 */
static void
ogmrip_main_export_chapters_activated (OGMRipData *data)
{
  GtkWidget *dialog;

  dialog = gtk_file_chooser_dialog_new (_("Select a file"), 
      GTK_WINDOW (data->window), GTK_FILE_CHOOSER_ACTION_SAVE, 
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE, GTK_RESPONSE_OK, NULL);
  gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_SAVE);
  gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
  {
    gchar *filename;

    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    if (filename)
      ogmrip_main_export_simple_chapters (data, filename);
    g_free (filename);
  }
  gtk_widget_destroy (dialog);
}

/*
 * When the select all menu item is activated
 */
static void
ogmrip_main_select_all_activated (OGMRipData *data)
{
  OGMDvdTitle *title;
  OGMDvdTime time_;

  ogmrip_chapter_list_select_all (OGMRIP_CHAPTER_LIST (data->chapter_list));

  title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
  if (ogmdvd_title_get_length (title, &time_) > 0)
  {
    gchar *str;

    str = g_strdup_printf ("%02d:%02d:%02d", time_.hour, time_.min, time_.sec);
    gtk_label_set_text (GTK_LABEL (data->length_label), str);
    g_free (str);
  }
  ogmdvd_title_unref (title);

  gtk_widget_set_sensitive (data->extract_button, TRUE);
  gtk_widget_set_sensitive (data->relative_check, FALSE);
  gtk_action_set_sensitive (data->extract_action, TRUE);
}

/*
 * When the deselect all menu item is activated
 */
static void
ogmrip_main_deselect_all_activated (OGMRipData *data)
{
  ogmrip_chapter_list_deselect_all (OGMRIP_CHAPTER_LIST (data->chapter_list));

  gtk_label_set_text (GTK_LABEL (data->length_label), "");
  gtk_widget_set_sensitive (data->extract_button, FALSE);
  gtk_widget_set_sensitive (data->relative_check, FALSE);
  gtk_action_set_sensitive (data->extract_action, FALSE);
}

/*
 * When the preferences menu item is activated
 */
static void
ogmrip_main_pref_activated (OGMRipData *data)
{
  gtk_widget_show (data->pref_dialog);
}

/*
 * When the about menu item is activated
 */
static void
ogmrip_main_about_activated (OGMRipData *data)
{
  static GdkPixbuf *icon = NULL;

  const gchar *authors[] = 
  { 
    "Olivier Rolland <billl@users.sf.net>", 
    NULL 
  };
  gchar *translator_credits = _("translator-credits");

  const gchar *documenters[] = 
  { 
    "Olivier Brisson <olivier.brisson@gmail.com>",
    NULL 
  };

  if (!icon)
    icon = gdk_pixbuf_new_from_file (OGMRIP_DATA_DIR "/" OGMRIP_ICON_FILE, NULL);

  if (strcmp (translator_credits, "translator-credits") == 0)
    translator_credits = NULL;

  gtk_show_about_dialog (GTK_WINDOW (data->window), 
      "name", PACKAGE_NAME,
      "version", PACKAGE_VERSION,
      "comments", _("A DVD Encoder for GNOME"),
      "copyright", "(c) 2004-2007 Olivier Rolland",
      "website", "http://ogmrip.sourceforge.net",
      "translator-credits", translator_credits,
      "documenters", documenters,
      "authors", authors,
      "logo", icon,
      NULL);
}

static void
ogmrip_main_title_chooser_changed (OGMRipData *data)
{
  GtkWidget *audio_chooser;
  GtkWidget *subp_chooser;

  OGMDvdTitle *title;
  OGMDvdTime time_;
  gchar *str;

  ogmrip_chooser_list_clear (OGMRIP_CHOOSER_LIST (data->audio_list));

  audio_chooser = ogmrip_audio_chooser_widget_new ();
  gtk_container_add (GTK_CONTAINER (data->audio_list), audio_chooser);
  gtk_widget_show (audio_chooser);

  gtk_widget_set_sensitive (data->audio_list, data->disc != NULL);

  ogmrip_chooser_list_clear (OGMRIP_CHOOSER_LIST (data->subp_list));
  subp_chooser = ogmrip_subtitle_chooser_widget_new ();
  gtk_container_add (GTK_CONTAINER (data->subp_list), subp_chooser);
  gtk_widget_show (subp_chooser);

  gtk_widget_set_sensitive (data->subp_list, data->disc != NULL);

  ogmdvd_chapter_list_clear (OGMDVD_CHAPTER_LIST (data->chapter_list));

  gtk_label_set_text (GTK_LABEL (data->length_label), "");

  if (data->disc)
  {
    gint /*def,*/ pref;
    GType container, codec;

    title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
    if (title)
    {
      if (ogmdvd_title_get_length (title, &time_) > 0)
      {
        str = g_strdup_printf ("%02d:%02d:%02d", time_.hour, time_.min, time_.sec);
        gtk_label_set_text (GTK_LABEL (data->length_label), str);
        g_free (str);
      }

      container = ogmrip_gconf_get_container_type (NULL);

      pref = ogmrip_preferences_get_int (OGMRIP_GCONF_PREF_AUDIO, OGMRIP_DEFAULT_PREF_AUDIO);
      ogmrip_source_chooser_set_title (OGMRIP_SOURCE_CHOOSER (audio_chooser), title);
      ogmrip_source_chooser_select_language (OGMRIP_SOURCE_CHOOSER (audio_chooser), pref);

      pref = ogmrip_preferences_get_int (OGMRIP_GCONF_PREF_SUBP, OGMRIP_DEFAULT_PREF_SUBP);
      ogmrip_source_chooser_set_title (OGMRIP_SOURCE_CHOOSER (subp_chooser), title);
      ogmrip_source_chooser_select_language (OGMRIP_SOURCE_CHOOSER (subp_chooser), pref);

      ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->audio_list), 
          container != G_TYPE_NONE ? ogmrip_plugin_get_container_max_audio (container) : 1);

      ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->subp_list), 
          container != G_TYPE_NONE ? ogmrip_plugin_get_container_max_subp (container) : 1);

      codec = ogmrip_gconf_get_audio_codec_type (NULL);
      gtk_widget_set_sensitive (data->audio_list, container != G_TYPE_NONE &&
          codec != G_TYPE_NONE && ogmrip_plugin_can_contain_audio (container, codec));

      codec = ogmrip_gconf_get_subp_codec_type (NULL);
      gtk_widget_set_sensitive (data->subp_list, container != G_TYPE_NONE &&
          codec != G_TYPE_NONE && ogmrip_plugin_can_contain_subp (container, codec));

      ogmdvd_chapter_list_set_title (OGMDVD_CHAPTER_LIST (data->chapter_list), title);
      ogmrip_chapter_list_select_all (OGMRIP_CHAPTER_LIST (data->chapter_list));
    }

    gtk_widget_set_sensitive (data->relative_check, FALSE);
  }
}

static void
ogmrip_main_audio_chooser_added (OGMRipData *data, OGMRipSourceChooser *chooser)
{
  OGMDvdTitle *title;

  title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
  if (title)
    ogmrip_source_chooser_set_title (OGMRIP_SOURCE_CHOOSER (chooser), title);
}

static void
ogmrip_main_audio_chooser_more_clicked (OGMRipData *data, OGMRipSourceChooser *chooser, OGMRipChooserList *list)
{
  GtkWidget *dialog;
  OGMRipAudioSource *source;
  gboolean use_defaults = FALSE;

  source = g_object_get_data (G_OBJECT (chooser), "__audio_source__");

  if (!source)
  {
    source = g_new0 (OGMRipAudioSource, 1);
    ogmrip_main_audio_source_get_defaults (source);

    use_defaults = TRUE;
  }

  dialog = ogmrip_audio_options_dialog_new ();
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

  ogmrip_audio_options_dialog_set_use_defaults (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog), use_defaults);
  ogmrip_audio_options_dialog_set_codec (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog), source->codec); 
  ogmrip_audio_options_dialog_set_quality (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog), source->quality);
  ogmrip_audio_options_dialog_set_srate (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog), source->srate);
  ogmrip_audio_options_dialog_set_channels (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog), source->channels);
  ogmrip_audio_options_dialog_set_normalize (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog), source->normalize);

  gtk_dialog_run (GTK_DIALOG (dialog));

  if (ogmrip_audio_options_dialog_get_use_defaults (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog)))
  {
    if (use_defaults)
      g_free (source);
    g_object_set_data (G_OBJECT (chooser), "__audio_source__", NULL);
  }
  else
  {
    source->codec = ogmrip_audio_options_dialog_get_codec (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog));
    source->quality = ogmrip_audio_options_dialog_get_quality (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog));
    source->srate = ogmrip_audio_options_dialog_get_srate (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog));
    source->channels = ogmrip_audio_options_dialog_get_channels (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog));
    source->normalize = ogmrip_audio_options_dialog_get_normalize (OGMRIP_AUDIO_OPTIONS_DIALOG (dialog));

    if (use_defaults)
      g_object_set_data_full (G_OBJECT (chooser), "__audio_source__", source, (GDestroyNotify) g_free);
  }

  gtk_widget_destroy (dialog);
}

static void
ogmrip_main_subp_chooser_added (OGMRipData *data, OGMRipSourceChooser *chooser)
{
  OGMDvdTitle *title;

  title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
  if (title)
    ogmrip_source_chooser_set_title (OGMRIP_SOURCE_CHOOSER (chooser), title);
}

static void
ogmrip_main_subp_chooser_more_clicked (OGMRipData *data, OGMRipSourceChooser *chooser, OGMRipChooserList *list)
{
  GtkWidget *dialog;
  OGMRipSubpSource *source;
  gboolean use_defaults = FALSE;

  source = g_object_get_data (G_OBJECT (chooser), "__subp_source__");

  if (!source)
  {
    source = g_new0 (OGMRipSubpSource, 1);
    ogmrip_main_subp_source_get_defaults (source);

    use_defaults = TRUE;
  }

  dialog = ogmrip_subp_options_dialog_new ();
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

  ogmrip_subp_options_dialog_set_use_defaults (OGMRIP_SUBP_OPTIONS_DIALOG (dialog), use_defaults);
  ogmrip_subp_options_dialog_set_codec (OGMRIP_SUBP_OPTIONS_DIALOG (dialog), source->codec); 
  ogmrip_subp_options_dialog_set_charset (OGMRIP_SUBP_OPTIONS_DIALOG (dialog), source->charset);
  ogmrip_subp_options_dialog_set_eol (OGMRIP_SUBP_OPTIONS_DIALOG (dialog), source->eol);
  ogmrip_subp_options_dialog_set_spell (OGMRIP_SUBP_OPTIONS_DIALOG (dialog), source->spell);
  ogmrip_subp_options_dialog_set_forced_only (OGMRIP_SUBP_OPTIONS_DIALOG (dialog), source->forced_subs);

  gtk_dialog_run (GTK_DIALOG (dialog));

  if (ogmrip_subp_options_dialog_get_use_defaults (OGMRIP_SUBP_OPTIONS_DIALOG (dialog)))
  {
    if (use_defaults)
      g_free (source);
    g_object_set_data (G_OBJECT (chooser), "__subp_source__", NULL);
  }
  else
  {
    source->codec = ogmrip_subp_options_dialog_get_codec (OGMRIP_SUBP_OPTIONS_DIALOG (dialog));
    source->charset = ogmrip_subp_options_dialog_get_charset (OGMRIP_SUBP_OPTIONS_DIALOG (dialog));
    source->eol = ogmrip_subp_options_dialog_get_eol (OGMRIP_SUBP_OPTIONS_DIALOG (dialog));
    source->spell = ogmrip_subp_options_dialog_get_spell (OGMRIP_SUBP_OPTIONS_DIALOG (dialog));
    source->forced_subs = ogmrip_subp_options_dialog_get_forced_only (OGMRIP_SUBP_OPTIONS_DIALOG (dialog));

    if (use_defaults)
      g_object_set_data_full (G_OBJECT (chooser), "__subp_source__", source, (GDestroyNotify) g_free);
  }

  gtk_widget_destroy (dialog);
}

static void
ogmrip_main_chapter_selection_changed (OGMRipData *data)
{
  OGMDvdTitle *title;
  OGMDvdTime time_;

  guint start_chap;
  gint end_chap;

  gboolean sensitive;
  gchar *str;

  sensitive = ogmrip_chapter_list_get_selected (OGMRIP_CHAPTER_LIST (data->chapter_list), &start_chap, &end_chap);
  if (sensitive)
  {
    title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
    if (ogmdvd_title_get_chapters_length (title, start_chap, end_chap, &time_) > 0)
    {
      str = g_strdup_printf ("%02d:%02d:%02d", time_.hour, time_.min, time_.sec);
      gtk_label_set_text (GTK_LABEL (data->length_label), str);
      g_free (str);
    }
  }

  gtk_widget_set_sensitive (data->extract_button, sensitive);
  gtk_action_set_sensitive (data->extract_action, sensitive);

  gtk_widget_set_sensitive (data->relative_check, sensitive && (start_chap > 0 || end_chap != -1));
}

static void
ogmrip_main_container_notified (GConfValue *value, OGMRipData *data)
{
  if (data->disc)
  {
    GType container, codec;
    const gchar *name;

    name = gconf_value_get_string (value);
    container = codec = ogmrip_gconf_get_container_type (name);

    ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->audio_list),
        container != G_TYPE_NONE ? ogmrip_plugin_get_container_max_audio (container) : 1);

    ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->subp_list), 
        container != G_TYPE_NONE ? ogmrip_plugin_get_container_max_subp (container) : 1);

    codec = ogmrip_gconf_get_audio_codec_type (NULL);
    gtk_widget_set_sensitive (data->audio_list, container != G_TYPE_NONE &&
        codec != G_TYPE_NONE && ogmrip_plugin_can_contain_audio (container, codec));

    codec = ogmrip_gconf_get_subp_codec_type (NULL);
    gtk_widget_set_sensitive (data->subp_list, container != G_TYPE_NONE &&
        codec != G_TYPE_NONE && ogmrip_plugin_can_contain_subp (container, codec));
  }
}

static void
ogmrip_main_item_selected (GtkWidget *item, GtkWidget *statusbar)
{
  gchar *hint;

  hint = g_object_get_data (G_OBJECT (item), "__menu_hint__");
  if (hint)
  {
    guint context_id;

    context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (statusbar), "__menu_hint__");
    gtk_statusbar_push (GTK_STATUSBAR (statusbar), context_id, hint);
  }
}

static void
ogmrip_main_item_deselected (GtkWidget *item, GtkWidget *statusbar)
{
  guint context_id;

  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (statusbar), "__menu_hint__");
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), context_id);
}

static void
ogmrip_main_connect_proxy (GtkUIManager *uimanager, GtkAction *action, GtkWidget *proxy, GtkWidget *statusbar)
{
  if (GTK_IS_MENU_ITEM (proxy))
  {
    gchar *hint;

    g_object_get (action, "tooltip", &hint, NULL);
    if (hint)
    {
      g_object_set_data_full (G_OBJECT (proxy), "__menu_hint__", hint, (GDestroyNotify) g_free);

      g_signal_connect (proxy, "select", G_CALLBACK (ogmrip_main_item_selected), statusbar);
      g_signal_connect (proxy, "deselect", G_CALLBACK (ogmrip_main_item_deselected), statusbar);
    }
  }
}

static OGMRipData *
ogmrip_main_new (void)
{
  GtkActionEntry action_entries[] =
  {
    { "FileMenu",     NULL,                  N_("_File"),               NULL,          NULL,                             NULL },
    { "Load",         GTK_STOCK_CDROM,       N_("_Load"),               "<ctl>L",      N_("Load a DVD disk"),            NULL },
    { "Open",         GTK_STOCK_OPEN,        NULL,                      NULL,          N_("Open a DVD structure"),       NULL },
    { "Extract",      GTK_STOCK_CONVERT,     N_("E_xtract"),            "<ctl>Return", N_("Extract selected streams"),   NULL },
    { "OpenChapters", NULL,                  N_("_Import Chapters..."), NULL,          N_("Import chapter information"), NULL },
    { "SaveChapters", NULL,                  N_("_Export Chapters..."), NULL,          N_("Export chapter information"), NULL },
    { "Quit",         GTK_STOCK_QUIT,        NULL,                      NULL,          N_("Exit OGMRip"),                NULL },
    { "EditMenu",     NULL,                  N_("_Edit"),               NULL,          NULL,                             NULL },
    { "SelectAll",    NULL,                  N_("Select _All"),         "<ctl>A",      N_("Select all chapters"),        NULL },
    { "DeselectAll",  NULL,                  N_("_Deselect All"),       "<ctl>D",      N_("Deselect all chapters"),      NULL },
    { "Preferences",  GTK_STOCK_PREFERENCES, NULL,                      NULL,          N_("Edit the preferences"),       NULL },
    { "HelpMenu",     NULL,                  N_("_Help"),               NULL,          NULL,                             NULL },
    { "About",        GTK_STOCK_ABOUT,       N_("_About"),              NULL,          N_("About OGMRip"),               NULL },
  };

  OGMRipData *data;
  GtkWidget *widget;
  GladeXML *xml;

  GtkAction *action;
  GtkActionGroup *action_group;
  GtkAccelGroup *accel_group;
  GtkUIManager *ui_manager;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, NULL, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return NULL;
  }

  data = g_new0 (OGMRipData, 1);

  ogmrip_preferences_add_notify (OGMRIP_GCONF_CONTAINER,
      (GFunc) ogmrip_main_container_notified, data);

  data->window = glade_xml_get_widget (xml, "main-window");
  gtk_window_set_default_size (GTK_WINDOW (data->window), 350, 500);
  gtk_window_set_icon_from_file (GTK_WINDOW (data->window), OGMRIP_DATA_DIR "/" OGMRIP_ICON_FILE, NULL);

  g_signal_connect_swapped (data->window, "delete-event", G_CALLBACK (ogmrip_main_delete_event), data);
  g_signal_connect_swapped (data->window, "destroy", G_CALLBACK (ogmrip_main_destroyed), data);
  g_signal_connect (data->window, "destroy", G_CALLBACK (gtk_main_quit), NULL);

  action_group = gtk_action_group_new ("MenuActions");
  gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
  gtk_action_group_add_actions (action_group, action_entries, G_N_ELEMENTS (action_entries), NULL);

  ui_manager = gtk_ui_manager_new ();

  widget = glade_xml_get_widget (xml, "statusbar");
  g_signal_connect (ui_manager, "connect-proxy", G_CALLBACK (ogmrip_main_connect_proxy), widget);

  gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);
  gtk_ui_manager_add_ui_from_file (ui_manager, OGMRIP_DATA_DIR "/" OGMRIP_UI_FILE, NULL);

  accel_group = gtk_ui_manager_get_accel_group (ui_manager);
  gtk_window_add_accel_group (GTK_WINDOW (data->window), accel_group);

  widget = gtk_ui_manager_get_widget (ui_manager, "/Menubar");
  gtk_box_pack_start (GTK_BOX (GTK_BIN (data->window)->child), widget, FALSE, FALSE, 0);

  action =  gtk_action_group_get_action (action_group, "Load");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_load_activated), data);

  action =  gtk_action_group_get_action (action_group, "Open");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_open_activated), data);

  data->import_chap_action =  gtk_action_group_get_action (action_group, "OpenChapters");
  g_signal_connect_swapped (data->import_chap_action, "activate", G_CALLBACK (ogmrip_main_import_chapters_activated), data);
  gtk_action_set_sensitive (data->import_chap_action, FALSE);

  data->export_chap_action =  gtk_action_group_get_action (action_group, "SaveChapters");
  g_signal_connect_swapped (data->export_chap_action, "activate", G_CALLBACK (ogmrip_main_export_chapters_activated), data);
  gtk_action_set_sensitive (data->export_chap_action, FALSE);

  action =  gtk_action_group_get_action (action_group, "Quit");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (gtk_widget_destroy), data->window);

  action =  gtk_action_group_get_action (action_group, "Preferences");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_pref_activated), data);

  action =  gtk_action_group_get_action (action_group, "SelectAll");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_select_all_activated), data);

  action =  gtk_action_group_get_action (action_group, "DeselectAll");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_deselect_all_activated), data);

  action =  gtk_action_group_get_action (action_group, "About");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_about_activated), data);

  data->extract_action =  gtk_action_group_get_action (action_group, "Extract");
  g_signal_connect_swapped (data->extract_action, "activate", G_CALLBACK (ogmrip_main_extract_activated), data);
  gtk_action_set_sensitive (data->extract_action, FALSE);

  widget = glade_xml_get_widget (xml, "load-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_main_load_activated), data);

  widget = glade_xml_get_widget (xml, "open-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_main_open_activated), data);

  data->extract_button = glade_xml_get_widget (xml, "extract-button");
  g_signal_connect_swapped (data->extract_button, "clicked", G_CALLBACK (ogmrip_main_extract_activated), data);

  widget = data->title_chooser = glade_xml_get_widget (xml, "title-chooser");
  gtk_widget_set_sensitive (widget, FALSE);
  gtk_widget_show (widget);

  g_signal_connect_swapped (widget, "changed", G_CALLBACK (ogmrip_main_title_chooser_changed), data);

  widget = glade_xml_get_widget (xml, "table");

  data->audio_list = ogmrip_chooser_list_new (OGMRIP_TYPE_AUDIO_CHOOSER_WIDGET);
  gtk_table_attach (GTK_TABLE (widget), data->audio_list, 1, 3, 2, 3, GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_widget_set_sensitive (data->audio_list, FALSE);
  gtk_widget_show (data->audio_list);

  g_signal_connect_swapped (data->audio_list, "add",
      G_CALLBACK (ogmrip_main_audio_chooser_added), data);
  g_signal_connect_swapped (data->audio_list, "more-clicked",
      G_CALLBACK (ogmrip_main_audio_chooser_more_clicked), data);

  data->subp_list = ogmrip_chooser_list_new (OGMRIP_TYPE_SUBTITLE_CHOOSER_WIDGET);
  gtk_table_attach (GTK_TABLE (widget), data->subp_list, 1, 3, 3, 4, GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_widget_set_sensitive (data->subp_list, FALSE);
  gtk_widget_show (data->subp_list);

  g_signal_connect_swapped (data->subp_list, "add",
      G_CALLBACK (ogmrip_main_subp_chooser_added), data);
  g_signal_connect_swapped (data->subp_list, "more-clicked",
      G_CALLBACK (ogmrip_main_subp_chooser_more_clicked), data);

  widget = ogmrip_audio_chooser_widget_new ();
  gtk_container_add (GTK_CONTAINER (data->audio_list), widget);
  gtk_widget_show (widget);

  widget = ogmrip_subtitle_chooser_widget_new ();
  gtk_container_add (GTK_CONTAINER (data->subp_list), widget);
  gtk_widget_show (widget);

  data->length_label = glade_xml_get_widget (xml, "length-label");
  data->relative_check = glade_xml_get_widget (xml, "relative-check");
  data->title_entry = glade_xml_get_widget (xml, "title-entry");

  data->chapter_list = glade_xml_get_widget (xml, "chapter-list");
  g_signal_connect_swapped (data->chapter_list, "selection-changed", 
      G_CALLBACK (ogmrip_main_chapter_selection_changed), data);
  gtk_widget_show (data->chapter_list);

  g_object_unref (xml);

  return data;
}

#ifdef G_ENABLE_DEBUG
static gboolean debug = TRUE;
#else
static gboolean debug = FALSE;
#endif

int
main (int argc, char *argv[])
{
  OGMRipData *data;

  GOptionEntry opts[] =
  {
    { "debug", 0,  0, G_OPTION_ARG_NONE, &debug, "Enable debug messages", NULL },
    { NULL,    0,  0, 0,                 NULL,  NULL,                     NULL }
  };

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif /* ENABLE_NLS */

  if (!gtk_init_with_args (&argc, &argv, "<DVD DEVICE>", opts, GETTEXT_PACKAGE, NULL))
    return EXIT_FAILURE;

  if (debug)
    ogmjob_log_set_print_stdout (TRUE);

  ogmrip_plugin_init ();

  ogmrip_preferences_init (OGMRIP_GCONF_ROOT);

#ifdef HAVE_LIBNOTIFY_SUPPORT
  notify_init (PACKAGE_NAME);
#endif /* HAVE_LIBNOTIFY_SUPPORT */

  data = ogmrip_main_new ();

  if (argc > 1)
    if (g_file_test (argv[1], G_FILE_TEST_EXISTS))
      ogmrip_main_load (data, argv[1]);

  ogmrip_main_pref_dialog_construct (data);
  ogmrip_main_options_dialog_construct (data);

  gtk_widget_show (data->window);

  gtk_main ();

  ogmrip_preferences_uninit ();

  ogmrip_plugin_uninit ();

  return EXIT_SUCCESS;
}

