/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "ogmrip-audio-options.h"
#include "ogmrip-preferences.h"
#include "ogmrip-helper.h"
#include "ogmrip-plugin.h"
#include "ogmrip-gconf.h"

#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-pref.glade"
#define OGMRIP_GLADE_ROOT "audio-page"

#define OGMRIP_AUDIO_OPTIONS_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_AUDIO_OPTIONS_DIALOG, OGMRipAudioOptionsDialogPriv))

struct _OGMRipAudioOptionsDialogPriv
{
  GType container;
  GtkWidget *default_button;
  GtkWidget *root_table;
  GtkWidget *codec_table;
  GtkWidget *quality_spin;
  GtkWidget *srate_combo;
  GtkWidget *channels_combo;
  GtkWidget *normalize_check;
};

static void
ogmrip_audio_options_dialog_reset (OGMRipAudioOptionsDialog *dialog)
{
  GtkTableChild *child;
  gint active;

  active = g_list_length (GTK_TABLE (dialog->priv->codec_table)->children) - 1 -
    ogmrip_gconf_get_audio_codec_index (NULL);

  child = g_list_nth_data (GTK_TABLE (dialog->priv->codec_table)->children, active);
  if (child)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (child->widget), TRUE);

  active = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_QUALITY, OGMRIP_DEFAULT_AUDIO_QUALITY);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->quality_spin), active);

  active = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_SRATE, OGMRIP_DEFAULT_AUDIO_SRATE);
  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->srate_combo), active);

  active = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_CHANNELS, OGMRIP_DEFAULT_AUDIO_CHANNELS);
  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->channels_combo), active);

  active = ogmrip_preferences_get_bool (OGMRIP_GCONF_AUDIO_NORMALIZE, OGMRIP_DEFAULT_AUDIO_NORMALIZE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->normalize_check), active);
}

static void
ogmrip_audio_options_dialog_default_toggled (OGMRipAudioOptionsDialog *dialog)
{
  gboolean active;

  active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button));
  gtk_widget_set_sensitive (dialog->priv->root_table, !active);
}

static GtkWidget *
ogmrip_audio_options_append_codec (GtkTable *table, const gchar *description)
{
  static GSList *group = NULL;
  static guint row = 0, col = 0;
  GtkWidget *radio;

  if (!table->children)
  {
    group = NULL;
    col = row = 0;
  }

  radio = gtk_radio_button_new_with_label (group, description);
  gtk_table_attach (table, radio, col, col + 1, row, row + 1, GTK_FILL, 0, 0, 0);
  gtk_widget_show (radio);

  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));

  col = (col + 1) % 2;
  if (!col)
    row ++;

  return radio;
}

static void
ogmrip_audio_options_append_audio_codec (GType type, gchar *name, gchar *description, OGMRipAudioOptionsDialog *dialog)
{
  GtkWidget *radio;
  gboolean can_contain;

  radio = ogmrip_audio_options_append_codec (GTK_TABLE (dialog->priv->codec_table), description);

  can_contain = ogmrip_plugin_can_contain_audio (dialog->priv->container, type);
  gtk_widget_set_sensitive (radio, can_contain);
}

G_DEFINE_TYPE (OGMRipAudioOptionsDialog, ogmrip_audio_options_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_audio_options_dialog_class_init (OGMRipAudioOptionsDialogClass *klass)
{
  g_type_class_add_private (klass, sizeof (OGMRipAudioOptionsDialogPriv));
}

static void
ogmrip_audio_options_dialog_init (OGMRipAudioOptionsDialog *dialog)
{
  GtkWidget *widget, *alignment, *reset;
  GladeXML *xml;
  gint n;

  dialog->priv = OGMRIP_AUDIO_OPTIONS_DIALOG_GET_PRIVATE (dialog);

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  reset = gtk_button_new_with_label (_("Reset"));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), reset);
  gtk_widget_show (reset);

  g_signal_connect_swapped (reset, "clicked", G_CALLBACK (ogmrip_audio_options_dialog_reset), dialog);

  gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);

  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Audio Track Options"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PROPERTIES);

  alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 6, 0, 6, 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), alignment);
  gtk_widget_show (alignment);

  dialog->priv->default_button = gtk_check_button_new_with_label ("Use default settings");
  gtk_container_add (GTK_CONTAINER (alignment), dialog->priv->default_button);
  gtk_widget_show (dialog->priv->default_button);

  dialog->priv->root_table = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), dialog->priv->root_table, TRUE, TRUE, 0);
  gtk_widget_show (dialog->priv->root_table);

  g_signal_connect_swapped (dialog->priv->default_button, "toggled",
      G_CALLBACK (ogmrip_audio_options_dialog_default_toggled), dialog);

  widget = glade_xml_get_widget (xml, "audio-lang-vbox");
  gtk_widget_hide (widget);

  dialog->priv->codec_table = glade_xml_get_widget (xml, "audio-table");

  n = ogmrip_plugin_get_n_audio_codecs ();
  gtk_table_resize (GTK_TABLE (dialog->priv->codec_table), (n + 1) / 2, 2);
  gtk_widget_set_sensitive (dialog->priv->codec_table, n > 0);

  dialog->priv->container = ogmrip_gconf_get_container_type (NULL);

  ogmrip_plugin_foreach_audio_codec ((OGMRipPluginFunc) ogmrip_audio_options_append_audio_codec, dialog);

  dialog->priv->quality_spin = glade_xml_get_widget (xml, "quality-spin");
  dialog->priv->srate_combo = glade_xml_get_widget (xml, "srate-combo");
  dialog->priv->channels_combo = glade_xml_get_widget (xml, "channels-combo");
  dialog->priv->normalize_check = glade_xml_get_widget (xml, "normalize-check");

  g_object_unref (xml);
}

GtkWidget *
ogmrip_audio_options_dialog_new (void)
{
  return g_object_new (OGMRIP_TYPE_AUDIO_OPTIONS_DIALOG, NULL);
}

void
ogmrip_audio_options_dialog_set_use_defaults (OGMRipAudioOptionsDialog *dialog, gboolean use_defaults)
{
  g_return_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button), use_defaults);
}

gboolean
ogmrip_audio_options_dialog_get_use_defaults (OGMRipAudioOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog), FALSE);

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button));
}

void
ogmrip_audio_options_dialog_set_codec (OGMRipAudioOptionsDialog *dialog, GType codec)
{
  GtkTableChild *child;
  gint n;

  g_return_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog));

  n = g_list_length (GTK_TABLE (dialog->priv->codec_table)->children) - 
    ogmrip_plugin_get_audio_codec_index (codec) - 1;

  child = g_list_nth_data (GTK_TABLE (dialog->priv->codec_table)->children, n);
  if (child)
  {
    if (GTK_WIDGET_SENSITIVE (child->widget))
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (child->widget), TRUE);
    else
    {
      n = g_list_length (GTK_TABLE (dialog->priv->codec_table)->children) - 1 -
        ogmrip_gconf_get_audio_codec_index (NULL);
      child = g_list_nth_data (GTK_TABLE (dialog->priv->codec_table)->children, n);
      if (child)
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (child->widget), TRUE);
    }
  }
}

GType
ogmrip_audio_options_dialog_get_codec (OGMRipAudioOptionsDialog *dialog)
{
  gint codec;

  g_return_val_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog), G_TYPE_NONE);

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button)))
    return ogmrip_gconf_get_audio_codec_type (NULL);
  else
  {
    GtkTableChild *child;
    GList *link;

    for (link = GTK_TABLE (dialog->priv->codec_table)->children, codec = 0; link; link = link->next, codec ++)
    {
      child = link->data;
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (child->widget)))
        break;
    }

    codec = g_list_length (GTK_TABLE (dialog->priv->codec_table)->children) - codec - 1;
  }

  return ogmrip_plugin_get_nth_audio_codec (codec);
}

void
ogmrip_audio_options_dialog_set_quality (OGMRipAudioOptionsDialog *dialog, guint quality)
{
  g_return_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog));

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->quality_spin), quality);
}

gint
ogmrip_audio_options_dialog_get_quality (OGMRipAudioOptionsDialog *dialog)
{
  gint quality;

  g_return_val_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog), -1);

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button)))
    quality = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_QUALITY, OGMRIP_DEFAULT_AUDIO_QUALITY);
  else
    quality = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->quality_spin));

  return quality;
}

void
ogmrip_audio_options_dialog_set_srate (OGMRipAudioOptionsDialog *dialog, guint srate)
{
  g_return_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog));

  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->srate_combo), srate);
}

gint
ogmrip_audio_options_dialog_get_srate (OGMRipAudioOptionsDialog *dialog)
{
  gint srate;

  g_return_val_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog), -1);

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button)))
    srate = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_SRATE, OGMRIP_DEFAULT_AUDIO_SRATE);
  else
    srate = gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->srate_combo));

  return srate;
}

void
ogmrip_audio_options_dialog_set_channels (OGMRipAudioOptionsDialog *dialog, guint channels)
{
  g_return_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog));

  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->channels_combo), channels);
}

gint
ogmrip_audio_options_dialog_get_channels (OGMRipAudioOptionsDialog *dialog)
{
  gint channels;

  g_return_val_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog), -1);

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button)))
    channels = ogmrip_preferences_get_int (OGMRIP_GCONF_AUDIO_CHANNELS, OGMRIP_DEFAULT_AUDIO_CHANNELS);
  else
    channels = gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->channels_combo));

  return channels;
}

void
ogmrip_audio_options_dialog_set_normalize (OGMRipAudioOptionsDialog *dialog, gboolean normalize)
{
  g_return_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->normalize_check), normalize);
}

gboolean
ogmrip_audio_options_dialog_get_normalize (OGMRipAudioOptionsDialog *dialog)
{
  gboolean normalize;

  g_return_val_if_fail (OGMRIP_IS_AUDIO_OPTIONS_DIALOG (dialog), FALSE);

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button)))
    normalize = ogmrip_preferences_get_bool (OGMRIP_GCONF_AUDIO_NORMALIZE, OGMRIP_DEFAULT_AUDIO_NORMALIZE);
  else
    normalize = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->normalize_check));

  return normalize;
}

