/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-mkv.h"
#include "ogmrip-version.h"
#include "ogmrip-plugin.h"
#include "ogmjob-exec.h"

#include <string.h>
#include <stdio.h>
#include <glib/gi18n-lib.h>

#define MKV_OVERHEAD 4

static gint ogmrip_matroska_run          (OGMJobSpawn     *spawn);
static gint ogmrip_matroska_get_overhead (OGMRipContainer *container);

static gdouble
ogmrip_matroska_watch (OGMJobExec *exec, const gchar *buffer, OGMRipContainer *matroska)
{
  gulong frames, total;
  guint percent;

  if (sscanf (buffer, "progress: %lu/%lu frames (%u%%)", &frames, &total, &percent) == 3)
    return percent / 100.0;
  else if (sscanf (buffer, "progress: %u%%", &percent) == 1)
    return percent / 100.0;

  return -1.0;
}

static gchar *
ogmrip_matroska_get_sync (OGMRipContainer *container)
{
  guint start_delay;

  start_delay = ogmrip_container_get_start_delay (container);
  if (start_delay > 0)
  {
    OGMDvdTitle *title;
    OGMRipVideo *video;
    guint num, denom;
    gchar *buf;

    video = ogmrip_container_get_video (container);

    title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
    ogmdvd_title_get_framerate (title, &num, &denom);

    buf = g_new0 (gchar, G_ASCII_DTOSTR_BUF_SIZE);
    g_ascii_formatd (buf, G_ASCII_DTOSTR_BUF_SIZE, "%.0f", (start_delay * denom * 1000) / (gdouble) num);

    return buf;
  }

  return NULL;
}

static void
ogmrip_matroska_append_audio_file (OGMRipContainer *matroska,
    const gchar *filename, gint language, GPtrArray *argv)
{
  gchar *sync;

  if (language > -1)
  {
    const gchar *iso639_2;

    iso639_2 = ogmdvd_get_language_iso639_2 (language);
    if (iso639_2)
    {
      g_ptr_array_add (argv, g_strdup ("--language"));
      g_ptr_array_add (argv, g_strconcat ("0:", iso639_2, NULL));
#if MPLAYER_CHECK_VERSION(1,0,0,8)
      g_ptr_array_add (argv, g_strdup ("--language"));
      g_ptr_array_add (argv, g_strconcat ("1:", iso639_2, NULL));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */
    }
  }

  sync = ogmrip_matroska_get_sync (matroska);
  if (sync)
  {
    g_ptr_array_add (argv, g_strdup ("--sync"));
    g_ptr_array_add (argv, g_strdup_printf ("0:%s", sync));
    g_free (sync);
  }

  g_ptr_array_add (argv, g_strdup ("-D"));
  g_ptr_array_add (argv, g_strdup ("-S"));

  g_ptr_array_add (argv, g_strdup (filename));
}

static void
ogmrip_matroska_append_subp_file (OGMRipContainer *matroska, const gchar *filename,
    gint demuxer, gint charset, gint language, GPtrArray *argv)
{
  if (language > -1)
  {
    const gchar *iso639_2;

    iso639_2 = ogmdvd_get_language_iso639_2 (language);
    if (iso639_2)
    {
      g_ptr_array_add (argv, g_strdup ("--language"));
      g_ptr_array_add (argv, g_strconcat ("0:", iso639_2, NULL));
    }
  }

  switch (charset)
  {
    case OGMRIP_CHARSET_UTF8:
      g_ptr_array_add (argv, g_strdup ("--sub-charset"));
      g_ptr_array_add (argv, g_strdup ("0:UTF-8"));
      break;
    case OGMRIP_CHARSET_ISO8859_1:
      g_ptr_array_add (argv, g_strdup ("--sub-charset"));
      g_ptr_array_add (argv, g_strdup ("0:ISO-8859-1"));
      break;
    case OGMRIP_CHARSET_ASCII:
      g_ptr_array_add (argv, g_strdup ("--sub-charset"));
      g_ptr_array_add (argv, g_strdup ("0:ASCII"));
      break;
  }

  g_ptr_array_add (argv, g_strdup ("-s"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-D"));
  g_ptr_array_add (argv, g_strdup ("-A"));

  if (demuxer == OGMRIP_SUBP_DEMUXER_VOBSUB && !g_str_has_suffix (filename, ".idx"))
    g_ptr_array_add (argv, g_strconcat (filename, ".idx", NULL));
  else
    g_ptr_array_add (argv, g_strdup (filename));
}

static void
ogmrip_matroska_foreach_audio (OGMRipContainer *matroska, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  input = ogmrip_codec_get_output (codec);
  ogmrip_matroska_append_audio_file (matroska, input, language, argv);
}

static void
ogmrip_matroska_foreach_subp (OGMRipContainer *matroska, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;
  gint charset;

  input = ogmrip_codec_get_output (codec);
  charset = ogmrip_subp_get_charset (OGMRIP_SUBP (codec));
  ogmrip_matroska_append_subp_file (matroska, input, demuxer, charset, language, argv);
}

static void
ogmrip_matroska_foreach_chapters (OGMRipContainer *matroska, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  if (language > -1)
  {
    const gchar *iso639_2;

    iso639_2 = ogmdvd_get_language_iso639_2 (language);
    if (iso639_2)
    {
      g_ptr_array_add (argv, g_strdup ("--chapter-language"));
      g_ptr_array_add (argv, g_strdup (iso639_2));
    }
  }

  g_ptr_array_add (argv, g_strdup ("--chapter-charset"));
  g_ptr_array_add (argv, g_strdup ("UTF-8"));
  g_ptr_array_add (argv, g_strdup ("--chapters"));

  input = ogmrip_codec_get_output (codec);
  g_ptr_array_add (argv, g_strdup (input));
}

static void
ogmrip_matroska_foreach_file (OGMRipContainer *matroska, OGMRipFile *file, GPtrArray *argv)
{
  gchar *filename;

  filename = ogmrip_file_get_filename (file);
  if (filename)
  {
    gint charset, lang;

    lang = ogmrip_file_get_language (file);

    switch (ogmrip_file_get_type (file))
    {
      case OGMRIP_FILE_TYPE_AUDIO:
        ogmrip_matroska_append_audio_file (matroska, filename, lang, argv);
        break;
      case OGMRIP_FILE_TYPE_SUBP:
        charset = ogmrip_subp_file_get_charset (OGMRIP_SUBP_FILE (file));
        ogmrip_matroska_append_subp_file (matroska, filename, OGMRIP_SUBP_DEMUXER_AUTO, charset, lang, argv);
        break;
      default:
        g_assert_not_reached ();
        break;
    }
  }
  g_free (filename);
}

gchar **
ogmrip_matroska_command (OGMRipContainer *matroska)
{
  GPtrArray *argv;
  OGMRipVideo *video;
  const gchar *output, *label, *filename, *fourcc;
  guint tsize, tnumber;

  g_return_val_if_fail (OGMRIP_IS_MATROSKA (matroska), NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mkvmerge"));

  output = ogmrip_container_get_output (matroska);
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  fourcc = ogmrip_container_get_fourcc (matroska);
  if (fourcc)
  {
    g_ptr_array_add (argv, g_strdup ("--fourcc"));
    g_ptr_array_add (argv, g_strconcat ("0:", fourcc, NULL));
  }

#if defined(HAVE_X264_SUPPORT) && MKVMERGE_MAJOR_VERSION < 2
  /*
   * Option to merge h264 streams
   */
  g_ptr_array_add (argv, g_strdup ("--engage"));
  g_ptr_array_add (argv, g_strdup ("allow_avc_in_vfw_mode"));
#endif /* HAVE_X264_SUPPORT */

  g_ptr_array_add (argv, g_strdup ("--command-line-charset"));
  g_ptr_array_add (argv, g_strdup ("UTF-8"));

  video = ogmrip_container_get_video (matroska);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));

  g_ptr_array_add (argv, g_strdup ("-d"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-A"));
  g_ptr_array_add (argv, g_strdup ("-S"));
  g_ptr_array_add (argv, g_strdup (filename));

  ogmrip_container_foreach_audio (matroska, 
      (OGMRipContainerCodecFunc) ogmrip_matroska_foreach_audio, argv);
  ogmrip_container_foreach_subp (matroska, 
      (OGMRipContainerCodecFunc) ogmrip_matroska_foreach_subp, argv);
  ogmrip_container_foreach_chapters (matroska, 
      (OGMRipContainerCodecFunc) ogmrip_matroska_foreach_chapters, argv);
  ogmrip_container_foreach_file (matroska, 
      (OGMRipContainerFileFunc) ogmrip_matroska_foreach_file, argv);

  label = ogmrip_container_get_label (matroska);
  if (label)
  {
    g_ptr_array_add (argv, g_strdup ("--title"));
    g_ptr_array_add (argv, g_strdup_printf ("%s", label));
  }

  ogmrip_container_get_split (matroska, &tnumber, &tsize);
  if (tnumber > 1)
  {
    g_ptr_array_add (argv, g_strdup ("--split"));
    g_ptr_array_add (argv, g_strdup_printf ("%dM", tsize));
  }

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipMatroska, ogmrip_matroska, OGMRIP_TYPE_CONTAINER)

static void
ogmrip_matroska_class_init (OGMRipMatroskaClass *klass)
{
  OGMJobSpawnClass *spawn_class;
  OGMRipContainerClass *container_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  spawn_class->run = ogmrip_matroska_run;

  container_class = OGMRIP_CONTAINER_CLASS (klass);
  container_class->get_overhead = ogmrip_matroska_get_overhead;
}

static void
ogmrip_matroska_init (OGMRipMatroska *matroska)
{
}

static gint
ogmrip_matroska_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_matroska_command (OGMRIP_CONTAINER (spawn));
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_matroska_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_matroska_parent_class)->run (spawn);

  /*
   * If mkvmerge resturns 1, it's only a warning
   */
  if (ogmjob_exec_get_status (OGMJOB_EXEC (child)) == 1)
    result = OGMJOB_RESULT_COMPLETED;

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

static gint
ogmrip_matroska_get_overhead (OGMRipContainer *container)
{
  return MKV_OVERHEAD;
}

/**
 * ogmrip_matroska_new:
 * @output: The output file
 *
 * Creates a new #OGMRipMatroska.
 *
 * Returns: The new #OGMRipMatroska
 */
OGMJobSpawn *
ogmrip_matroska_new (const gchar *output)
{
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_MATROSKA, "output", output, NULL);
}

static OGMRipPluginContainer mkv_plugin =
{
  NULL,
  G_TYPE_NONE,
  "mkv",
  N_("Matroska Media (MKV)"),
  TRUE,
  G_MAXINT,
  G_MAXINT,
  NULL
};

static gint formats[] =
{
  OGMRIP_FORMAT_MPEG4,
  OGMRIP_FORMAT_H264,
  OGMRIP_FORMAT_THEORA,
  OGMRIP_FORMAT_AAC,
  OGMRIP_FORMAT_AC3,
  OGMRIP_FORMAT_DTS,
  OGMRIP_FORMAT_COPY,
  OGMRIP_FORMAT_MP3,
  OGMRIP_FORMAT_VORBIS,
  OGMRIP_FORMAT_PCM,
  OGMRIP_FORMAT_SRT,
  OGMRIP_FORMAT_VOBSUB,
  OGMRIP_FORMAT_SSA,
  -1
};

OGMRipPluginContainer *
ogmrip_init_plugin (void)
{
  mkv_plugin.type = OGMRIP_TYPE_MATROSKA;
  mkv_plugin.formats = formats;

  return &mkv_plugin;
}

