/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.test.remoting.transport.web;

import junit.framework.TestCase;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.transport.http.HTTPMetadataConstants;
import org.jboss.remoting.transport.web.WebUtil;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * @author <a href="mailto:tom.elrod@jboss.com">Tom Elrod</a>
 */
public abstract class WebInvokerTestClient extends TestCase
{
   public abstract String getLocatorURI();

   public void testPostInvocation() throws Exception
   {
      Client remotingClient = null;

      try
      {
         InvokerLocator locator = new InvokerLocator(getLocatorURI());
         System.out.println("Calling remoting server with locator uri of: " + getLocatorURI());

         remotingClient = new Client(locator);
         remotingClient.connect();

         Map metadata = new HashMap();
         metadata.put(Client.RAW, Boolean.TRUE);
         metadata.put("TYPE", "POST");

         Properties headerProps = new Properties();
         headerProps.put("Content-Type", "application/soap+xml");

         addHeaders(headerProps);

         metadata.put("HEADER", headerProps);

         Object response = null;

         // test with null return expected
         response = remotingClient.invoke(WebInvocationHandler.NULL_RETURN_PARAM, metadata);
         System.out.println("First response should be null and was: " + response);
         assertNull(response);

         response = remotingClient.invoke("Do something", metadata);
         System.out.println("Second response should be " + WebInvocationHandler.HTML_PAGE_RESPONSE + " and was: " + response);
         assertEquals(WebInvocationHandler.HTML_PAGE_RESPONSE, response);

         headerProps.put("Content-Type", WebUtil.BINARY);
         response = remotingClient.invoke(new ComplexObject(2, "foo", true), metadata);
         System.out.println("Third response should be " + WebInvocationHandler.OBJECT_RESPONSE_VALUE + " and was: " + response);
         assertEquals(WebInvocationHandler.OBJECT_RESPONSE_VALUE, response);

         response = remotingClient.invoke(new ComplexObject(2, "foo", true, 3000), metadata);
         System.out.println("Fourth response should be " + WebInvocationHandler.LARGE_OBJECT_RESPONSE_VALUE + " and was: " + response);
         assertEquals(WebInvocationHandler.LARGE_OBJECT_RESPONSE_VALUE, response);

         headerProps.put("Content-Type", "application/soap+xml");
         response = remotingClient.invoke(WebInvocationHandler.STRING_RETURN_PARAM, metadata);
         System.out.println("Fifth response should be " + WebInvocationHandler.RESPONSE_VALUE + " and was: " + response);
         assertEquals(WebInvocationHandler.RESPONSE_VALUE, response);

         checkUserAgent(remotingClient, metadata);

         makeExceptionInvocation(remotingClient, metadata);

         remotingClient.invokeOneway("Do something", metadata, true);

         remotingClient.invokeOneway("Do something", metadata, false);
      }
      catch (Throwable throwable)
      {
         throw new Exception(throwable);
      }
      finally
      {
         if (remotingClient != null)
         {
            remotingClient.disconnect();
         }
      }


   }

   protected void checkUserAgent(Client remotingClient, Map metadata)
         throws Throwable
   {
      Object response;
      String remotingUserAgentValue = "JBossRemoting - ";
      response = remotingClient.invoke(WebInvocationHandler.USER_AGENT_PARAM, metadata);
      System.out.println("Sixth response start with " + remotingUserAgentValue + " and was: " + response);
      boolean correctUserAgent = ((String) response).startsWith(remotingUserAgentValue);
      assertTrue("User-Agent should be begin with " + remotingUserAgentValue + " but was " + response, correctUserAgent);
   }

   protected void makeExceptionInvocation(Client remotingClient, Map metadata)
         throws Throwable
   {


      Object response = null;
      try
      {
         response = remotingClient.invoke(WebInvocationHandler.THROW_EXCEPTION_PARAM, metadata);
         assertTrue("Should have thrown WebServerError and not made it to here.", false);
      }
      catch (IOException error)
      {
         // having to check class name instead of just catching type WebServerError so
         // can use for backwards compatibility tests since WebServerError is new since 2.0.0.CR1.
         if (error.getClass().getName().endsWith("WebServerError"))
         {
            assertTrue(true);
         }
         else
         {
            assertTrue("Did not get WebServerError thrown as expected", false);
         }
      }
      metadata.put(HTTPMetadataConstants.NO_THROW_ON_ERROR, "true");
      response = remotingClient.invoke(WebInvocationHandler.THROW_EXCEPTION_PARAM, metadata);
      if (response instanceof Exception)
      {
         System.out.println("Return from invocation is of type Exception as expected.");
         assertTrue("Received exception return as expected.", true);
      }
      else
      {
         System.out.println("Did not get Exception type returned as expected.");
         assertTrue("Should have received Exception as return.", false);
      }
   }

   public void testGetInvocation() throws Exception
   {
      Client remotingClient = null;

      try
      {
         InvokerLocator locator = new InvokerLocator(getLocatorURI());
         System.out.println("Calling remoting server with locator uri of: " + getLocatorURI());

         remotingClient = new Client(locator);
         remotingClient.connect();

         Map metadata = new HashMap();
         metadata.put(Client.RAW, Boolean.TRUE);
         metadata.put("TYPE", "GET");

         Object response = null;

         // test with null return expected
         response = remotingClient.invoke((Object) null, metadata);
         System.out.println("Response should be " + WebInvocationHandler.HTML_PAGE_RESPONSE + " and was: " + response);
         assertEquals(WebInvocationHandler.HTML_PAGE_RESPONSE, response);

         response = remotingClient.invoke((Object) null, metadata);
         System.out.println("Response should be " + WebInvocationHandler.HTML_PAGE_RESPONSE + " and was: " + response);
         assertEquals(WebInvocationHandler.HTML_PAGE_RESPONSE, response);
      }
      catch (Throwable throwable)
      {
         throw new Exception(throwable);
      }
      finally
      {
         if (remotingClient != null)
         {
            remotingClient.disconnect();
         }
      }


   }

   protected void addHeaders(Properties headerProps)
   {
      //NO OP - for overriding by sub-classes.
   }
}