/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.remoting.callback;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import org.jboss.logging.Logger;
import org.jboss.remoting.Client;

/**
 * CallbackPoller is used to simulate push callbacks on transports that don't support
 * bidirectional connections.  It will periodically pull callbacks from the server
 * and pass them to the InvokerCallbackHandler.
 *
 * @author <a href="mailto:tom.elrod@jboss.com">Tom Elrod</a>
 * @author <a href="mailto:ron.sigal@jboss.com">Ron Sigal</a>
 */
public class CallbackPoller extends TimerTask
{
   /*
    * Implementation note.
    *
    * CallbackPoller uses two, or possibly three, threads.  The first thread is the
    * Timer thread, which periodically pulls callbacks from the server and adds them
    * to toHandleList.  The second thread takes callbacks from toHandleList, passes
    * them to the CallbackHandler, and, if an acknowledgement is requested for a
    * callback, it adds the callback to toAcknowledgeList.  The third thread, which is
    * created in response to the first callback for which an acknowledgement is requested,
    * takes the contents of toAcknowledgeList and acknowledges them in a batch.
    *
    * CallbackPoller will not shut down until all received callbacks have been processed
    * by the CallbackHandler and acknowledgements have been sent for all callbacks for
    * which acknowledgements have been requested.
    */

   /**
    * Default polling period for getting callbacks from the server.
    * Default is 5000 milliseconds.
    */
   public static final long DEFAULT_POLL_PERIOD = 5000;

   /**
    * The key value to use to specify the desired poll period
    * within the metadata Map.
    */
   public static final String CALLBACK_POLL_PERIOD = "callbackPollPeriod";

   /** The key value to use in metadata Map to specify the desired scheduling mode. */
   public static final String CALLBACK_SCHEDULE_MODE = "scheduleMode";

   /** Use java.util.timer.schedule(). */
   public static final String SCHEDULE_FIXED_RATE = "scheduleFixedRate";

   /** Use java.util.timer.scheduleAtFixedRate(). */
   public static final String SCHEDULE_FIXED_DELAY = "scheduleFixedDelay";

   /** The key to use in metadata Map to request statistics.  The associated
    *  is ignored. */
   public static final String REPORT_STATISTICS = "reportStatistics";

   private Client client = null;
   private InvokerCallbackHandler callbackHandler = null;
   private Map metadata = null;
   private Object callbackHandlerObject = null;
   private long pollPeriod = DEFAULT_POLL_PERIOD;
   private Timer timer;
   private String scheduleMode = SCHEDULE_FIXED_RATE;
   private boolean reportStatistics;

   private ArrayList toHandleList = new ArrayList();
   private ArrayList toAcknowledgeList = new ArrayList();
   private HandleThread handleThread;
   private AcknowledgeThread acknowledgeThread;

   private static final Logger log = Logger.getLogger(CallbackPoller.class);


   public CallbackPoller(Client client, InvokerCallbackHandler callbackhandler, Map metadata, Object callbackHandlerObject)
   {
      this.client = client;
      this.callbackHandler = callbackhandler;
      this.metadata = metadata;
      this.callbackHandlerObject = callbackHandlerObject;
   }

   public void start() throws Exception
   {
      if (callbackHandler == null)
      {
         throw new NullPointerException("Can not poll for callbacks when InvokerCallbackHandler is null.");
      }
      if (client != null)
      {
         client.connect();
      }
      else
      {
         throw new NullPointerException("Can not poll for callbacks when Client is null.");
      }

      if (metadata != null)
      {
         Object val = metadata.get(CALLBACK_POLL_PERIOD);
         if (val != null)
         {
            if (val instanceof String)
            {
               try
               {
                  pollPeriod = Long.parseLong((String) val);
               }
               catch (NumberFormatException e)
               {
                  log.warn("Error converting " + CALLBACK_POLL_PERIOD + " to type long.  " + e.getMessage());
               }
            }
            else
            {
               log.warn("Value for " + CALLBACK_POLL_PERIOD + " configuration must be of type " + String.class.getName() +
                        " and is " + val.getClass().getName());
            }
         }
         val = metadata.get(CALLBACK_SCHEDULE_MODE);
         if (val != null)
         {
            if (val instanceof String)
            {
               if (SCHEDULE_FIXED_DELAY.equals(val) || SCHEDULE_FIXED_RATE.equals(val))
               {
                  scheduleMode = (String) val;
               }
               else
               {
                  log.warn("Unrecognized value for " + CALLBACK_SCHEDULE_MODE + ": " + val);
                  log.warn("Using " + scheduleMode);
               }
            }
            else
            {
               log.warn("Value for " + CALLBACK_SCHEDULE_MODE + " must be of type " + String.class.getName() +
                     " and is " + val.getClass().getName());
            }
         }
         if (metadata.get(REPORT_STATISTICS) != null)
         {
            reportStatistics = true;
         }
      }

      handleThread = new HandleThread("HandleThread");
      handleThread.start();

      timer = new Timer(true);

      if (SCHEDULE_FIXED_DELAY.equals(scheduleMode))
         timer.schedule(this, pollPeriod, pollPeriod);
      else
         timer.scheduleAtFixedRate(this, pollPeriod, pollPeriod);
   }

   public synchronized void run()
   {
      // need to pull callbacks from server and give them to callback handler
      try
      {
         List callbacks = client.getCallbacks(callbackHandler);

         if (callbacks != null && callbacks.size() > 0)
         {
            synchronized (toHandleList)
            {
               toHandleList.addAll(callbacks);
               if (toHandleList.size() == callbacks.size())
                  toHandleList.notify();
            }
         }

         if (reportStatistics)
            reportStatistics(callbacks);
      }
      catch (Throwable throwable)
      {
         log.error("Error getting callbacks from server.", throwable);
      }
   }

   /**
    * stop() will not return until all received callbacks have been processed
    * by the CallbackHandler and acknowledgements have been sent for all callbacks for
    * which acknowledgements have been requested.
    */
   public synchronized void stop()
   {
      log.debug(this + " is shutting down");
      
      // run() and stop() are synchronized so that stop() will wait until run() has finished
      // adding any callbacks it has received to toHandleList.  Therefore, once cancel()
      // returns, no more callbacks will arrive from the server.
      cancel();

      // HandleThread.shutdown() will not return until all received callbacks have been
      // processed and, if necessary, added to toAcknowledgeList.
      if (handleThread != null)
      {
         handleThread.shutdown();
         handleThread = null;
      }

      // AcknowledgeThread.shutdown() will not return until acknowledgements have been sent
      // for all callbacks for which acknowledgements have been requested.
      if (acknowledgeThread != null)
      {
         acknowledgeThread.shutdown();
         acknowledgeThread = null;
      }

      if (timer != null)
      {
         timer.cancel();
         timer = null;
      }
      
      log.debug(this + " has shut down");
   }

   class HandleThread extends Thread
   {
      boolean running = true;
      boolean done;
      ArrayList toHandleListCopy = new ArrayList();
      Callback callback;

      HandleThread(String name)
      {
         super(name);
      }
      public void run()
      {
         while (true)
         {
            synchronized (toHandleList)
            {
               if (toHandleList.isEmpty() && running)
               {
                  try
                  {
                     toHandleList.wait();
                  }
                  catch (InterruptedException e)
                  {
                     log.warn("unexpected interrupt");
                     continue;
                  }
               }

               // If toHandleList is empty, then running must be false.  We return
               // only when both conditions are true.
               if (toHandleList.isEmpty())
               {
                  done = true;
                  toHandleList.notify();
                  return;
               }

               toHandleListCopy.addAll(toHandleList);
               toHandleList.clear();
            }

            while (!toHandleListCopy.isEmpty())
            {
               try
               {
                  callback = (Callback) toHandleListCopy.remove(0);
                  callback.setCallbackHandleObject(callbackHandlerObject);
                  callbackHandler.handleCallback(callback);
               }
               catch (HandleCallbackException e)
               {
                  log.error("Error delivering callback to callback handler (" + callbackHandler + ").", e);
               }

               checkForAcknowledgeRequest(callback);
            }
         }
      }

      /**
       *  Once CallbackPoller.stop() has called HandleThread.shutdown(), CallbackPoller.run()
       *  has terminated and no additional callbacks will be received.  shutdown() will
       *  not return until HandleThread has processed all received callbacks.
       *
       *  Either run() or shutdown() will enter its own synchronized block first.
       *
       *  case 1): run() enters its synchronized block first:
       *     If toHandleList is empty, then run() will reach toHandleList.wait(), shutdown()
       *     will wake up run(), and run() will exit.  If toHandleList is not empty, then run()
       *     will process all outstanding callbacks and return to its synchronized block.  At
       *     this point, either case 1) (with toHandleList empty) or case 2) applies.
       *
       *  case 2): shutdown() enters its synchronized block first:
       *     run() will process all outstanding callbacks and return to its synchronized block.
       *     After shutdown() reaches toHandleList.wait(), run() will enter its synchronized
       *     block, find running == false and toHandleList empty, and it will exit.
       */
      protected void shutdown()
      {
         log.debug(this + " is shutting down");
         synchronized (toHandleList)
         {
            running = false;
            toHandleList.notify();
            while (!done)
            {
               try
               {
                  toHandleList.wait();
               }
               catch (InterruptedException ignored) {}
               return;
            }
         }
         log.debug(this + " has shut down");
      }
   }


   class AcknowledgeThread extends Thread
   {
      boolean running = true;
      boolean done;
      ArrayList toAcknowledgeListCopy = new ArrayList();

      AcknowledgeThread(String name)
      {
         super(name);
      }
      public void run()
      {
         while (true)
         {
            synchronized (toAcknowledgeList)
            {
               while (toAcknowledgeList.isEmpty() && running)
               {
                  try
                  {
                     toAcknowledgeList.wait();
                  }
                  catch (InterruptedException e)
                  {
                     log.warn("unexpected interrupt");
                     continue;
                  }
               }

               // If toAcknowledgeList is empty, then running must be false.  We return
               // only when both conditions are true.
               if (toAcknowledgeList.isEmpty())
               {
                  done = true;
                  toAcknowledgeList.notify();
                  return;
               }

               toAcknowledgeListCopy.addAll(toAcknowledgeList);
               toAcknowledgeList.clear();
            }

            try
            {
               if (log.isTraceEnabled())
               {
                  Iterator it = toAcknowledgeListCopy.iterator();
                  while (it.hasNext())
                  {
                     Callback cb = (Callback) it.next();
                     Map map = cb.getReturnPayload();
                     log.trace("acknowledging: " + map.get(ServerInvokerCallbackHandler.CALLBACK_ID));
                  }
               }
               client.acknowledgeCallbacks(callbackHandler, toAcknowledgeListCopy);
               toAcknowledgeListCopy.clear();
            }
            catch (Throwable t)
            {
               log.error("Error acknowledging callback for callback handler (" + callbackHandler + ").", t);
            }
         }
      }

      /**
       *  Once CallbackPoller.stop() has called AcknowledgeThread.shutdown(), HandleThread
       *  has terminated and no additional callbacks will be added to toAcknowledgeList.
       *  shutdown() will not return until AcknowledgeThread has acknowledged all callbacks
       *  in toAcknowledgeList.
       *
       *  Either run() or shutdown() will enter its own synchronized block first.
       *
       *  case 1): run() enters its synchronized block first:
       *     If toAcknowledgeList is empty, then run() will reach toAcknowledgeList.wait(),
       *     shutdown() will wake up run(), and run() will exit.  If toAcknowledgeList is not
       *     empty, then run() will process all callbacks in toAcknowledgeList and return to
       *     its synchronized block.  At this point, either case 1) (with toAcknowledgeList
       *     empty) or case 2) applies.
       *
       *  case 2): shutdown() enters its synchronized block first:
       *     run() will process all callbacks in toAcknowledgeList and return to its
       *     synchronized block.  After shutdown() reaches toAcknowledgeList.wait(), run()
       *     will enter its synchronized block, find running == false and toAcknowledgeList
       *     empty, and it will exit.
       */
      public void shutdown()
      {
         log.debug(this + " is shutting down");      
         synchronized (toAcknowledgeList)
         {
            running = false;
            toAcknowledgeList.notify();
            while (!done)
            {
               try
               {
                  toAcknowledgeList.wait();
               }
               catch (InterruptedException ignored) {}
               return;
            }
         }
         log.debug(this + " has shut down");
      }
   }


   private void checkForAcknowledgeRequest(Callback callback)
   {
      Map returnPayload = callback.getReturnPayload();
      if (returnPayload != null)
      {
         Object callbackId = returnPayload.get(ServerInvokerCallbackHandler.CALLBACK_ID);
         if (callbackId != null)
         {
            Object o = returnPayload.get(ServerInvokerCallbackHandler.REMOTING_ACKNOWLEDGES_PUSH_CALLBACKS);
            if (o instanceof String  && Boolean.valueOf((String)o).booleanValue() ||
                o instanceof Boolean && ((Boolean)o).booleanValue())
            {
               synchronized (toAcknowledgeList)
               {
                  toAcknowledgeList.add(callback);
                  if (toAcknowledgeList.size() == 1)
                  {
                     if (acknowledgeThread == null)
                     {
                        acknowledgeThread = new AcknowledgeThread("AcknowledgeThread");
                        acknowledgeThread.start();
                     }
                     else
                     {
                        toAcknowledgeList.notify();
                     }
                  }
               }
            }
         }
      }
   }


   private void reportStatistics(List callbacks)
   {
      int toHandle;
      int toAcknowledge = 0;

      synchronized (toHandleList)
      {
         toHandle = toHandleList.size() + handleThread.toHandleListCopy.size();
      }

      synchronized (toAcknowledgeList)
      {
         if (acknowledgeThread != null)
            toAcknowledge = toAcknowledgeList.size() + acknowledgeThread.toAcknowledgeListCopy.size();
      }

      StringBuffer message = new StringBuffer("\n");
      message.append("================================\n")
             .append("  retrieved " + callbacks.size() + " callbacks\n")
             .append("  callbacks waiting to be processed: " + toHandle + "\n")
             .append("  callbacks waiting to be acknowledged: " + toAcknowledge + "\n")
             .append("================================");
      log.info(message);
   }
}