/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2007 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)ConnectionConsumerImpl.java	1.19 06/27/07
 */ 

package com.sun.messaging.jmq.jmsclient;

import java.io.*;
import javax.jms.*;

import com.sun.messaging.AdministeredObject;
import com.sun.messaging.jmq.io.*;

/** For application servers, Connections provide a special facility for
  * creating a ConnectionConsumer. The messages it is to consume is
  * specified by a Destination and a Property Selector. In addition, a
  * ConnectionConsumer must be given a ServerSessionPool to use for
  * processing its messages.
  *
  * <P>Normally, when traffic is light, a ConnectionConsumer gets a
  * ServerSession from its pool; loads it with a single message; and,
  * starts it. As traffic picks up, messages can back up. If this happens,
  * a ConnectionConsumer can load each ServerSession with more than one
  * message. This reduces the thread context switches and minimizes resource
  * use at the expense of some serialization of a message processing.
  *
  * @see javax.jms.QueueConnection#createConnectionConsumer
  * @see javax.jms.TopicConnection#createConnectionConsumer
  * @see javax.jms.TopicConnection#createDurableConnectionConsumer
  */
public class ConnectionConsumerImpl extends Consumer
                implements ConnectionConsumer, Traceable {

    protected ServerSessionPool serverSessionPool;
    protected int maxMessages;

    private SessionQueue readQueue = null;
    private ConnectionConsumerReader reader = null;
    private Long readQueueId = null;

    private ServerSession serverSession = null;
    private SessionImpl session = null;

/*
    public ConnectionConsumerImpl(ConnectionImpl connection,
           Destination d, String messageSelector,
           ServerSessionPool sessionPool,
           //int maxMessages, boolean isTopic) throws JMSException {

        //super(connection, d, messageSelector, isTopic, false);
        this.serverSessionPool = sessionPool;
        this.maxMessages = maxMessages;
        init();
    }
*/
    public ConnectionConsumerImpl(ConnectionImpl connection, Destination d,
           String messageSelector, ServerSessionPool sessionPool,
           int maxMessages, String durablename)
           throws JMSException {

        super(connection, d, messageSelector, false);
        if (durablename != null) {
            if (connection.clientID == null) {
                String errorString =
                AdministeredObject.cr.getKString(AdministeredObject.cr.X_INVALID_CLIENT_ID, "\"\"");
                throw new JMSException (errorString, AdministeredObject.cr.X_INVALID_CLIENT_ID);
            }
            setDurable(true);
            setDurableName (durablename);
        }
        this.serverSessionPool = sessionPool;
        this.maxMessages = maxMessages;
        init();
    }

    public void init() throws JMSException {
        readQueue = new SessionQueue();

        if (connection.getIsStopped()) {
            readQueue.setIsLocked(true);
        }

        readQueueId = connection.getNextSessionId();
        connection.addToReadQTable(readQueueId, readQueue);

        reader = new ConnectionConsumerReader(this);
        reader.start();
        addInterest();
    }

    /**
     * add this consumer's interest
     *
     * @exception JMSException if fails to register the interest to broker
     */
    private void addInterest() throws JMSException {
        connection.addConnectionConsumer(this);
        registerInterest();
    }

    /**
     * remove this consumer's interest
     *
     * @param destroy if true deregister interest from broker
     *                if false do not deregister from broker (for durable)
     *
     * @exception JMSException if fails to deregister from broker
     */
    private void removeInterest() throws JMSException {
        connection.removeConnectionConsumer(this);
        deregisterInterest();
    }

    /**
     * get the reader's Id for this consumer
     *
     * @return the reader Id
     */
    protected Long getReadQueueId() {
        return readQueueId;
    }

    /**
     * get the read queue for this connection consumer
     *
     * @return the associated read queue
     */
    protected SessionQueue getReadQueue() {
        return readQueue;
    }

    /**
     * Load a message to the current ServerSession, if no current
     * ServerSession, a ServerSession will be retrieved from ServerSessionPool.
     *  This method is called by the ConnectionConsumerReader.
     *
     * @param message the message to be consumed
     *
     * @exception JMSException if fails to obtain a valid session from app server
     */
    protected void onMessage(MessageImpl message) throws JMSException {
        if (session == null) {
            serverSession = serverSessionPool.getServerSession(); //may block

            try {

            session = (SessionImpl)serverSession.getSession();
            if (session.getConnection() != connection) {
                String errorString = AdministeredObject.cr.getKString(AdministeredObject.cr.X_SVRSESSION_INVALID);
                throw new JMSException(errorString, AdministeredObject.cr.X_SVRSESSION_INVALID);
            }
            if (session.getMessageListener() == null) {
                String errorString = AdministeredObject.cr.getKString(AdministeredObject.cr.X_SVRSESSION_INVALID);
                throw new javax.jms.IllegalStateException(errorString, AdministeredObject.cr.X_SVRSESSION_INVALID);
            }

            } catch (JMSException e) {
            if (session != null && serverSession instanceof com.sun.messaging.jmq.jmsspi.ServerSession)
                ((com.sun.messaging.jmq.jmsspi.ServerSession)serverSession).destroy();
            session = null;
            serverSession = null;
            throw e;
            }
        }
        message.setSession(session);
        session.loadMessageToServerSession(message, serverSession);
    }

    /**
     * Start the current ServerSession.  This method is called from
     * the ConnectionConsumerReader when it has loaded maxMessages
     * to the current ServerSession.
     *
     * @exception JMSException if ServerSession.start() fails
     */
    protected void startServerSession() throws JMSException {
        if (serverSession != null) {
            serverSession.start();
            serverSession = null;
            session = null;
        }
    }

    protected int getMaxMessages() {
        return maxMessages;
    }

    /** Get the server session pool associated with this connection consumer.
      *
      * @return the server session pool used by this connection consumer.
      *
      * @exception JMSException if a JMS error occurs.
      */
    public ServerSessionPool getServerSessionPool() throws JMSException {
        return serverSessionPool;
    }

    /** Since a provider may allocate some resources on behalf of a
      * ConnectionConsumer outside the JVM, clients should close them when
      * they are not needed. Relying on garbage collection to eventually
      * reclaim these resources may not be timely enough.
      *
      * @exception JMSException if a JMS error occurs.
      */
    public void close() throws JMSException {
        reader.close();
        removeInterest();
        connection.removeFromReadQTable(readQueueId);

        isClosed = true;
    }

    //should be called after connection stopped all sessions
    protected void stop() {
        readQueue.stop(false);
    }

    protected void start() {
        readQueue.start();
    }

    public void dump (PrintStream ps) {

        ps.println ("------ ConnectionConsumerImpl dump ------");

        ps.println ("Interest ID: " + getInterestId());
        ps.println ("is registered: " + getIsRegistered());
        //ps.println ("isTopic: " + getIsTopic());
        ps.println ("is durable: " + getDurable());

        if ( durable ) {
            ps.println ("durableName: " + getDurableName());
        }

        ps.println ("destination: " + getDestination());
        ps.println ("selector: " + messageSelector);
        ps.println("maxMessages: " + maxMessages);

    }

    protected java.util.Hashtable getDebugState(boolean verbose) {
        java.util.Hashtable ht = super.getDebugState(verbose);

        ht.put("maxMessages", String.valueOf(maxMessages));
        return ht;
    }
}
