/* GSnes9x -- globalprefs.c: Global Preferences dialog related functions.
 * Copyright (C) 1999 Canek Pelez Valds <canek@abulafia.fciencias.unam.mx>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsnes9x.h"

void
global_preferences_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	GlobalPrefsData *gpd = global_prefs_data_new ();
	GtkWidget       *prefs_dialog;
	GtkWidget       *notebook;
	GtkWidget       *cont;

	F_NAME();
	
	gsnes_data->dialog_open = TRUE;
	
	gpd->data = gsnes_data;
	
	prefs_dialog = gnome_dialog_new (_("GSnes9x Global Preferences"),
					 GNOME_STOCK_BUTTON_OK,
					 GNOME_STOCK_BUTTON_APPLY,
					 GNOME_STOCK_BUTTON_CLOSE,
					 GNOME_STOCK_BUTTON_HELP,
					 NULL);

	gtk_window_set_modal (GTK_WINDOW (prefs_dialog), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (prefs_dialog),
				      GTK_WINDOW (gsnes_data->app));
	
	gtk_object_set_data (GTK_OBJECT (prefs_dialog), "global-prefs",
			     (gpointer) gpd);
	
	gpd->global_prefs_dialog = prefs_dialog;

	notebook = gtk_notebook_new ();

	cont = global_prefs_system_ops (gpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("System")));
	gtk_widget_show (cont);
	
	cont = global_prefs_graphics_ops (gpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Graphics")));
	gtk_widget_show (cont);

	cont = global_prefs_sound_ops (gpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Sound")));
	gtk_widget_show (cont);
	
	cont = global_prefs_speed_ops (gpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Speed up/down")));
	gtk_widget_show (cont);

	cont = global_prefs_joystick_ops (gpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Joystick")));
	gtk_widget_show (cont);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (prefs_dialog)->vbox),
			    notebook, FALSE, FALSE, 10);
	gtk_widget_show (notebook);
	
	global_prefs_dialog_set_status (gpd);
	global_prefs_connect_changes   (gpd);
	global_prefs_set_tooltips      (gpd);
	
	gtk_signal_connect (GTK_OBJECT (prefs_dialog), "close",
			    GTK_SIGNAL_FUNC (global_prefs_delete_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT (prefs_dialog), "destroy",
			    GTK_SIGNAL_FUNC (global_prefs_free_cb),
			    (gpointer) gpd);
	
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog), 0,
				     GTK_SIGNAL_FUNC (global_prefs_apply_cb),
				     (gpointer) gpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog), 0,
				     GTK_SIGNAL_FUNC (global_prefs_close_cb),
				     (gpointer) gpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog), 1,
				     GTK_SIGNAL_FUNC (global_prefs_apply_cb),
				     (gpointer) gpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog), 2,
				     GTK_SIGNAL_FUNC (global_prefs_close_cb),
				     (gpointer) gpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog), 3,
				     GTK_SIGNAL_FUNC (global_prefs_help_cb),
				     (gpointer) notebook);
				     
	gnome_dialog_set_sensitive (GNOME_DIALOG (prefs_dialog), 0, FALSE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (prefs_dialog), 1, FALSE);

	gtk_widget_show (prefs_dialog);
}

GlobalPrefs*
global_prefs_new ()
{
	gint i, j;
	GlobalPrefs *gp = g_new (GlobalPrefs, 1);

	F_NAME();
	
	for (i = 0; i < MIX_BOOL_PREFS+GLOB_BOOL_PREFS; i++)
		gp->bool_prefs[i] = FALSE;

	for (i = 0; i < GLOB_STR_PREFS; i++)
		gp->string_prefs[i] = NULL;
	
	for (i = 0; i < MIX_NUM_PREFS; i++)
		gp->number_prefs[i] = 0;
	
	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			gp->buttons[i][j] = j;
	return gp;
}

GlobalPrefsData*
global_prefs_data_new ()
{
	gint i, j;
	GlobalPrefsData *gp = g_new (GlobalPrefsData, 1);

	F_NAME();
	
	gp->changed = FALSE;
	gp->nodata  = FALSE;

	gp->global_prefs_dialog = NULL;

	for (i = 0; i < MIX_BOOL_PREFS+GLOB_BOOL_PREFS; i++)
		gp->check[i] = NULL;
	for (i = 0; i < CODES; i++) {
		gp->code_add[i]   = NULL;
		gp->code_quit[i]  = NULL;
	}
	gp->mono           = NULL;
	
	for (i = 0; i < MIX_STR_PREFS+GLOB_STR_PREFS; i++)
		gp->entry[i] = NULL;

	for (i = 0; i < MIX_NUM_PREFS; i++)
		gp->spin[i] = NULL;

	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			gp->buttons[i][j] = NULL;

	for (i = 0; i < CODES; i++)
		for (j = 0; j < MAX_CODES; j++)
			gp->code[i][j] = NULL;

	gp->data = NULL;
	
	return gp;
}

GtkWidget*
global_prefs_system_ops (GlobalPrefsData *gpd)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *text;
	GtkWidget *check;

	F_NAME();
	
	table = gtk_table_new (4, 3, FALSE);
	
	label = gtk_label_new (_("Snes9X command:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);

	label = gtk_label_new (_("ROM directory:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);
	
	label = gtk_label_new (_("Snes9X server:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);

	label = gtk_label_new (_("Snes9X server port:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);
	
	text = gtk_entry_new_with_max_length (CONF_STR_LEN);
	gtk_table_attach (GTK_TABLE (table), text, 2, 3, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	gpd->entry[SNES9X_COMMAND-ROM_STR_PREFS] = text;

	text = gnome_file_entry_new(NULL, _("Select ROM directory"));
	gnome_file_entry_set_directory  (GNOME_FILE_ENTRY(text), TRUE);
	gtk_table_attach (GTK_TABLE (table), text, 2, 3, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	gpd->entry[ROMS_DIR-ROM_STR_PREFS] = gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(text));	

	text = gtk_entry_new_with_max_length (CONF_STR_LEN);
	gtk_table_attach (GTK_TABLE (table), text, 2, 3, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	gpd->entry[SERVER-ROM_STR_PREFS] = text;

	text = gtk_entry_new_with_max_length (CONF_STR_LEN);
	gtk_table_attach (GTK_TABLE (table), text, 2, 3, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	gpd->entry[NETPORT-ROM_STR_PREFS] = text;

	check = gtk_check_button_new_with_label (_("Esound support"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[WITH_ESD-ROM_BOOL_PREFS] = check;

	check = gtk_check_button_new_with_label (_("Network play"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[NETPLAY-ROM_BOOL_PREFS] = check;

	gtk_signal_connect (GTK_OBJECT (gpd->check[NETPLAY-ROM_BOOL_PREFS]),
			    "toggled", GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) gpd->entry[SERVER-ROM_STR_PREFS]);
	gtk_signal_connect (GTK_OBJECT (gpd->check[NETPLAY-ROM_BOOL_PREFS]),
			    "toggled", GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) gpd->entry[NETPORT-ROM_STR_PREFS]);
		
	return table;
}

GtkWidget*
global_prefs_graphics_ops (GlobalPrefsData *gpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *frame;
	GtkWidget *option;
	GtkWidget *menu;
	GtkWidget *item;

	F_NAME();
	
	table = gtk_table_new (4, 3, TRUE);
	
	check = gtk_check_button_new_with_label (_("Transparency effects"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[TRANSPARENCY] = check;
	
	check = gtk_check_button_new_with_label
		(_("16-bit internal screen rendering"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[SIXTEEN] = check;
	
	check = gtk_check_button_new_with_label (_("Hi-res"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[HIRES] = check;

	check = gtk_check_button_new_with_label (_("No mode switch"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[NOMODESWITCH] = check;

	check = gtk_check_button_new_with_label (_("Full screen"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[FULLSCREEN] = check;

	check = gtk_check_button_new_with_label (_("Scale"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[SCALE] = check;
	
	check = gtk_check_button_new_with_label (_("Display frame rate"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[DISPLAYFRAMERATE] = check;

	frame = gtk_frame_new (_("Image Processing"));

	option = gtk_option_menu_new ();
	
	menu = gtk_menu_new ();

	item = gtk_menu_item_new_with_label (_("None"));
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_widget_show (item);
	
	item = gtk_menu_item_new_with_label (_("TV Mode"));
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_widget_show (item);	
	
	item = gtk_menu_item_new_with_label (_("Super 2xSaI"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	item = gtk_menu_item_new_with_label (_("Super Eagle"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	item = gtk_menu_item_new_with_label (_("2xSaI"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	item = gtk_menu_item_new_with_label (_("Bi-linear filter"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	gtk_widget_show (menu);
	
	gtk_option_menu_set_menu (GTK_OPTION_MENU (option), menu);
	gtk_widget_show (option);
	gpd->spin[FILTER_NUMBER] = option;

	gtk_container_add (GTK_CONTAINER (frame), option);
	gtk_table_attach (GTK_TABLE (table), frame, 2, 3, 0, 2,
			  GTK_FILL, GTK_FILL, 5, 4);
	gtk_widget_show (frame);

	return table;
}

GtkWidget*
global_prefs_sound_ops (GlobalPrefsData *gpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *radio;
	GtkWidget *spin;
	GtkObject *adj;
	
	F_NAME();
	
	table = gtk_table_new (4, 4, TRUE);

	check = gtk_check_button_new_with_label (_("No sound"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[NOSOUND] = check;
	
	check = gtk_check_button_new_with_label (_("Sound skip"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[SOUNDSKIP] = check;
	
	adj = gtk_adjustment_new (0, 0, 3, 1, 1, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 2, 3, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	gpd->spin[SOUNDSKIP_NUMBER] = spin;

	radio = gtk_radio_button_new_with_label (NULL, _("Stereo"));
	gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (radio);
	gpd->check[STEREO] = radio;

	radio = gtk_radio_button_new_with_label (
		gtk_radio_button_group (GTK_RADIO_BUTTON (radio)),
		_("Mono"));
	gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (radio);
	gpd->mono = radio;
	
	check = gtk_check_button_new_with_label (_("Sound quality"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[SOUNDQUALITY] = check;

	adj = gtk_adjustment_new (4, 0, 7, 1, 1, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 2, 3, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	gpd->spin[SOUNDQUALITY_NUMBER] = spin;

	check = gtk_check_button_new_with_label (_("Volume envelope"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[ENVX] = check;

	check = gtk_check_button_new_with_label (_("Sound sync"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[SOUNDSYNC] = check;

	check = gtk_check_button_new_with_label (_("Alternative decoding"));
	gtk_table_attach (GTK_TABLE (table), check, 2, 3, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[ALTSAMPLEDECODE] = check;

	check = gtk_check_button_new_with_label (_("Interpolated sound"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[INTERPOLATEDSOUND] = check;

	check = gtk_check_button_new_with_label (_("Buffer size"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[BUFFERSIZE] = check;

	adj = gtk_adjustment_new (128, 128, 4096, 1, 8, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 2, 3, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	gpd->spin[BUFFERSIZE_NUMBER] = spin;

	check = gtk_check_button_new_with_label (_("No sample caching"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[NOSAMPLECACHING] = check;

	check = gtk_check_button_new_with_label (_("No echo"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[NOECHO] = check;
		
	check = gtk_check_button_new_with_label (_("No master volume"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[NOMASTERVOLUME] = check;

	gtk_signal_connect (GTK_OBJECT (gpd->check[NOSOUND]), "toggled",
			    GTK_SIGNAL_FUNC (global_prefs_nosound_cb),
			    (gpointer) gpd);
	  
	gtk_signal_connect (GTK_OBJECT (gpd->check[SOUNDSKIP]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) (gpd->spin[SOUNDSKIP_NUMBER]));
	  
	gtk_signal_connect (GTK_OBJECT (gpd->check[SOUNDQUALITY]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) (gpd->spin[SOUNDQUALITY_NUMBER]));
	
	gtk_signal_connect (GTK_OBJECT (gpd->check[BUFFERSIZE]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) (gpd->spin[BUFFERSIZE_NUMBER]));
	
	return table;
}

GtkWidget*
global_prefs_speed_ops (GlobalPrefsData *gpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *spin;
	GtkObject *adj;

	F_NAME();
	
	table = gtk_table_new (3, 2, TRUE);

	check = gtk_check_button_new_with_label (_("Frame skip"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[FRAMESKIP] = check;

	adj = gtk_adjustment_new (1, 1, G_MAXINT, 1, 10, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	gpd->spin[FRAMESKIP_NUMBER] = spin;


	check = gtk_check_button_new_with_label (_("Frame time"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[FRAMETIME] = check;

	adj = gtk_adjustment_new (1, 1, 100, 0.1, 1, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 1);
	gtk_table_attach (GTK_TABLE (table), spin, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	gpd->spin[FRAMETIME_NUMBER] = spin;

	check = gtk_check_button_new_with_label (_("Cycles"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[CYCLES] = check;

	adj = gtk_adjustment_new (100, 0, 200, 1, 10, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	gpd->spin[CYCLES_NUMBER] = spin;

	gtk_signal_connect (GTK_OBJECT (gpd->check[FRAMESKIP]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) gpd->spin[FRAMESKIP_NUMBER]);
	  
	gtk_signal_connect (GTK_OBJECT (gpd->check[FRAMETIME]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) gpd->spin[FRAMETIME_NUMBER]);
	
	gtk_signal_connect (GTK_OBJECT (gpd->check[CYCLES]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) gpd->spin[CYCLES_NUMBER]);
	
	return table;
}

GtkWidget*
global_prefs_joystick_ops (GlobalPrefsData *gpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *spin;
	GtkWidget *button;	
	GtkObject *adj;
	gchar     *label;
	gint       i, j;
	const gchar *button_names[] = {"A", "B", "X", "Y", "TL", "TR", 
		"Start", "Select"};

	F_NAME();
	
	table = gtk_table_new (5, 11, FALSE);

	for (j = 0; j < NUM_OF_BUTTONS; j++) {
		button = gtk_button_new_with_label (_(button_names[j]));
		gtk_table_attach (GTK_TABLE (table), button, j+2, j+3, 0, 1,
				  GTK_FILL, GTK_FILL, 5, 2);
		gtk_widget_show (button);
	}

	check = gtk_check_button_new_with_label (_("No joystick"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[NOJOY] = check;

	check = gtk_check_button_new_with_label (_("Swap joysticks"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	gpd->check[SWAP] = check;

	label = g_strdup (_("Joymap x"));
	
	for (i = 0; i < NUM_OF_JOYS; i++) {
		label[strlen(_("Joymap x"))-1] = (gchar)('0'+i+1);
		check = gtk_check_button_new_with_label (label);
		gtk_table_attach (GTK_TABLE (table), check,
				  1, 2, i+1, i+2,
				  GTK_FILL, GTK_FILL,
				  5, 2);
		gtk_widget_show (check);
		gpd->check[JOYMAP0+i] = check;
		for (j = 0; j < NUM_OF_BUTTONS; j++) {
			adj = gtk_adjustment_new
				(gpd->data->prefs->buttons[i][j],
				 0, 9, 1, 1, 0);
			spin = gtk_spin_button_new
				(GTK_ADJUSTMENT (adj), 0.5, 0);
			gtk_table_attach
				(GTK_TABLE (table), spin, j+2, j+3, i+1, i+2,
				 GTK_FILL, GTK_FILL, 5, 2);
			gtk_widget_set_usize (spin, 35, 23);
			gtk_widget_show (spin);
			gpd->buttons[i][j] = spin;
		}
	}

	gtk_signal_connect (GTK_OBJECT (gpd->check[NOJOY]), "toggled",
			    GTK_SIGNAL_FUNC (global_prefs_nojoy_cb),
			    (gpointer) gpd);
	
	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			gtk_signal_connect (GTK_OBJECT (gpd->check[JOYMAP0+i]),
					    "toggled", GTK_SIGNAL_FUNC
					    (toggle_to_active_cb),
					    (gpointer) gpd->buttons[i][j]);

	g_free (label);
	
	return table;
}

void
global_prefs_nosound_cb (GtkWidget *widget, GlobalPrefsData *gpd)
{
	gint i;	

	F_NAME();
	
	if (GTK_TOGGLE_BUTTON (widget)->active) {
		gtk_widget_set_sensitive (gpd->spin[SOUNDSKIP_NUMBER],
					  GTK_TOGGLE_BUTTON
					  (gpd->check[SOUNDSKIP])->active);
		gtk_widget_set_sensitive (gpd->mono, FALSE);

		for (i = STEREO; i <= NOMASTERVOLUME; i++)
			gtk_widget_set_sensitive (gpd->check[i], FALSE);
		
		gtk_widget_set_sensitive (gpd->check[SOUNDSKIP], TRUE);
		
		gtk_widget_set_sensitive (gpd->spin[SOUNDQUALITY_NUMBER],
					  FALSE);
		gtk_widget_set_sensitive (gpd->spin[BUFFERSIZE_NUMBER],
					  FALSE);
	} else {
		gtk_widget_set_sensitive
			(gpd->spin[SOUNDSKIP_NUMBER], FALSE);
		gtk_widget_set_sensitive (gpd->mono, TRUE);

		for (i = STEREO; i <= NOMASTERVOLUME; i++)
			gtk_widget_set_sensitive (gpd->check[i], TRUE);

		gtk_widget_set_sensitive (gpd->check[SOUNDSKIP], FALSE);
		
		gtk_widget_set_sensitive (gpd->spin[SOUNDQUALITY_NUMBER],
					  GTK_TOGGLE_BUTTON
					  (gpd->check[SOUNDQUALITY])->active);
		gtk_widget_set_sensitive (gpd->spin[BUFFERSIZE_NUMBER],
					  GTK_TOGGLE_BUTTON
					  (gpd->check[BUFFERSIZE])->active);
	}	
}

void
global_prefs_nojoy_cb (GtkWidget *widget, GlobalPrefsData *gpd)
{
	gint i, j;	
	
	F_NAME();
	
	if (GTK_TOGGLE_BUTTON (widget)->active) {
		gtk_widget_set_sensitive (gpd->check[SWAP], FALSE);
		for (i = 0; i < NUM_OF_JOYS; i++) {
			gtk_widget_set_sensitive (gpd->check[JOYMAP0+i],
						  FALSE);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive (gpd->buttons[i][j],
							  FALSE);
		}
	} else {
		gtk_widget_set_sensitive (gpd->check[SWAP], TRUE);
		for (i = 0; i < NUM_OF_JOYS; i++) {
			gtk_widget_set_sensitive (gpd->check[JOYMAP0+i], TRUE);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive (
					gpd->buttons[i][j],
					GTK_TOGGLE_BUTTON
					(gpd->check[JOYMAP0+i])->
					active);
		}
	}	
}

void
global_prefs_dialog_changed_cb (GtkWidget *widget, GlobalPrefsData *gpd)
{	
	F_NAME();
	
	gnome_dialog_set_sensitive (GNOME_DIALOG (gpd->global_prefs_dialog),
				    0, TRUE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (gpd->global_prefs_dialog),
				    1, TRUE);

	gpd->nodata = FALSE;
		
	gpd->changed = TRUE;
}

void
global_prefs_connect_changes (GlobalPrefsData *gpd)
{
	gint i, j;

	F_NAME();

	for (i = 0; i < GLOB_STR_PREFS; i++)
		gtk_signal_connect (GTK_OBJECT (gpd->entry[i]), "changed",
				    GTK_SIGNAL_FUNC
				    (global_prefs_dialog_changed_cb),
				    (gpointer) gpd);

	for (i = 0; i < MIX_BOOL_PREFS+GLOB_BOOL_PREFS; i++)
		gtk_signal_connect (GTK_OBJECT (gpd->check[i]), "toggled",
				    GTK_SIGNAL_FUNC
				    (global_prefs_dialog_changed_cb),
				    (gpointer) gpd);

	for (i = 0; i < MIX_NUM_PREFS+GLOB_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER)
			gtk_signal_connect
			(GTK_OBJECT (gtk_option_menu_get_menu 
				     (GTK_OPTION_MENU (gpd->spin[FILTER_NUMBER]))),
			 "selection-done",
		 	GTK_SIGNAL_FUNC (global_prefs_dialog_changed_cb),
			(gpointer) gpd);
		else
		gtk_signal_connect
			(GTK_OBJECT (gpd->spin[i]), "changed",
			 GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
			 (gpointer) gpd);
		}

	

	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			gtk_signal_connect (GTK_OBJECT (gpd->buttons[i][j]),
					    "changed",
					    GTK_SIGNAL_FUNC
					    (global_prefs_dialog_changed_cb),
					    (gpointer) gpd);
}

void
global_prefs_set_tooltips (GlobalPrefsData *gpd)
{
	gint         i;
	GtkTooltips *tool;

	F_NAME();
	
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[WITH_ESD-
					      ROM_BOOL_PREFS],
			      _("Enable Esound support."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[NETPLAY-ROM_BOOL_PREFS],
			      _("Use network play. You must provide the "
				"server name and optionally the server "
				"port."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[TRANSPARENCY],
			      _("Enable transparency effects, also enables "
				"16-bit screen mode selection."
				"Transparency effects are automatically "
				"enabled if the depth of your X  "
				"server is 15 or greater."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[SIXTEEN],
			      _("Enable 16-bit internal screen rendering, "
				"allows palette changes but no "
				"transparency effects."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[HIRES],
			      _("Enable support for SNES hi-res. and "
				"interlace modes. USE ONLY IF GAME "
				"REQUIRES IT (FEW DO) BECAUSE IT REALLY "
				"SLOWS DOWN THE EMULATOR."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[NOMODESWITCH],
			      _("The Linux X Windows System port can change "
				"the screen resolution when "
				"switching to full-screen mode so the SNES "
				"display fills the whole screen. "
				"Specify this option to stop it if causes "
				"you problems."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[FULLSCREEN],
			      _("Start the emulator in full screen mode, "
				"rather than a window on the X11 "
				"port if supported by your X server, a "
				"suitable screen mode is defined in "
				"your XF86Config file and the snes9x binary "
				"has root permissions (see above)."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[SCALE],
			      _("Stretch the SNES display to fit the whole "
				"of the computer display. "
				"Linux X Window System full-screen mode or "
				"SVGA port only. Use only if you "
				"have a very fast machine."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[DISPLAYFRAMERATE],
			      _("Display a frame rate counter superimposed "
				"over the bottom, left-hand corner "
				"of the SNES display. The value before the "
				"slash (/) indicates how many "
				"frames per second are being rendered "
				"compared to a real SNES, the value of "
				"which is displayed after the slash."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[NOSOUND],
			      _("Disable sound CPU emulation and sound "
				"output, useful for the few ROMs "
				"where sound emulation causes them to lock "
				"up due to timing errors."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[SOUNDSKIP],
			      _("Sound skip. ONLY USED IF SOUND IS DISABLED."),
			      NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[STEREO],
			      _("Enable stereo sound output."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->mono,
			      _("Enable mono sound output. Faster, but "
				"doesn\'t sound as nice."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[SOUNDQUALITY],
			      _("Sound playback rate/quality: "
				"0 - disable sound, 1 - 8192, 2 - 11025, "
				"3 - 16500, 4 - 22050 (default), "
				"5 - 29300, 6 - 36600, 7 - 44000."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[ENVX],
			      _("Enable volume envelope height reading by "
				"the sound CPU. Can cure sound "
				"repeat problems with some games (e.g. "
				"Mortal Kombat series), while causing "
				"others to lock if enabled (eg. Bomberman "
				"series). Use with the \'Sound sync\' "
				"option to fix the problems this option can "
				"otherwise create."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[SOUNDSYNC],
			      _("Turns on sound-sync code where sound data "
				"is generated in sync with the "
				"emulation of the SNES sound CPU "
				"instructions. Slighty slower emulation "
				"speed but helps reduce note and sound "
				"effects timing problems and one or two  "
				"games actually require this mode, notably "
				"Williams Arcade Classics. "
				"Also turns on interpolated sound and "
				"envelope height reading. "
				"Pressing the \']\' key during gameplay "
				"toggles sound-sync only on and off."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[ALTSAMPLEDECODE],
			      _("Switch to using Anti Resonance's sample "
				"decoding method. Some games, notably "
				"Squaresoft RPG games, make use of a SNES "
				"hardare sample decoding bug/feature to "
				"generate wind and swish-type sound effects; "
				"selecting this option enables code that "
				"tries to simulate the feature. The "
				"disadvantage is that it can decode some "
				"\'normal\' samples incorrectly."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[INTERPOLATEDSOUND],
			      _("Turns on interpolated sound. Any samples "
				"played at a lower frequency than "
				"the current playback rate (\'Sound quality\' "
				"option) have additional sample points "
				"computed and played where previously the "
				"same sample data was played "
				"repeatedly. Lower frequency sounds have "
				"improved bass and sound much less "
				"harsh - I suspect the real SNES does this. "
				"Pressing \'[\' during gameplay toggles the "
				"feature on and off."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[BUFFERSIZE],
			      _("Sound playback buffer size in bytes "
				"128-4096."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[NOSAMPLECACHING],
			      _("Disable decompressed sound sample caching. "
				"Decompressing samples takes time, "
				"slowing down the emulator. Normally the "
				"decompressed samples are saved "
				"just in case they need to be played again, "
				"but the way samples are stored "
				"and played on the SNES, it can result in a "
				"click sound or distortion when "
				"caching samples with loops in them."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[NOECHO],
			      _("Turn off sound echo and FIR filter effects. "
				"Processing these effects can "
				"really slow down a non-MMX Pentium machine "
				"due to the number of calculations "
				"required to implement these features."),
			      NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[NOMASTERVOLUME],
			      _("Disable emulation of the sound DSP master "
				"volume control. Some ROMs set "
				"the volume level very low requiring you to "
				"turn up the volume level of "
				"your speakers introducing more background "
				"noise. Use this option to "
				"always have the master volume set on full "
				"and to by-pass a bug which "
				"prevents the music and sound effects being "
				"heard on Turrican."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[FRAMESKIP],
			      _("Set this value to deliberately fix the "
				"frame skip rate and disable auto-speed "
				"regulation. Use a larger value faster "
				"emulation but more jerky movement and a "
				"smaller value for smooth but slower screen "
				"updates. Use \'+\' and \'-\' keys to modify "
				"the value during a game. Ideal for some "
				"Super FX games that confuse the auto-adjust "
				"code or for games that deliberately flash "
				"the screen every alternate frame."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[FRAMETIME],
			      _("(default: 16.6ms NTSC games and 20ms for "
				"PAL games) If auto-adjust frame skip option "
				"is in effect, then the emulator will try "
				"to maintain a constant game and music speed "
				"locked to this value by skipping "
				"the rendering of some frames or waiting "
				"until the required time is reached. "
				"Increase the value to slow down games, "
				"decrease it to speed up games. "
				"During a game the value can be adjusted "
				"in millisecond steps by pressing "
				"Shift \'-\' or Shift \'+\'."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[CYCLES],
			      _("Percentage of CPU cycles to execute per "
				"scan line, decrease value to "
				"increase emulation frame rate. Most "
				"ROMs work with a value of 85 or above."),
			      NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[SWAP],
			      _("Swap emulated joy-pad 1 and 2 around, "
				"pressing \'6\' during a game does the "
				"same thing."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, gpd->check[NOJOY],
			      _("Turn off joystick, SideWinder and GrIP "
				"detection (joystick polling on the "
				"PC slows the emulator down)."), NULL);
	
	for (i = 0; i < NUM_OF_JOYS; i++) {
		tool = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tool,
				      gpd->check[JOYMAP0+i],
				      _("Specify the SNES joypad buttons to "
					"Linux joystick driver buttons "
					"mapping for each of the four "
					"supported joypads. Specify the Linux "
					"joystick button number for the "
					"corresponding SNES button in the "
					"following order: A, B, X, Y, TL, TR, "
					"Start and Select The default map is: "
					"1 0 4 3 6 7 8 9 which is suitable "
					"for Sidewinder gamepads."), NULL);
	}
}

void
global_prefs_dialog_set_status (GlobalPrefsData *gpd)
{
	gint i, j;
	gchar *path;
	gint numberops[] = {
		SOUNDSKIP,
		SOUNDQUALITY,
		BUFFERSIZE,
		FRAMESKIP,
		FRAMETIME,
		CYCLES
	};


	F_NAME();

	gtk_entry_set_text (GTK_ENTRY
			    (gpd->entry[SNES9X_COMMAND-ROM_STR_PREFS]),
			    gpd->data->prefs->string_prefs
			    [SNES9X_COMMAND-ROM_STR_PREFS]);
	
	path = gpd->data->prefs->string_prefs[ROMS_DIR-ROM_STR_PREFS];
	if (!path)
		path = get_default_roms_dir();
	if (path){			
		gtk_entry_set_text (GTK_ENTRY
			(gpd->entry[ROMS_DIR-ROM_STR_PREFS]), path);
	}

	if (gpd->data->prefs->string_prefs[SERVER-ROM_STR_PREFS])
		
		gtk_entry_set_text (GTK_ENTRY
				    (gpd->entry[SERVER-ROM_STR_PREFS]),
				    gpd->data->prefs->string_prefs
				    [SERVER-ROM_STR_PREFS]);

	if (gpd->data->prefs->string_prefs[NETPORT-ROM_STR_PREFS])
		gtk_entry_set_text (GTK_ENTRY
				    (gpd->entry[NETPORT-ROM_STR_PREFS]),
				    gpd->data->prefs->string_prefs
				    [NETPORT-ROM_STR_PREFS]);
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
				      (gpd->check[WITH_ESD-ROM_BOOL_PREFS]),
				      gpd->data->prefs->bool_prefs
				      [WITH_ESD-ROM_BOOL_PREFS]);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
				      (gpd->check[NETPLAY-ROM_BOOL_PREFS]),
				      gpd->data->prefs->bool_prefs
				      [NETPLAY-ROM_BOOL_PREFS]);
	
	for (i = 0; i < MIX_BOOL_PREFS+GLOB_BOOL_PREFS; i++)
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (gpd->check[i]),
			 gpd->data->prefs->bool_prefs[i]);

	
	for (i = 0; i < MIX_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER) {
			 gtk_option_menu_set_history (
					 GTK_OPTION_MENU (gpd->spin[i]),
					 gpd->data->prefs->number_prefs[i]);
		}
		else {
			if (gpd->data->prefs->bool_prefs[(numberops[i])])
				gtk_spin_button_set_value
					(GTK_SPIN_BUTTON (gpd->spin[i]),
				 	gpd->data->prefs->number_prefs[i]);
			else
				gtk_widget_set_sensitive
					(gpd->spin[i], FALSE);
		}
	}		
	
	if (gpd->data->prefs->bool_prefs[STEREO])
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (gpd->check[STEREO]), TRUE);
	else
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (gpd->mono), TRUE);
	
	if (gpd->data->prefs->bool_prefs[NOSOUND])
		for (i = STEREO; i < NOMASTERVOLUME; i++)
			gtk_widget_set_sensitive (gpd->check[i], FALSE);
	else {
		gtk_widget_set_sensitive
			(gpd->check[SOUNDSKIP], FALSE);
		gtk_widget_set_sensitive
			(gpd->spin[SOUNDSKIP_NUMBER], FALSE);
	}

	if (gpd->data->prefs->bool_prefs[NOJOY]) {
		gtk_widget_set_sensitive (gpd->check[SWAP], FALSE);
		for (i = 0; i < NUM_OF_JOYS; i++) {
			gtk_widget_set_sensitive (gpd->check[JOYMAP0+i],
						  FALSE);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive (gpd->buttons[i][j],
							  FALSE);
		}
	}

	for (i = 0; i < NUM_OF_JOYS; i++)
		if (!gpd->data->prefs->bool_prefs[JOYMAP0+i])
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive (gpd->buttons[i][j],
							  FALSE);
#ifdef NO_ESD_SUPPORT
	gtk_widget_set_sensitive (gpd->check[WITH_ESD-ROM_BOOL_PREFS], FALSE);
#endif
}

void
global_prefs_dialog_get_status (GlobalPrefsData *gpd)
{
	gchar *text;
	gint   i, j, number;

	F_NAME();
	
	text = gtk_entry_get_text
		(GTK_ENTRY (gpd->entry[SNES9X_COMMAND-ROM_STR_PREFS]));
	g_free (gpd->data->prefs->string_prefs[SNES9X_COMMAND-ROM_STR_PREFS]);
	gpd->data->prefs->string_prefs[SNES9X_COMMAND-ROM_STR_PREFS] =
		g_strdup (text);

	text = gtk_entry_get_text
		(GTK_ENTRY (gpd->entry[ROMS_DIR-ROM_STR_PREFS]));
	g_free (gpd->data->prefs->string_prefs[ROMS_DIR-ROM_STR_PREFS]);
	gpd->data->prefs->string_prefs[ROMS_DIR-ROM_STR_PREFS] =
		g_strdup (text);
	
	for (i = 0; i < MIX_BOOL_PREFS+GLOB_BOOL_PREFS; i++)
		gpd->data->prefs->bool_prefs[i] =
			GTK_TOGGLE_BUTTON (gpd->check[i])->active;

	text = gtk_entry_get_text
		(GTK_ENTRY (gpd->entry[SERVER-ROM_STR_PREFS]));
	
	if (strlen (text)) {
		if (gpd->data->prefs->string_prefs[SERVER-ROM_STR_PREFS])
			g_free (gpd->data->prefs->string_prefs
				[SERVER-ROM_STR_PREFS]);
		gpd->data->prefs->string_prefs[SERVER-ROM_STR_PREFS] =
			g_strdup (text);
	}
	
	text = gtk_entry_get_text
		(GTK_ENTRY (gpd->entry[NETPORT-ROM_STR_PREFS]));
	
	if (strlen (text)) {
		if (gpd->data->prefs->string_prefs[NETPORT-ROM_STR_PREFS])
			g_free (gpd->data->prefs->string_prefs
				[NETPORT-ROM_STR_PREFS]);
		gpd->data->prefs->string_prefs[NETPORT-ROM_STR_PREFS] =
			g_strdup (text);
	}

	for (i = 0; i < MIX_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER) 
			gpd->data->prefs->number_prefs[i] =
			       	g_list_index (GTK_MENU_SHELL (gtk_option_menu_get_menu 
						 (GTK_OPTION_MENU 
						  (gpd->spin[i])))->children, 
						GTK_MENU_ITEM (gtk_menu_get_active 
							(GTK_MENU
							 (gtk_option_menu_get_menu
							  (GTK_OPTION_MENU 
							   (gpd->spin[i]))))));
		else 
			gpd->data->prefs->number_prefs[i] =
				gtk_spin_button_get_value_as_float
				(GTK_SPIN_BUTTON (gpd->spin[i]));
	}

	

	for (i = 0; i < NUM_OF_JOYS; i++) {
		if (gpd->data->prefs->bool_prefs[JOYMAP0+i])
			for (j = 0; j < NUM_OF_BUTTONS; j++) {
				number = gtk_spin_button_get_value_as_int
					(GTK_SPIN_BUTTON (gpd->buttons[i][j]));
				gpd->data->prefs->buttons[i][j] =
					(gshort) number;
			}
	}
}

void
global_prefs_apply_cb (GtkWidget *widget, GlobalPrefsData *gpd)
{
	F_NAME();
	
	if (!gpd->changed)
		return;

if (!strlen (gtk_entry_get_text
		     (GTK_ENTRY (gpd->entry[ROMS_DIR-ROM_STR_PREFS]))))
	gtk_entry_set_text (GTK_ENTRY (gpd->entry[ROMS_DIR-ROM_STR_PREFS]),
		       	get_default_roms_dir());
		
	if (!strlen (gtk_entry_get_text
		     (GTK_ENTRY (gpd->entry[SNES9X_COMMAND-ROM_STR_PREFS]))) ||
	    (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (gpd->check[NETPLAY-ROM_BOOL_PREFS])) &&
	     (!strlen (gtk_entry_get_text
		       (GTK_ENTRY (gpd->entry[SERVER-ROM_STR_PREFS])))))) {
		gpd->nodata = TRUE;
		no_global_data (gpd);
		return;
	}

	global_prefs_dialog_get_status (gpd);
	
	save_global_prefs (gpd->data->prefs);

	gpd->changed = FALSE;

	gnome_dialog_set_sensitive (GNOME_DIALOG (gpd->global_prefs_dialog),
				    0, FALSE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (gpd->global_prefs_dialog),
				    1, FALSE);
}

void
global_prefs_close_cb (GtkWidget *widget, GlobalPrefsData *gpd)
{
	GtkWidget *warning;

	F_NAME();
	
	if (gpd->changed && !gpd->nodata) {
		warning = gnome_message_box_new
			(_("You changed the global preferences.\n"
			 "Are you sure?"),
			 GNOME_MESSAGE_BOX_WARNING,
			 GNOME_STOCK_BUTTON_YES,
			 GNOME_STOCK_BUTTON_NO,
			 NULL);

			gtk_window_set_modal (GTK_WINDOW (warning), TRUE);
			gtk_window_set_transient_for
				(GTK_WINDOW (warning),
				 GTK_WINDOW (gpd->global_prefs_dialog));
		
		gnome_dialog_button_connect (GNOME_DIALOG (warning),
					     0, GTK_SIGNAL_FUNC
					     (global_prefs_destroy_cb),
					     (gpointer) gpd);
		gnome_dialog_button_connect (GNOME_DIALOG (warning),
					     1, GTK_SIGNAL_FUNC
					     (destroy_data_cb),
					     (gpointer) warning);

		gtk_widget_show (warning);
	}
	
	if (!gpd->changed && !gpd->nodata)
		global_prefs_destroy_cb (NULL, gpd);
}

void
global_prefs_help_cb (GtkWidget *widget, GtkWidget *notebook)
{
	gint part;
	gint page[] = { 8, 1, 2, 4, 6 };

	F_NAME();
	
	part = gtk_notebook_get_current_page(GTK_NOTEBOOK (notebook));

	part = page[part];
	
	readme_cb (NULL, part);
}

void
global_prefs_destroy_cb (GtkWidget *widget, GlobalPrefsData *gpd)
{
	F_NAME();

	if (gpd->changed)
		save_global_prefs (gpd->data->prefs);

	gpd->data->dialog_open = FALSE;
	
	gtk_widget_destroy (gpd->global_prefs_dialog);
}

gboolean
global_prefs_delete_cb (GtkWidget *widget)
{
	GlobalPrefsData *gpd =
		(GlobalPrefsData*) gtk_object_get_data (GTK_OBJECT (widget),
							"global-prefs");

	F_NAME();
	
	global_prefs_close_cb (NULL, gpd);

	return TRUE;
}

void
global_prefs_free_cb (GtkWidget *widget, GlobalPrefsData *gpd)
{
	F_NAME();
	
	free (gpd);
}

void
no_global_data (GlobalPrefsData *gpd)
{
	GtkWidget *error;
	gchar     *mesg;

	F_NAME();
	
	if (gtk_toggle_button_get_active
	    (GTK_TOGGLE_BUTTON (gpd->check[NETPLAY-ROM_BOOL_PREFS])))
		mesg = g_strdup (_("You must fill the \"Snes9X command\"\n"
				   "and the \"Snes9X server\" entries."));
	else
		mesg = g_strdup (_("You must fill the \"Snes9X command\" "
				   "entry."));
	
	error = gnome_message_box_new(mesg,
				      GNOME_MESSAGE_BOX_ERROR, 
				      GNOME_STOCK_BUTTON_OK,
				      NULL);

	gtk_window_set_modal (GTK_WINDOW (error), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (error),
				      GTK_WINDOW (gpd->global_prefs_dialog));
	
	gtk_widget_show (error);
}

/* globalprefs.c ends here. */
