/* NVClock 0.8 - Linux overclocker for NVIDIA cards
 * 
 * Copyright(C) 2001-2005 Roderick Colenbrander
 *
 * site: http://nvclock.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "nvclock.h"
#include <stdint.h>

/* Thanks to Alexey Nicolaychuk (Unwinder), the author of Rivatuner, for providing
   these macros to get nv30 clock detection working.
*/
#define NV30_PLL_M1(PLL) ( ( PLL ) & 0x0f )
#define NV30_PLL_M2(PLL) ( ( ( PLL )  >>  4 ) & 0x07 )
#define NV30_PLL_N1(PLL) ( ( ( PLL ) >> 8 ) & 0xff )
#define NV30_PLL_N2(PLL) ( ( ( ( PLL ) >> 19 ) & 0x07 ) | ( ( ( PLL ) >> 21 ) & 0x18 ) )
#define NV30_PLL_P(PLL) ( ( ( PLL ) >> 16 ) & 0x07 )

#define PCI_CAPABILITY_LIST 0x34
#define PCI_CAP_LIST_ID 0x0
#define PCI_CAP_LIST_NEXT 0x1
#define PCI_CAP_ID_AGP 0x2 /* AGP */
#define PCI_CAP_ID_EXP 0x10 /* PCI-Express */

#define PCI_GET_BUS(devbusfn) ((devbusfn >> 8) & 0xff)
#define PCI_GET_DEVICE(devbusfn) ((devbusfn & 0xff) >> 3)
#define PCI_GET_FUNCTION(devbusfn) (devbusfn & 0x7)
#define PCI_GET_DEVBUSFN(dev, bus, fn) ((bus << 8) | (dev << 3) | (fn & 0x7))

/* Set the card object to the requested card */
int set_card(int number);
/* Basic version of set_card */
int set_card_info(int number);

/* AGP related functions */
short get_agp_rate();
char* get_agp_status();
char* get_bus_type();
char* get_sba_status();
char* get_sba_support();
char* get_fw_status();
char* get_fw_support();
char* get_supported_agp_rates();

/* Memory related functions */
short get_memory_size();
char* get_memory_type();
short get_memory_width();

/* Some internally needed functions */
int probe_devices(); /* detect Nvidia cards */
const char* get_card_name(int device_id, gpu_type *gpu);
short get_gpu_arch(int device_id);
void set_speed_range();
int map_mem(const char* dev_name);
void unmap_mem();
int32_t pciReadLong(unsigned short devbusfn, long offset);
char* dummy_str();
void dummy();

/* Pipeline modding */
int nv40_get_pixel_pipelines();
void nv40_set_pixel_pipelines(unsigned char number);
int nv40_get_vertex_pipelines();
void nv40_set_vertex_pipelines(unsigned char number);

/* Fanspeed adjustment stuff using card registers; Note that this stuff is useally only available on reference cards */
float nv30_get_fanspeed();
float nv40_get_fanspeed();
float nv43_get_fanspeed();
void nv30_set_fanspeed(float speed);
void nv40_set_fanspeed(float speed);
void nv43_set_fanspeed(float speed);


/* Overclocking related functions */
/* Note that they are splitted out into different architectures because
 * new chips work differently than the old ones. The cleanest way is to
 * seperate it all out
 */

float get_gpu_speed();
float nv30_get_gpu_speed();
float nv31_get_gpu_speed();
float nv40_get_gpu_speed();

float get_memory_speed(); /* tnt up to the geforcefx5200 + geforcefx5500*/
float nforce_get_memory_speed();
float nv30_get_memory_speed(); /* fx5800 + fx5900 */
float nv31_get_memory_speed(); /* fx5600 + fx5700 */
float nv40_get_memory_speed(); /* 6800 series */
float dummy_get_memory_speed();

void set_gpu_speed(unsigned int nvclk);
void nv30_set_gpu_speed(unsigned int nvclk);
void nv31_set_gpu_speed(unsigned int nvclk);
void nv40_set_gpu_speed(unsigned int nvclk);

void set_memory_speed(unsigned int memclk);
void nv30_set_memory_speed(unsigned int memclk);
void nv31_set_memory_speed(unsigned int memclk);
void nv40_set_memory_speed(unsigned int memclk);

void reset_gpu_speed();
void nv30_reset_gpu_speed();
void nv31_reset_gpu_speed();
void nv40_reset_gpu_speed();

void reset_memory_speed();
void nv30_reset_memory_speed();
void nv31_reset_memory_speed();
void nv40_reset_memory_speed();

void dump_bios(char *filename);
struct nvbios* read_bios(char *filename);
int CalcSpeed(int base_freq, int m, int n, int p);
int CalcSpeed_nv30(int base_freq, int m1, int m2, int n1, int n2, int p);

/* Basic clock/pll validation routines; needed for the config file and the nvclock wrapper */
int validate_clock(int arch, int freq);
int validate_pll(int arch, int base_freq, unsigned int pll, unsigned int pll2);

float GetClock(int base_freq, unsigned int pll);
float GetClock_nv30(int base_freq, unsigned int pll);
float GetClock_nv31(int base_freq, unsigned int pll, unsigned int pll2);
float GetClock_nv40(int base_freq, unsigned int pll, unsigned int pll2);
