/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * The plugin uses the ARB_fragment_program extension to access the
 * per-fragment programmability of the OpenGL pipeline (pixel-shading). We
 * could have used the "OpenGL Shading Language" but at the time this plugin
 * is written, the needed extensions have just been integrated in Mesa, and
 * thus not widely available on systems using DRI drivers.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>     /* strlen */
#include "pgmtexture.h"

GST_DEBUG_CATEGORY_EXTERN (pgm_gl_debug);
#define GST_CAT_DEFAULT pgm_gl_debug

/* fragment program header and footer strings */
#define PROGRAM_HEADER "!!ARBfp1.0"
#define PROGRAM_FOOTER "END"

/* Program limits checking */
typedef struct {
  PgmGlEnum    query;
  PgmGlEnum    maximum;
  const gchar *description;
} ProgramQuery;

/* Limits table */
static ProgramQuery limits[] = {
  { PGM_GL_PROGRAM_INSTRUCTIONS,
    PGM_GL_MAX_PROGRAM_INSTRUCTIONS,
    "total instructions" },
  { PGM_GL_PROGRAM_NATIVE_INSTRUCTIONS,
    PGM_GL_MAX_PROGRAM_NATIVE_INSTRUCTIONS,
    "native total instructions" },
  { PGM_GL_PROGRAM_PARAMETERS,
    PGM_GL_MAX_PROGRAM_PARAMETERS,
    "parameter bindings" },
  { PGM_GL_PROGRAM_NATIVE_PARAMETERS,
    PGM_GL_MAX_PROGRAM_NATIVE_PARAMETERS,
    "native parameter bindings" }

#if 0
  /* Sadly that needs to be disabled due to a bug in Mesa which returns
   * uninitialized values in glGetProgramiv for the following limits. */
  { PGM_GL_PROGRAM_ALU_INSTRUCTIONS,
    PGM_GL_MAX_PROGRAM_ALU_INSTRUCTIONS,
    "ALU instructions" },
  { PGM_GL_PROGRAM_TEX_INSTRUCTIONS,
    PGM_GL_MAX_PROGRAM_TEX_INSTRUCTIONS,
    "texture instructions" },
  { PGM_GL_PROGRAM_TEX_INDIRECTIONS,
    PGM_GL_MAX_PROGRAM_TEX_INDIRECTIONS,
    "texture indirections" },
  { PGM_GL_PROGRAM_NATIVE_ALU_INSTRUCTIONS,
    PGM_GL_MAX_PROGRAM_NATIVE_ALU_INSTRUCTIONS,
    "native ALU instructions" },
  { PGM_GL_PROGRAM_NATIVE_TEX_INSTRUCTIONS,
    PGM_GL_MAX_PROGRAM_NATIVE_TEX_INSTRUCTIONS,
    "native texture instructions" },
  { PGM_GL_PROGRAM_NATIVE_TEX_INDIRECTIONS,
    PGM_GL_MAX_PROGRAM_NATIVE_TEX_INDIRECTIONS,
    "native texture indirections" }
#endif
};

/* Convert from YV12/I420 planar YCbCr to RGB color space using coefficients
 * from the ITU-R BT.601 standard for SDTV and the equations detailed in the
 * book "Video Demystified, 4th Edition". */
static const gchar *per_plane_i420_yv12_rgb_header_string =
  "OPTION ARB_precision_hint_fastest;"
  "ATTRIB position = fragment.texcoord[0];";

/* Program meant to run correctly with all OpenGL implementations */
static const gchar *per_plane_i420_yv12_rgb_string =
  "TEMP color, tmp;"
  "TEX color, position, texture[0], 2D;"
  "MAD color, color, 1.164, -0.073;"
  "TEX tmp.r, position, texture[%s], 2D;"
  "TEX tmp.g, position, texture[%s], 2D;"
  "SUB tmp.rg, tmp, { 0.5, 0.5 };"
  "MAD color.rgb, { 0, -0.391, 2.018 }, tmp.rrra, color;"
  "MAD color.rgb, { 1.596, -0.813, 0 }, tmp.ggga, color;";

/* Program dedicated to the DRI R300 driver which doesn't take in account the
 * writemask during a texture lookup. A patch has been pushed the 7 Oct 2007
 * in the Mesa git repository:
 * http://gitweb.freedesktop.org/?p=mesa/mesa.git;a=commit;h=32699696e31234c8d4e4b08f255ba2134ec12db5
 */
static const gchar *per_plane_i420_yv12_rgb_r300_string =
  "TEMP color, cb, cr;"
  "TEX color, position, texture[0], 2D;"
  "MAD color, color, 1.164, -0.073;"
  "TEX cb, position, texture[%s], 2D;"
  "TEX cr, position, texture[%s], 2D;"
  "SUB cb.r, cb, 0.5;"
  "SUB cr.r, cr, 0.5;"
  "MAD color.rgb, { 0, -0.391, 2.018 }, cb.rrra, color;"
  "MAD color.rgb, { 1.596, -0.813, 0 }, cr.rrra, color;";

/* Combine the color with the primary vertex color */
static const gchar *primary_color_combine_string =
  "MUL result.color, fragment.color, color;";

/* Programs and lookup table indexed on the name */
static PgmProgram *per_plane_i420_rgb = NULL;
static PgmProgram *per_plane_yv12_rgb = NULL;
static PgmProgram *program_table[PGM_PROGRAM_NAME_LAST] = { NULL };

/* OpenGL procedure addresses */
static const PgmContextProcAddress *gl = NULL;

/* Private functions */

/* Check if a program exceeds GPU limits */
static gboolean
is_over_limits (PgmProgram *program)
{
  gboolean over = FALSE;
  gint i = 0;
  gint limit_count;
  gint value, max;

  limit_count = sizeof (limits) / sizeof (ProgramQuery);

  while (!over && i < limit_count)
    {
      gl->get_program_iv (PGM_GL_FRAGMENT_PROGRAM, limits[i].query, &value);
      gl->get_program_iv (PGM_GL_FRAGMENT_PROGRAM, limits[i].maximum, &max);

      GST_DEBUG ("program %p '%-27s': %d, limit is %d", program,
                 limits[i].description, value, max);

      if (value >= max)
        {
          GST_WARNING ("program %p: too many %s (%d, limit is %d)", program,
                       limits[i].description, value, max);
          over = TRUE;
        }

      i++;
    }

  return over;
}

/* Define the string of the fragment program */
static void
set_program_string (PgmProgram *program,
                    const gchar *string)
{
  if (program->string)
    g_free (program->string);

  program->string = g_strdup (string);
  program->string_len = strlen (string);
}

/* Compile a fragment program */
static gboolean
compile_program (PgmProgram *program)
{
  gint error = 0;

  if (!program->string)
    {
      GST_WARNING ("no string to compile\n");
      return FALSE;
    }

  /* Clean up errors */
  while (gl->get_error () != PGM_GL_NO_ERROR);

  /* Compile the program */
  gl->bind_program (PGM_GL_FRAGMENT_PROGRAM, program->id);
  gl->program_string (PGM_GL_FRAGMENT_PROGRAM, PGM_GL_PROGRAM_FORMAT_ASCII,
                      program->string_len, program->string);

  /* Check compilation errors */
  gl->get_integer_v (PGM_GL_PROGRAM_ERROR_POSITION, &error);
  if (error != -1)
    {
      GST_WARNING ("error at pos %d beginning with '%.40s'\n",
                   error, program->string + error);
      gl->bind_program (PGM_GL_FRAGMENT_PROGRAM, 0);
      return FALSE;
    }

  /* Validate limits */
  if (is_over_limits (program))
    {
      GST_WARNING ("exceeds native resource limits\n");
      gl->bind_program (PGM_GL_FRAGMENT_PROGRAM, 0);
      return FALSE;
    }

#if 0
  /* Check the number of local parameters
   * FIXME: local parameters are currently not used */
  gint max;
  gl->get_integer_v (PGM_GL_FRAGMENT_PROGRAM,
                     PGM_GL_MAX_PROGRAM_LOCAL_PARAMETERS, &max);
  if (program->local_count > max)
    {
      GST_WARNING ("exceeds local parameters limit (%d, limit is %d)\n",
                   program->local_count, max);
      gl->bind_program (PGM_GL_FRAGMENT_PROGRAM, 0);
      return FALSE;
    }
#endif

  return TRUE;
}

/* Fragment program contructor */
static PgmProgram*
new_program (void)
{
  PgmProgram *program = NULL;

  program = g_slice_new0 (PgmProgram);

  program->string = NULL;
  program->string_len = 0;
  gl->gen_programs (1, &program->id);

  return program;
}

/* Fragment program destructor */
static void
free_program (PgmProgram *program)
{
  if (program->string)
    {
      g_free (program->string);
      program->string = NULL;
      program->string_len = 0;
    }

  if (program->id)
    gl->delete_programs (1, &program->id);

  g_slice_free (PgmProgram, program);
}

/* Public functions */

gboolean
pgm_program_create (PgmContext *context)
{
  gchar buffer[1024]; /* should be enough */
  gchar *p;

  /* The function should be called only once */
  if (G_UNLIKELY (gl))
    return FALSE;

  /* Check validity of the context */
  if (G_UNLIKELY (!context))
    return FALSE;

  gl = context->gl;

  if (context->feature_mask & PGM_GL_FEAT_PER_PLANE_YCBCR_PROGRAM)
    {
      const gchar *_per_plane_i420_yv12_rgb_string = NULL;

      /* Hack to workaround the bug commented above with the DRI R300 driver */
      if (strncmp ("Mesa DRI R300", context->renderer, 13) == 0)
        _per_plane_i420_yv12_rgb_string = per_plane_i420_yv12_rgb_r300_string;
      else
        _per_plane_i420_yv12_rgb_string = per_plane_i420_yv12_rgb_string;

      /* Program converting from per plane I420 to RGB */
      per_plane_i420_rgb = new_program ();
      if (per_plane_i420_rgb)
        {
          p = buffer;
          p += sprintf (p, PROGRAM_HEADER);
          p += sprintf (p, per_plane_i420_yv12_rgb_header_string);
          p += sprintf (p, _per_plane_i420_yv12_rgb_string, "1", "2");
          p += sprintf (p, primary_color_combine_string);
          sprintf (p, PROGRAM_FOOTER);

          set_program_string (per_plane_i420_rgb, buffer);

          if (!compile_program (per_plane_i420_rgb))
            {
              free_program (per_plane_i420_rgb);
              per_plane_i420_rgb = NULL;
            }
        }
      program_table[PGM_PROGRAM_PER_PLANE_I420_RGB] = per_plane_i420_rgb;

      /* Program converting from per plane YV12 to RGB */
      per_plane_yv12_rgb = new_program ();
      if (per_plane_yv12_rgb)
        {
          p = buffer;
          p += sprintf (p, PROGRAM_HEADER);
          p += sprintf (p, per_plane_i420_yv12_rgb_header_string);
          p += sprintf (p, _per_plane_i420_yv12_rgb_string, "2", "1");
          p += sprintf (p, primary_color_combine_string);
          sprintf (p, PROGRAM_FOOTER);

          set_program_string (per_plane_yv12_rgb, buffer);

          if (!compile_program (per_plane_yv12_rgb))
            {
              free_program (per_plane_yv12_rgb);
              per_plane_yv12_rgb = NULL;
            }
        }
      program_table[PGM_PROGRAM_PER_PLANE_YV12_RGB] = per_plane_yv12_rgb;
    }

  return TRUE;
}

void
pgm_program_delete (void)
{
  guint i;

  for (i = 0; i < PGM_PROGRAM_NAME_LAST; i++)
    {
      if (program_table[i])
        {
          free_program (program_table[i]);
          program_table[i] = NULL;
        }
    }

  gl = NULL;
}

PgmProgram *
pgm_program_get (PgmProgramName name)
{
  return program_table[name];
}

void
pgm_program_bind (PgmProgram *program)
{
  g_return_if_fail (program != NULL);

  if (program->id)
    {
      gl->enable (PGM_GL_FRAGMENT_PROGRAM);
      gl->bind_program (PGM_GL_FRAGMENT_PROGRAM, program->id);
    }
}

void
pgm_program_unbind (void)
{
  gl->disable (PGM_GL_FRAGMENT_PROGRAM);
}
